(* signed: Yanlin Duan *)

{ 
	open Parser 
    let lineno = ref 1
    let depth = ref 0
    let filename = ref ""

    let unescape s =
    	Scanf.sscanf ("\"" ^ s ^ "\"") "%S%!" (fun x -> x)
}

let alpha = ['a'-'z' 'A'-'Z']
let escape = '\\' ['\\' ''' '"' 'n' 'r' 't']
let escape_char = ''' (escape) '''
let ascii = ([' '-'!' '#'-'[' ']'-'~'])
let digit = ['0'-'9']
let id = alpha (alpha | digit | '_')*
let string = '"' ( (ascii | escape)* as s) '"'
let char = ''' ( ascii | digit ) '''
let float = (digit+) ['.'] digit+
let int = digit+
let whitespace = [' ' '\t' '\r']
let newline = '\n'

rule token = parse
  whitespace { token lexbuf }
| newline 	 { incr lineno; token lexbuf}
| "/*"     { incr depth; comment lexbuf }
| "//"     { comment2 lexbuf }
| '('      { LPAREN }
| ')'      { RPAREN }
| '{'      { LBRACE }
| '}'      { RBRACE }
| ';'      { SEMI }
| ':'      { COLON }
| ','      { COMMA }
(* Operators *)
| '+'      { PLUS }
| '-'      { MINUS }
| '*'      { TIMES }
| '/'      { DIVIDE }
| '%'      { MODULO }
| '='      { ASSIGN }
| "=="     { EQ }
| "!="     { NEQ }
| '<'      { LT }
| "<="     { LEQ }
| ">"      { GT }
| ">="     { GEQ }
| "and"    { AND }
| "or"     { OR }
| "not"    { NOT }
| '.'      { DOT }
| '['      { LBRACK }
| ']'      { RBRACK }
(* Ownership *)
| "&"      { BORROW }
| "mut"    { MUTABLE }
| "&mut"   { MUTABLEBORROW }
(* Branch Control *)
| "if"     { IF }
| "else"   { ELSE }
| "for"    { FOR }
| "while"  { WHILE }
| "loop"   { LOOP }
| "in"     { IN }
(* Data Types *)
| "int"    { INT }
| "float"  { FLOAT }
| "bool"   { BOOL }
| "char"   { CHAR }
| "true"   { TRUE }
| "false"  { FALSE }
| "string" { STRING }
| "struct" { STRUCT }
| "impl"   { IMPL }
| "let"    { LET }
| "as"     { AS  }
| "void"   { VOID }
(* function *)
| "fn"     { FUNC }
| "->"     { OUTPUT }
| "return" { RETURN }

(* Other *)
| int as lxm   		{ INT_LITERAL(int_of_string lxm) }
| float as lxm 		{ FLOAT_LITERAL(float_of_string lxm) }
| char as lxm  		{ CHAR_LITERAL( String.get lxm 1 ) }
| escape_char as lxm{ CHAR_LITERAL( String.get (unescape lxm) 1) }
| string       		{ STRING_LITERAL(unescape s) }
| id as lxm    		{ ID(lxm) }
| eof          		{ EOF }
| '"' 			      { raise (Failure("Unmatched quotation at " ^ string_of_int !lineno)) }
| _ as illegal    { raise (Failure("IllegalCharacter: " ^ Char.escaped illegal ^ " at " ^ string_of_int !lineno)) }

and comment = parse
    newline { incr lineno; comment lexbuf }
  |	"*/" 	  { decr depth; if !depth > 0 then comment lexbuf else token lexbuf }
  | "/*" 	  { incr depth; comment lexbuf }
  |	_    	  { comment lexbuf }

and comment2 = parse
    newline {token lexbuf}
  | _ {comment lexbuf}