(* signed: Yanlin Duan, Emily Meng, Shiyu Qiu *)

open Ast

type sexpr =
  | SNoexpr
  | SIntLit of int * typ
  | SBoolLit of bool * typ
  | SFloatLit of float * typ
  | SCharLit of char * typ
  | SArrayLit of sexpr list * int * typ
  | SStringLit of string * typ
  | SArrayAccess of sexpr * sexpr * typ
  | SStructCreate of ((string * sexpr) list)
  | SStructAccess of sexpr * string * typ
  | SId of string * typ
  | SBinop of sexpr * op * sexpr * typ
  | SUnop of uop * sexpr * typ
  | SCast of sexpr * typ
  | SCall of string * sexpr list * typ
  | SStructMethodCall of string * string * sexpr list * typ

type sstmt =
    SBlock of sstmt list
  | SDeclaration of borrowType * bind * sexpr
  | SExpr of sexpr * typ
  | SReturn of sexpr * typ
  | SBreak
  | SIf of sexpr * sstmt * sstmt
  | SFor of sexpr * sexpr * sexpr * sstmt
  | SWhile of sexpr * sstmt
  | SLoop of sstmt
  | SStructDef of string * bind list
  | SImplDef of string * sfunc_decl * typ

and sfunc_decl = {
    sfname : string;
    sformals : bind list;
    soutputType : typ;
    sbody : sstmt list;
  }

type program = sfunc_decl list

let get_type_from_sexpr = function
    SNoexpr -> DataT(VoidT)
  | SIntLit(_,t) -> t
  | SBoolLit(_,t) -> t
  | SFloatLit(_,t) -> t
  | SCharLit(_,t) -> t
  | SArrayLit(_,_,t) -> t
  | SStringLit(_,t) -> t
  | SArrayAccess(_,_,t) -> t
  | SStructCreate(_) -> StructT("unknown_struct")
  | SStructAccess(_,_,t) -> t
  | SId(_,t) -> t
  | SBinop(_,_,_,t) -> t
  | SUnop(_,_,t) -> t
  | SCall(_,_,t) -> t
  | SStructMethodCall(_,_,_,t) -> t
  | _ -> DataT(VoidT) 

(* Pretty-printing functions *)

let rec string_of_sexpr = function
    SStringLit(s,_) -> s
  | SNoexpr -> ""
  | SIntLit(i,_) -> string_of_int i
  | SBoolLit(b,_) -> string_of_bool b
  | SFloatLit(f,_) -> string_of_float f
  | SCharLit(c,_) -> Char.escaped c
  | SCall(f, el,_) -> f ^ "(" ^ String.concat ", " (List.map string_of_sexpr el) ^ ")"
  | SId(a,_) -> a 
  | SStructAccess(a,_,_)-> string_of_sexpr a 
  | _ -> "string_of_sexpr not implemented yet"

let string_of_sstmt = function
    SDeclaration(_,(_,_),e) -> "s" ^ string_of_sexpr e
  | _ -> "string_of_sstmt not implemented yet"