/* signed: Yanlin Duan, Emily Meng, Shiyu Qiu */

/* Ocamlyacc parser for rusty */

%{
open Ast
%}

%token SEMI LPAREN RPAREN LBRACE RBRACE LBRACK RBRACK COMMA COLON DOT
%token PLUS MINUS TIMES DIVIDE MODULO ASSIGN NOT
%token EQ NEQ LT LEQ GT GEQ TRUE FALSE AND OR
%token RETURN IF ELSE FOR IN WHILE LOOP AS INT BOOL FLOAT CHAR STRING
%token BORROW MUTABLE MUTABLEBORROW VOID
%token STRUCT IMPL LET
%token FUNC OUTPUT
%token <int> INT_LITERAL
%token <float> FLOAT_LITERAL
%token <char> CHAR_LITERAL
%token <string> STRING_LITERAL
%token <string> ID
%token EOF

%nonassoc NOELSE
%nonassoc ELSE
%right ASSIGN
%left OR
%left AND
%left EQ NEQ
%left LT GT LEQ GEQ
%left PLUS MINUS
%left TIMES DIVIDE MODULO
%right NOT NEG BORROW MUTABLEBORROW DEREF AS
%right RBRACK RPAREN RBRACE
%nonassoc NOACCESS
%left DOT LPAREN LBRACK LBRACE OUTPUT

%start program
%type <Ast.program> program

%%

program:
  decls EOF { $1 }

decls:
    /* nothing */ { [] }
  | decls fdecl   { $2 :: $1 }

fdecl: /* fn foo(x:int) -> int { ... } */
  FUNC ID LPAREN formals_opt RPAREN OUTPUT typ LBRACE stmt_list RBRACE 
  {
	 {
	  fname = $2;
	  formals = $4;
    outputType = $7;
	  body = List.rev $9 
	 }
	}

formals_opt:
    /* nothing */ { [] }
  | formal_list   { List.rev $1 }
   
typ:
	  INT   { DataT(IntT) }
  | BOOL  { DataT(BoolT) }
  | FLOAT { DataT(FloatT) }
	| VOID  { DataT(VoidT) }
  | CHAR  { DataT(CharT) }
  | STRING {StringT}
  | LBRACK typ SEMI ID RBRACK          { ArrayTD($2, $4) }
  | LBRACK typ SEMI INT_LITERAL RBRACK { ArrayT($2, $4) } /* [int;10] type is an integer array of length 10 */
  | BORROW typ                         { RefT(Immut, $2) } /* &int */
  | MUTABLEBORROW typ                  { RefT(Mut, $2) } /* &mut int */
  | ID                                 { StructT($1) }

formal_list:
    ID COLON typ                   { [($1, $3)] }
  | formal_list COMMA ID COLON typ { ($3, $5) :: $1 }

struct_list_opt:
  | struct_list { List.rev $1 }

struct_list:
	  ID COLON expr                   { [$1, $3] }
  | struct_list COMMA ID COLON expr { ($3, $5) :: $1 }


stmt_list:
  | stmt           { [$1] }
  | stmt_list stmt { $2 :: $1 }

stmt:
    expr SEMI        { Expr $1 }
  | RETURN SEMI      { Return Noexpr }
  | RETURN expr SEMI { Return $2 }
	| LET ID COLON typ ASSIGN expr SEMI         { Declaration(Immut, ($2, $4), $6) } /* Assignment */
  | LET MUTABLE ID COLON typ ASSIGN expr SEMI { Declaration(Mut, ($3, $5), $7) }
  | LBRACE stmt_list RBRACE                 { Block(List.rev $2) }
  | IF LPAREN expr RPAREN stmt %prec NOELSE { If($3, $5, Block([])) }
  | IF LPAREN expr RPAREN stmt ELSE stmt    { If($3, $5, $7) }
  | FOR LPAREN expr SEMI expr SEMI expr RPAREN stmt { For($3, $5, $7, $9) }
  | LOOP stmt                           { Loop($2) } /* loop {...} is an infinite loop */
  | WHILE LPAREN expr RPAREN stmt       { While($3, $5) }
  | STRUCT ID LBRACE formals_opt RBRACE { StructDef($2, $4) } /* Struct */
  | IMPL ID LBRACE fdecl RBRACE         { ImplDef($2, $4) }

expr_list:
	  /* nothing */ { [] }
  | expr { [$1] }
  | expr_list COMMA expr { $3 :: $1 }


expr:
    INT_LITERAL      { IntLit($1) } /* Literals */
  | FLOAT_LITERAL    { FloatLit($1) }
  | CHAR_LITERAL     { CharLit($1) }
  | STRING_LITERAL   { StringLit($1) }
  | TRUE             { BoolLit(true) }
  | FALSE            { BoolLit(false) }
  | ID %prec NOACCESS { Id($1) }
  | LBRACK expr_list RBRACK { ArrayLit(List.rev $2) } /* Array */
  | expr LBRACK expr RBRACK { ArrayAccess($1,$3) }
  | LBRACE struct_list_opt RBRACE { StructCreate($2) }
  | expr DOT ID  { StructAccess($1,$3) } /* Point.x is struct access */
  | expr PLUS   expr { Binop($1, Add,   $3) } /* Binary Operation */
  | expr MINUS  expr { Binop($1, Sub,   $3) }
  | expr TIMES  expr { Binop($1, Mult,  $3) }
  | expr DIVIDE expr { Binop($1, Div,   $3) }
  | expr MODULO expr { Binop($1, Mod,   $3) }
  | expr EQ     expr { Binop($1, Equal, $3) }
  | expr NEQ    expr { Binop($1, Neq,   $3) }
  | expr LT     expr { Binop($1, Less,  $3) }
  | expr LEQ    expr { Binop($1, Leq,   $3) }
  | expr GT     expr { Binop($1, Greater, $3) }
  | expr GEQ    expr { Binop($1, Geq,   $3) }
  | expr AND    expr { Binop($1, And,   $3) }
  | expr OR     expr { Binop($1, Or,    $3) }
  | expr AS     typ  { Cast($1, $3) } /* 32 as float */
  | expr ASSIGN  expr  { Binop($1, Assign, $3)} 
  | MINUS expr %prec NEG   { Unop(Neg, $2) } /* Unary Operation */
  | NOT expr               { Unop(Not, $2) }
  | TIMES expr %prec DEREF { Unop(Deref, $2) }
  | BORROW expr            { Unop(Borrow(Immut), $2)}
  | MUTABLEBORROW expr     { Unop(Borrow(Mut), $2)}
  | ID LPAREN actuals_opt RPAREN                { Call($1, $3) } /* Function Call */
  | ID COLON COLON ID LPAREN actuals_opt RPAREN { StructMethodCall($1, $4, $6) }   /* Call Struct Methods */
  | LPAREN expr RPAREN { $2 } /* Parenthesis */
  

actuals_opt:
    /* nothing */ { [] }
  | actuals_list  { List.rev $1 }

actuals_list:
    expr                    { [$1] }
  | actuals_list COMMA expr { $3 :: $1 }
