(* signed: Yanlin Duan, Zhuo Kong, Emily Meng, Shiyu Qiu *)

(* Abstract Syntax Tree and functions for printing it *)

type op = Add | Sub | Mult | Div | Mod |
          Equal | Neq | Less | Leq | Greater | Geq |
          And | Or | Assign | AS

type borrowType = Mut | Immut

type uop = Neg | Not | Borrow of borrowType | Deref

type primitive = IntT | BoolT | FloatT | CharT | VoidT

type typ = DataT of primitive |
					 StringT |
           StructT of string |
					 RefT of borrowType * typ |
					 ArrayT of typ * int|
           ArrayTD of typ * string
					 
and bind = string * typ

and variable_decl = borrowType * bind

and expr =
  | Noexpr
  | IntLit of int
  | BoolLit of bool
  | FloatLit of float
  | CharLit of char
  | ArrayLit of expr list
  | StringLit of string
  | ArrayAccess of expr * expr
  | StructCreate of ((string * expr) list)
  | StructAccess of expr * string
  | Id of string
  | Binop of expr * op * expr
  | Unop of uop * expr
  | Call of string * expr list
	| Cast of expr * typ
  | StructMethodCall of string * string * expr list

type stmt =
    Block of stmt list
	| Declaration of borrowType * bind * expr
	| Expr of expr
  | Return of expr
  | If of expr * stmt * stmt
  | For of expr * expr * expr * stmt
  | While of expr * stmt
  | Loop of stmt
  | StructDef of string * bind list
  | ImplDef of string * func_decl
  | Break

and func_decl = {
    fname : string;
    formals : bind list;
    outputType : typ;
    body : stmt list;
}

type program = func_decl list

(* Pretty-printing functions *)

let string_of_op = function
    Add -> "+"
  | Sub -> "-"
  | Mult -> "*"
  | Div -> "/"
  | Mod -> "%"
  | Equal -> "=="
  | Neq -> "!="
  | Less -> "<"
  | Leq -> "<="
  | Greater -> ">"
  | Geq -> ">="
  | And -> "and"
  | Or -> "or"
	| Assign -> "="
  | AS -> "AS"

let string_of_uop = function
    Neg -> "-"
  | Not -> "!"
  | Deref -> "*"
  | Borrow(Immut) -> "&"
  | Borrow(Mut) -> "&mut"

let rec string_of_expr = function
    StringLit(s) -> s
  | Call(f, el) ->
      f ^ "(" ^ String.concat ", " (List.map string_of_expr el) ^ ")"
  | Noexpr -> ""
  | IntLit(n) -> string_of_int n
  | BoolLit(b) -> string_of_bool b
  | FloatLit(f) -> string_of_float f
  | CharLit(c) -> Char.escaped c
  | _ -> "string_of_expr not implemented yet"

let string_of_primitive = function
    IntT -> "int"
  | BoolT -> "bool"
  | FloatT -> "float" 
  | CharT -> "char"
	| VoidT -> "void"

let rec string_of_typ = function
    DataT(t) -> string_of_primitive t
  | ArrayT(t,i) -> "["^string_of_typ t^";"^string_of_int  i^"]"
  | ArrayTD(t,s) -> "["^string_of_typ t^";"^ s ^"]"
  | StructT(t) -> t
  | StringT -> "string"
  | _ -> "string_of_typ no implementation yet."