module L = Llvm
module A = Ast
module Fcmp = Llvm.Fcmp
module StringMap = Map.Make(String)

let translate(globals,functions) =
    let context = L.global_context () in
    let the_module = L.create_module context "tape"
	
    and i32_t  = L.i32_type context
	and i8_t   = L.i8_type context
	and i1_t   = L.i1_type context (*bool*)
	and ptr_t  = L.pointer_type (L.i8_type context)
	and void_t = L.void_type context in

    let ltype_of_typ = function
	  A.Int -> i32_t
	| A.String -> ptr_t
	| A.Void -> void_t
	| A.Bool -> i1_t
    | A.Char -> i8_t
       
    (*There may have more things need to be put*)
    in
    (*global variable*)
    let global_vars = 
	let global_var m (t,n) = 
	    let init = L.const_int (ltype_of_typ t) 0
	    in StringMap.add n (L.define_global n init the_module) m in
	List.fold_left global_var StringMap.empty globals in

    let function_decls = 
	let function_decl m fdecl = 
	    let name  = fdecl.A.fname
	    and formal_types = Array.of_list (List.map (fun (t,_) -> ltype_of_typ t) fdecl.A.formals)
	in
	    let ftype = L.function_type (ltype_of_typ fdecl.A.typ) formal_types in
		StringMap.add name (L.define_function name ftype the_module, fdecl) m
	in 
	    List.fold_left function_decl StringMap.empty functions
    in

 
    (*declare external function printf*)
    let printf_t = L.var_arg_function_type i32_t [|L.pointer_type i8_t |] in  
    let printf_func = L.declare_function "printf" printf_t the_module in
    
    let prints_t = L.var_arg_function_type ptr_t [|L.pointer_type i8_t|] in
    let prints_func = L.declare_function "puts" prints_t the_module in

    (*file open and close*)
    let open_file_t = L.function_type ptr_t [| L.pointer_type i8_t;L.pointer_type i8_t |] in
    let open_file_func = L.declare_function "fopen" open_file_t the_module in

    let close_file_t = L.function_type i32_t [| i32_t |] in
    let close_file_func = L.declare_function "fclose" close_file_t the_module in
   
    let write_t = L.function_type i32_t [| i32_t; ptr_t |] in 
    let write_func = L.declare_function "fputs" write_t the_module in

    let get_t = L.function_type ptr_t [|ptr_t; i32_t; ptr_t|] in 
    let get_func = L.declare_function "fgets" get_t the_module in
    
    let fwrite_t = L.function_type i32_t [|ptr_t; i32_t; i32_t; ptr_t|] in
    let fwrite_func = L.declare_function "fwrite" fwrite_t the_module in

    let read_t = L.function_type i32_t [|ptr_t; i32_t; i32_t; ptr_t|] in
    let read_func = L.declare_function "fread" read_t the_module in

    let toupper_t = L.function_type i8_t [| i8_t |] in
    let toupper_func = L.declare_function "toupper" toupper_t the_module in

    let tolower_t = L.function_type i8_t [| i8_t |] in
    let tolower_func = L.declare_function "tolower" tolower_t the_module in

    let calloc_t = L.function_type ptr_t [|i32_t; i32_t|] in
    let calloc_fun = L.declare_function "calloc" calloc_t the_module in

    let strfind_t = L.function_type ptr_t [|ptr_t;ptr_t|] in
    let strfind_func = L.declare_function "strstr" strfind_t the_module in

    let memcpy_t = L.function_type ptr_t [|ptr_t; ptr_t; i32_t|] in
    let memcpy_func = L.declare_function "memcpy" memcpy_t the_module in
    
    let strlen_t = L.function_type i32_t [|ptr_t|] in
    let strlen_func = L.declare_function "strlen" strlen_t the_module in

    (*build function body - fill in the body of the given function*)
    let build_function_body fdecl = 
	let (the_function, _) = 
	    StringMap.find fdecl.A.fname function_decls in
	let builder = (*create an instruction builder*)
	    L.builder_at_end context (L.entry_block the_function) in 
	let int_format_str = (*Format string for printf calls*)
	    L.build_global_stringptr "%d\n" "fmt" builder in 
    let char_format_str = L.build_global_stringptr "%c\n" "fmt" builder in

    (* formals and locals  *)
    let local_vars = 
	let add_formal m (t,n) p = L.set_value_name n p;
	    let local = L.build_alloca (ltype_of_typ t) n builder in
	    ignore (L.build_store p local builder); 
	    StringMap.add n local m in
        
        let add_local m (t, n) = 
          let local_var = L.build_alloca (ltype_of_typ t) n builder
          in StringMap.add n local_var m in

        let formals = List.fold_left2 add_formal StringMap.empty
	    fdecl.A.formals (Array.to_list (L.params the_function)) in 
	List.fold_left add_local formals fdecl.A.locals in

    (*look up  for a variable among locals/formal arguments, then the golbals*)
    let lookup n = try StringMap.find n local_vars
		   with Not_found -> StringMap.find n global_vars
    in

     
    (*expression*)
    let rec expr builder = function
	  A.Literal i -> L.const_int i32_t i   (*boolean not included*)
	| A.Noexpr ->	L.const_int i32_t 0
    | A.Char_Lit c -> L.const_int i8_t (Char.code c)
    | A.StringLit s -> L.build_load (lookup s) s builder
    | A.NewstringLit sl -> L.build_global_stringptr sl "string" builder
    | A.Array(e1,e2) -> let para1=(expr builder (A.StringLit e1))
        and  para2=(expr builder e2) in 
        let k=L.build_in_bounds_gep para1 [|para2|] "tmpp" builder in
        L.build_load k "deref" builder
    | A.Arrayassign(e1,e2,e3) -> let para1 = (expr builder (A.StringLit e1))
        and para2= (expr builder e2)
        and para3= (expr builder e3)
        in let k=L.build_in_bounds_gep para1 [|para2|] "tmpp" builder in
        L.build_store para3 k builder
    | A.Init(e1,e2) -> let cnt1=(lookup e1) and cnt2= expr builder e2 in
        let tp= L.element_type (L.type_of cnt1) in 
        let sz=L.size_of tp in
        let sz1=L.build_intcast sz (i32_t) "intc" builder in
        let dt=L.build_bitcast (L.build_call calloc_fun [|cnt2;sz1|] "tmpa" builder) tp "tmpb" builder in
        L.build_store dt cnt1 builder
    | A.Assign (s,e) -> let e' = expr builder e in
		ignore (L.build_store e' (lookup s) builder); e'
	| A.Binop (e1, op, e2) ->
	    let e1' = expr builder e1
	    and e2' = expr builder e2 in
	    (match op with 
		A.Plus -> L.build_add
	      | A.Minus -> L.build_sub
	      | A.Times -> L.build_mul
          | A.Equal -> L.build_icmp L.Icmp.Eq  (*Fcmp is a module imported, no unequal*)
          | A.Unequal -> L.build_icmp L.Icmp.Ne
          | A.Less  -> L.build_icmp L.Icmp.Slt
	      | A.Great -> L.build_icmp L.Icmp.Sgt
	      | A.LessEQ -> L.build_icmp L.Icmp.Sle
	      | A.GreatEQ -> L.build_icmp L.Icmp.Sge
	    ) e1' e2' "tmp" builder
	| A.Unop(op, e) ->
	    let e' = expr builder e in
	    (match op with
		A.Not -> L.build_not) e' "tmp" builder

	(*build in function filled below*)
        | A.Call("print_i",[e]) ->
            L.build_call printf_func [|int_format_str; (expr builder e)|]
            "printf" builder

	    | A.Call("print_s",[e]) -> 
            L.build_call prints_func [|(expr builder e)|]
            "puts" builder
        | A.Call("print_c",[e]) ->
                L.build_call printf_func [|char_format_str; (expr builder e)|]
                "printf" builder


        | A.Call("toupper",[e]) ->
            L.build_call toupper_func [|(expr builder e)|]
            "toupper" builder

        | A.Call("tolower",[e]) ->
            L.build_call tolower_func [|(expr builder e)|]
            "tolower" builder
        | A.Call("write",e) -> let actuals= List.rev (List.map (expr builder) (List.rev e)) in
            L.build_call fwrite_func (Array.of_list actuals) "tmpy" builder

        | A.Call("open", e) -> let actuals= List.rev (List.map (expr builder) (List.rev e)) in
            L.build_call open_file_func (Array.of_list actuals) "fopen" builder
        | A.Call("fget",e) -> let actuals= List.rev (List.map (expr builder) (List.rev e)) in
            L.build_call get_func (Array.of_list actuals) "tmpz" builder

        | A.Call("read", e) -> let actuals = List.rev (List.map (expr builder) (List.rev e)) in
            L.build_call read_func (Array.of_list actuals) "tmpx" builder
	
        | A.Call("find", e)-> let actuals = List.rev (List.map (expr builder) (List.rev e)) in
            L.build_call strfind_func (Array.of_list actuals) "find" builder
        
        | A.Call("cpy",e) -> let actuals = List.rev (List.map (expr builder) (List.rev e)) in
            L.build_call memcpy_func (Array.of_list actuals) "mcpy" builder

        | A.Call("length",e) -> let actuals = List.rev (List.map (expr builder) (List.rev e)) in
            L.build_call strlen_func (Array.of_list actuals) "len" builder
        
        | A.Call (f, act) ->
	    let (fdef, fdecl) = StringMap.find f function_decls in 
   
  let actuals = List.rev (List.map (expr builder) (List.rev act)) in
  let result = (match fdecl.A.typ with A.Void -> ""
						| _ -> f ^ "_result") in
	  L.build_call fdef (Array.of_list actuals) result builder
    in   
    (*statements*)	
    let add_terminal builder f = 
	match L.block_terminator (L.insertion_block builder) with
	    Some _ -> ()
	  | None -> ignore (f builder) in
    let rec stmt builder = function
	A.Block sl -> List.fold_left stmt builder sl

      | A.Expr e -> ignore (expr builder e); builder
      | A.Return e -> ignore (match fdecl.A.typ with 
	  A.Void -> L.build_ret_void builder
	  | _ -> L.build_ret (expr builder e) builder); builder
      (*If Statement*)
      | A.If (predicate, then_stmt, else_stmt) ->
         let bool_val = expr builder predicate in
      let merge_bb = L.append_block context "merge" the_function in
       
      let then_bb = L.append_block context "then" the_function in
      add_terminal (stmt (L.builder_at_end context then_bb) then_stmt)
        (L.build_br merge_bb);
       
      let else_bb = L.append_block context "else" the_function in
      add_terminal (stmt (L.builder_at_end context else_bb) else_stmt)
        (L.build_br merge_bb);
      
      ignore (L.build_cond_br bool_val then_bb else_bb builder);
      L.builder_at_end context merge_bb
      (*While Statement *)
        | A.While (predicate, body) ->
        let pred_bb = L.append_block context "while" the_function in
        ignore (L.build_br pred_bb builder);
        
        let body_bb = L.append_block context "while_body" the_function in
        add_terminal (stmt (L.builder_at_end context body_bb) body)
          (L.build_br pred_bb);
        
        let pred_builder = L.builder_at_end context pred_bb in
        let bool_val = expr pred_builder predicate in
        
        let merge_bb = L.append_block context "merge" the_function in
        ignore (L.build_cond_br bool_val body_bb merge_bb pred_builder);
        L.builder_at_end context merge_bb
      (*For Statements*)
      
        | A.For (e1, e2, e3, body) -> stmt builder
          ( A.Block [A.Expr e1 ; A.While (e2, A.Block [body ; A.Expr e3]) ] )
      in
      
      let builder = stmt builder (A.Block fdecl.A.body) in 

      add_terminal builder (match fdecl.A.typ with
          A.Void -> L.build_ret_void
        | t-> L.build_ret(L.const_int (ltype_of_typ t) 0))

    in

    List.iter build_function_body functions;
    the_module



