/*
 * Seven-segment LED emulator
 *
 * Stephen A. Edwards, Columbia University
 */

module BALL_Emulator(
 input logic 	    clk50, reset,
 input logic [31:0] over,
 input logic [31:0] on,
 input logic peng,
 input logic [9:0]  x_coords,y_coords,
 input logic [31:0] score, life,
 input logic [9:0] px, //paddle address
 output logic [7:0] VGA_R, VGA_G, VGA_B,
 output logic 	    VGA_CLK, VGA_HS, VGA_VS, VGA_BLANK_n, VGA_SYNC_n);

/*
 * 640 X 480 VGA timing for a 50 MHz clock: one pixel every other cycle
 * 
 * HCOUNT 1599 0             1279       1599 0
 *             _______________              ________
 * ___________|    Video      |____________|  Video
 * 
 * 
 * |SYNC| BP |<-- HACTIVE -->|FP|SYNC| BP |<-- HACTIVE
 *       _______________________      _____________
 * |____|       VGA_HS          |____|
 */
   // Parameters for hcount
   parameter HACTIVE      = 11'd 1280,
             HFRONT_PORCH = 11'd 32,
             HSYNC        = 11'd 192,
             HBACK_PORCH  = 11'd 96,   
             HTOTAL       = HACTIVE + HFRONT_PORCH + HSYNC + HBACK_PORCH; // 1600
   
   // Parameters for vcount
   parameter VACTIVE      = 10'd 480,
             VFRONT_PORCH = 10'd 10,
             VSYNC        = 10'd 2,
             VBACK_PORCH  = 10'd 33,
             VTOTAL       = VACTIVE + VFRONT_PORCH + VSYNC + VBACK_PORCH; // 525

	  
   logic [10:0]			     hcount; // Horizontal counter
                                             // Hcount[10:1] indicates pixel column (0-639)
   logic 			     endOfLine;
   
   always_ff @(posedge clk50 or posedge reset)
     if (reset)          hcount <= 0;
     else if (endOfLine) hcount <= 0;
     else  	         hcount <= hcount + 11'd 1;

   assign endOfLine = hcount == HTOTAL - 1;

   // Vertical counter
   logic [9:0] 			     vcount;
   logic 			     endOfField;
   
   always_ff @(posedge clk50 or posedge reset)
     if (reset)          vcount <= 0;
     else if (endOfLine)
       if (endOfField)   vcount <= 0;
       else              vcount <= vcount + 10'd 1;

   assign endOfField = vcount == VTOTAL - 1;

   // Horizontal sync: from 0x520 to 0x5DF (0x57F)
   // 101 0010 0000 to 101 1101 1111
   assign VGA_HS = !( (hcount[10:8] == 3'b101) & !(hcount[7:5] == 3'b111));
   assign VGA_VS = !( vcount[9:1] == (VACTIVE + VFRONT_PORCH) / 2);

   assign VGA_SYNC_n = 1; // For adding sync to video signals; not used for VGA
   
   // Horizontal active: 0 to 1279     Vertical active: 0 to 479
   // 101 0000 0000  1280	       01 1110 0000  480
   // 110 0011 1111  1599	       10 0000 1100  524
   assign VGA_BLANK_n = !( hcount[10] & (hcount[9] | hcount[8]) ) &
			!( vcount[9] | (vcount[8:5] == 4'b1111) );   

   /* VGA_CLK is 25 MHz
    *             __    __    __
    * clk50    __|  |__|  |__|
    *        
    *             _____       __
    * hcount[0]__|     |_____|
    */
   assign VGA_CLK = hcount[0]; // 25 MHz clock: pixel latched on rising edge
	
	
	logic inZone1, inZone2, inZone3, inZone4, inZone5, inZone6, inZone7, inZone8, inZone9, inZone10;
   // inZone11, inZone12;
/*
   assign inZone1 = ((hcount >> 1) == 10'd320) && (vcount >= 0 && vcount < 10'd60) || (((hcount >> 1) < 10'd320) && (vcount < 10'd60) && (vcount/(10'd320 - (hcount >> 1)) >= 1) || ((hcount >> 1) < 10'd320) && (vcount >= 10'd60) && (vcount/(10'd320 - (hcount >> 1)) >= 1) && ((vcount - 10'd60)/(10'd320 - (hcount >> 1)) < 1)) || (((hcount >> 1) >10'd320) && (vcount < 10'd60) && (vcount/((hcount >> 1) - 10'd320) >= 1) || ((hcount >> 1) > 10'd320) && (vcount >= 10'd60) && (vcount/((hcount >> 1) - 10'd320) >= 1) && ((vcount - 10'd60)/((hcount >> 1) - 10'd320) < 1));

   assign inZone2 = ((hcount >> 1) == 10'd320) && (vcount >= 10'd120 && vcount < 10'd180) || (((hcount >> 1) < 10'd320) && (vcount >= 10'd120 && vcount < 10'd180) && ((vcount - 10'd120)/(10'd320 - (hcount >> 1)) >= 1) || ((hcount >> 1) < 10'd320) && (vcount >= 10'd180) && ((vcount - 10'd120)/(10'd320 - (hcount >> 1)) >= 1) && ((vcount - 10'd180)/(10'd320 - (hcount >> 1)) < 1)) || (((hcount >> 1) >10'd320) && (vcount >= 10'd120 && vcount < 10'd180) && ((vcount - 10'd120)/((hcount >> 1) - 10'd320) >= 1) || ((hcount >> 1) > 10'd320) && (vcount >= 10'd180) && ((vcount - 10'd120)/((hcount >> 1) - 10'd320) >= 1) && ((vcount - 10'd180)/((hcount >> 1) - 10'd320) < 1));
   
   assign inZone3 = ((hcount >> 1) == 10'd320) && (vcount >= 10'd240 && vcount < 10'd300) || (((hcount >> 1) < 10'd320) && (vcount >= 10'd240 && vcount < 10'd300) && ((vcount - 10'd240)/(10'd320 - (hcount >> 1)) >= 1) || ((hcount >> 1) < 10'd320) && (vcount >= 10'd300) && ((vcount - 10'd240)/(10'd320 - (hcount >> 1)) >= 1) && ((vcount - 10'd300)/(10'd320 - (hcount >> 1)) < 1)) || (((hcount >> 1) >10'd320) && (vcount >= 10'd240 && vcount < 10'd300) && ((vcount - 10'd240)/((hcount >> 1) - 10'd320) >= 1) || ((hcount >> 1) > 10'd320) && (vcount >= 10'd300) && ((vcount - 10'd240)/((hcount >> 1) - 10'd320) >= 1) && ((vcount - 10'd300)/((hcount >> 1) - 10'd320) < 1));

   assign inZone4 = ((hcount >> 1) == 10'd320) && (vcount >= 10'd360 && vcount < 10'd420) || (((hcount >> 1) < 10'd320) && (vcount >= 10'd360 && vcount < 10'd420) && ((vcount - 10'd360)/(10'd320 - (hcount >> 1)) >= 1) || ((hcount >> 1) < 10'd320) && (vcount >= 10'd420) && ((vcount - 10'd360)/(10'd320 - (hcount >> 1)) >= 1) && ((vcount - 10'd420)/(10'd320 - (hcount >> 1)) < 1)) || (((hcount >> 1) >10'd320) && (vcount >= 10'd360 && vcount < 10'd420) && ((vcount - 10'd360)/((hcount >> 1) - 10'd320) >= 1) || ((hcount >> 1) > 10'd320) && (vcount >= 10'd420) && ((vcount - 10'd360)/((hcount >> 1) - 10'd320) >= 1) && ((vcount - 10'd420)/((hcount >> 1) - 10'd320) < 1));

   assign inZone5 = (vcount == 0) && ((hcount >> 1) > 10'd200 && (hcount >> 1) <= 10'd260) || ((hcount >> 1) >= 10'd200 && (hcount >> 1) < 10'd260) && (2 * vcount/(10'd260 - (hcount >> 1)) <= 1) || ((hcount >> 1) < 10'd200) && (2 * vcount/(10'd200 - (hcount >> 1)) > 1) && (2 * vcount/(10'd260 - (hcount >> 1)) <= 1);

   assign inZone6 = (vcount == 0) && ((hcount >> 1) > 10'd80 && (hcount >> 1) <= 10'd140) || ((hcount >> 1) >= 10'd80 && (hcount >> 1) < 10'd140) && (2 * vcount/(10'd140 - (hcount >> 1)) <= 1) || ((hcount >> 1) < 10'd80) && (2 * vcount/(10'd80 - (hcount >> 1)) > 1) && (2 * vcount/(10'd140 - (hcount >> 1)) <= 1);

   assign inZone7 = (vcount == 0) && ((hcount >> 1) == 10'd20) || ((hcount >> 1) < 10'd20) && (2 * vcount/(10'd20 - (hcount >> 1)) <= 1);

   assign inZone8 = (vcount == 0) && ((hcount >> 1) < 10'd440 && (hcount >> 1) >= 10'd380) || ((hcount >> 1) <= 10'd440 && (hcount >> 1) > 10'd380) && (2 * vcount/((hcount >> 1) - 10'd380) <= 1) || ((hcount >> 1) > 10'd440) && (2 * vcount/((hcount >> 1) - 10'd440) > 1) && (2 * vcount/((hcount >> 1) - 10'd380) <= 1);

   assign inZone9 = (vcount == 0) && ((hcount >> 1) < 10'd560 && (hcount >> 1) >= 10'd500) || ((hcount >> 1) <= 10'd560 && (hcount >> 1) > 10'd500) && (2 * vcount/((hcount >> 1) - 10'd500) <= 1) || ((hcount >> 1) > 10'd560) && (2 * vcount/((hcount >> 1) - 10'd560) > 1) && (2 * vcount/((hcount >> 1) - 10'd500) <= 1);

   assign inZone10 = (vcount == 0) && ((hcount >> 1) == 10'd620) || ((hcount >> 1) > 10'd620) && (2 * vcount/((hcount >> 1) - 10'd620) <= 1);
 */
/*
 logic z1_1, z1_2, z1_3, z1_4, z1_5, z2_1, z2_2, z2_3, z2_4, z2_5, z3_1, z3_2, z3_3, z3_4, z3_5, z4_1, z4_2, z4_3, z4_4, z4_5, z5_1, z5_2, z5_3, z6_1, z6_2, z6_3, z7_1, z7_2, z8_1, z8_2, z8_3, z9_1, z9_2, z9_3, z10_1, z10_2;

   assign z1_1 = ((hcount[10:1]) == 10'd320) && (vcount >= 0 && vcount < 10'd60);
   assign z1_2 = ((hcount[10:1]) < 10'd320) && (vcount < 10'd60) && (vcount/(10'd320 - (hcount[10:1])) >= 1);
   assign z1_3 = ((hcount[10:1]) < 10'd320) && (vcount >= 10'd60) && (vcount/(10'd320 - (hcount[10:1])) >= 1) && ((vcount - 10'd60)/(10'd320 - (hcount[10:1])) < 1);
   assign z1_4 = ((hcount[10:1]) >10'd320) && (vcount < 10'd60) && (vcount/((hcount[10:1]) - 10'd320) >= 1);
   assign z1_5 = ((hcount[10:1]) > 10'd320) && (vcount >= 10'd60) && (vcount/((hcount[10:1]) - 10'd320) >= 1) && ((vcount - 10'd60)/((hcount[10:1]) - 10'd320) < 1);
   assign inZone1 = z1_1 || z1_2 || z1_3 || z1_4 || z1_5;

   assign z2_1 = ((hcount[10:1]) == 10'd320) && (vcount >= 10'd120 && vcount < 10'd180);
   assign z2_2 = ((hcount[10:1]) < 10'd320) && (vcount >= 10'd120 && vcount < 10'd180) && ((vcount - 10'd120)/(10'd320 - (hcount[10:1])) >= 1);
   assign z2_3 = ((hcount[10:1]) < 10'd320) && (vcount >= 10'd180) && ((vcount - 10'd120)/(10'd320 - (hcount[10:1])) >= 1) && ((vcount - 10'd180)/(10'd320 - (hcount[10:1])) < 1);
   assign z2_4 = ((hcount[10:1]) >10'd320) && (vcount >= 10'd120 && vcount < 10'd180) && ((vcount - 10'd120)/((hcount[10:1]) - 10'd320) >= 1);
   assign z2_5 = ((hcount[10:1]) > 10'd320) && (vcount >= 10'd180) && ((vcount - 10'd120)/((hcount[10:1]) - 10'd320) >= 1) && ((vcount - 10'd180)/((hcount[10:1]) - 10'd320) < 1);
   assign inZone2 = z2_1 || z2_2 || z2_3 || z2_4 || z2_5;

   assign z3_1 = ((hcount[10:1]) == 10'd320) && (vcount >= 10'd240 && vcount < 10'd300);
   assign z3_2 = ((hcount[10:1]) < 10'd320) && (vcount >= 10'd240 && vcount < 10'd300) && ((vcount - 10'd240)/(10'd320 - (hcount[10:1])) >= 1);
   assign z3_3 = ((hcount[10:1]) < 10'd320) && (vcount >= 10'd300) && ((vcount - 10'd240)/(10'd320 - (hcount[10:1])) >= 1) && ((vcount - 10'd300)/(10'd320 - (hcount[10:1])) < 1);
   assign z3_4 = ((hcount[10:1]) >10'd320) && (vcount >= 10'd240 && vcount < 10'd300) && ((vcount - 10'd240)/((hcount[10:1]) - 10'd320) >= 1);
   assign z3_5 = ((hcount[10:1]) > 10'd320) && (vcount >= 10'd300) && ((vcount - 10'd240)/((hcount[10:1]) - 10'd320) >= 1) && ((vcount - 10'd300)/((hcount[10:1]) - 10'd320) < 1);
   assign inZone3 = z3_1 || z3_2 || z3_3 || z3_4 || z3_5;

   assign z4_1 = ((hcount[10:1]) == 10'd320) && (vcount >= 10'd360 && vcount < 10'd420);
   assign z4_2 = (((hcount[10:1]) < 10'd320) && (vcount >= 10'd360 && vcount < 10'd420) && ((vcount - 10'd360)/(10'd320 - (hcount[10:1])) >= 1));
   assign z4_3 = ((hcount[10:1]) < 10'd320) && (vcount >= 10'd420) && ((vcount - 10'd360)/(10'd320 - (hcount[10:1])) >= 1) && ((vcount - 10'd420)/(10'd320 - (hcount[10:1])) < 1);
   assign z4_4 = ((hcount[10:1]) >10'd320) && (vcount >= 10'd360 && vcount < 10'd420) && ((vcount - 10'd360)/((hcount[10:1]) - 10'd320) >= 1);
   assign z4_5 = ((hcount[10:1]) > 10'd320) && (vcount >= 10'd420) && ((vcount - 10'd360)/((hcount[10:1]) - 10'd320) >= 1) && ((vcount - 10'd420)/((hcount[10:1]) - 10'd320) < 1);
   assign inZone4 = z4_1 || z4_2 || z4_3 || z4_4 || z4_5;

   assign z5_1 = (vcount == 0) && ((hcount[10:1]) > 10'd200 && (hcount[10:1]) <= 10'd260);
   assign z5_2 = ((hcount[10:1]) >= 10'd200 && (hcount[10:1]) < 10'd260) && (2 * vcount/(10'd260 - (hcount[10:1])) <= 1);
   assign z5_3 = ((hcount[10:1]) < 10'd200) && (2 * vcount/(10'd200 - (hcount[10:1])) > 1) && (2 * vcount/(10'd260 - (hcount[10:1])) <= 1);
   assign inZone5 = z5_1 || z5_2 || z5_3;

   assign z6_1 = (vcount == 0) && ((hcount[10:1]) > 10'd80 && (hcount[10:1]) <= 10'd140);
   assign z6_2 = ((hcount[10:1]) >= 10'd80 && (hcount[10:1]) < 10'd140) && (2 * vcount/(10'd140 - (hcount[10:1])) <= 1);
   assign z6_3 = ((hcount[10:1]) < 10'd80) && (2 * vcount/(10'd80 - (hcount[10:1])) > 1) && (2 * vcount/(10'd140 - (hcount[10:1])) <= 1);
   assign inZone6 = z6_1 || z6_2 || z6_3;

   assign z7_1 = (vcount == 0) && ((hcount[10:1]) == 10'd20);
   assign z7_2 = ((hcount[10:1]) < 10'd20) && (2 * vcount/(10'd20 - (hcount[10:1])) <= 1);
   assign inZone7 = z7_1 || z7_2;

   assign z8_1 = (vcount == 0) && ((hcount[10:1]) < 10'd440 && (hcount[10:1]) >= 10'd380);
   assign z8_2 = ((hcount[10:1]) <= 10'd440 && (hcount[10:1]) > 10'd380) && (2 * vcount/((hcount[10:1]) - 10'd380) <= 1);
   assign z8_3 = ((hcount[10:1]) > 10'd440) && (2 * vcount/((hcount[10:1]) - 10'd440) > 1) && (2 * vcount/((hcount[10:1]) - 10'd380) <= 1);
   assign inZone8 = z8_1 || z8_2 || z8_3;

   assign z9_1 = (vcount == 0) && ((hcount[10:1]) < 10'd560 && (hcount[10:1]) >= 10'd500);
   assign z9_2 = ((hcount[10:1]) <= 10'd560 && (hcount[10:1]) > 10'd500) && (2 * vcount/((hcount[10:1]) - 10'd500) <= 1);
   assign z9_3 = ((hcount[10:1]) > 10'd560) && (2 * vcount/((hcount[10:1]) - 10'd560) > 1) && (2 * vcount/((hcount[10:1]) - 10'd500) <= 1);
   assign inZone9 = z9_1 || z9_2 || z9_3;

   assign z10_1 = (vcount == 0) && ((hcount[10:1]) == 10'd620);
   assign z10_2 = ((hcount[10:1]) > 10'd620) && (2 * vcount/((hcount[10:1]) - 10'd620) <= 1);
   assign inZone10 = z10_1 || z10_2; 

*/

logic[26:0]    changeColor;

   always_ff @(posedge clk50 or posedge reset)
     if (reset)  changeColor <= 0;
     else if (changeColor[25:0] == 26'h3ffffff)  begin
        changeColor[26] <= 1 - changeColor[26];
        changeColor[25:0] <= 0;
     end
     else   changeColor <= changeColor + 27'd 1;


logic[6:0] line;
//always_ff @(posedge clk50) begin
//
//     if(vcount[6:0] > 7'd119) begin
//         line = vcount[6:0] - 7'd120;
//     end
//     else
//         line = vcount[6:0];
//end

assign line = vcount%120;

logic[19:0] left_sprite_data;
logic background;

left_rom left_unit(
      .clock(clk50),
      .address(line),
      .q(left_sprite_data)
);


logic[19:0] right_sprite_data;

right_rom right_unit(
      .clock(clk50),
      .address(line),
      .q(right_sprite_data)
);

logic[59:0] mid_sprite_data;
logic[1:0] mid_sprite;
logic[8:0] mid_address;

mid_rom mid_unit(
      .clock(clk50),
      .address(mid_address),
      .q(mid_sprite_data)
);



   logic isBall; // hcount or vcount are close to the center of the ball
   
   logic [10:0] 			     x_dist; 
   logic [10:0] 			     y_dist; 
	
	//
	logic  [5:0]				  character_address;
	logic 							rom_output;
	logic [7:0]						char_rom_data;
	
	/* x distance has to be divided by two since x_axis has double the pixels for some reason */
	assign x_dist = (x_coords > hcount[10:1]) ? ((x_coords - hcount[10:1])):((hcount[10:1] - x_coords));
	assign y_dist = (y_coords > vcount) ? (y_coords - vcount):(vcount - y_coords);

   assign isBall = ( (x_dist*x_dist + y_dist*y_dist) < 200);
	
	
	//
	char_gen_rom char_rom_unit(
			.clock(clk50),
			.address({character_address,vcount[2:0]}),
			.q(char_rom_data)
	);

	logic [23:0] brick_output;
	logic [14:0] brick_address;
	logic [3:0]  brick_type;//100.011.010,001,000 brick_type=4'o3 {brick_type, brick_type*32+hcount[7:1]}
	
	bricks brick_unit(
	.clock(clk50),
	.address(brick_address),
	.q(brick_output)
	);
	logic en;

	always_ff @(posedge clk50) begin
		
		if(on[0] && hcount[10:8] == 3'd0 && vcount[8:5] == 4'd1 ) begin
			en <= 1'b1;
			brick_address<=hcount[10:8]*32*128+vcount[4:0]*128+hcount[7:1];
		end
		else if(on[1] && hcount[10:8] == 3'd1 && vcount[8:5] == 4'd1 ) begin
			en <= 1'b1;
			brick_address<=hcount[10:8]*32*128+vcount[4:0]*128+hcount[7:1];
			end
		else if(on[2] && hcount[10:8] == 3'd2 && vcount[8:5] == 4'd1 ) begin
			en <= 1'b1;
			brick_address<=hcount[10:8]*32*128+vcount[4:0]*128+hcount[7:1];
			end
		else if(on[3] && hcount[10:8] == 3'd3 && vcount[8:5] == 4'd1) begin
			en <= 1'b1;
			brick_address<=(hcount[10:8])*32*128+vcount[4:0]*128+hcount[7:1];
		end 
		else if(on[4] && hcount[10:8] == 3'd4 && vcount[8:5] == 4'd1 ) begin
			en <= 1'b1;
			brick_address<=hcount[10:8]*32*128+vcount[4:0]*128+hcount[7:1];
			end
			
		else if(on[5] && hcount[10:8] == 3'd0 && vcount[8:5] == 4'd2 ) begin
			en <= 1'b1;
			brick_address<= (1'b1+hcount[10:8])*32*128+vcount[4:0]*128+hcount[7:1];
		end
		else if(on[6] && hcount[10:8] == 3'd1 && vcount[8:5] == 4'd2 ) begin
			en <= 1'b1;
			brick_address<=(1'b1+hcount[10:8])*32*128+vcount[4:0]*128+hcount[7:1];
			end
		else if(on[7] && hcount[10:8] == 3'd2 && vcount[8:5] == 4'd2 ) begin
			en <= 1'b1;
			brick_address<=(1'b1+hcount[10:8])*32*128+vcount[4:0]*128+hcount[7:1];
			end
		else if(on[8] && hcount[10:8] == 3'd3 && vcount[8:5] == 4'd2) begin
			en <= 1'b1;
			brick_address<=(hcount[10:8]+1'b1)*32*128+vcount[4:0]*128+hcount[7:1];
		end 
			
		
		else if(on[10] && hcount[10:8] == 3'd0 && vcount[8:5] == 4'd3 ) begin
			en <= 1'b1;
			brick_address<=(2'd2+hcount[10:8])*32*128+vcount[4:0]*128+hcount[7:1];
		end
		else if(on[11] && hcount[10:8] == 3'd1 && vcount[8:5] == 4'd3 ) begin
			en <= 1'b1;
			brick_address<=(2'd2+hcount[10:8])*32*128+vcount[4:0]*128+hcount[7:1];
			end
		else if(on[13] && hcount[10:8] == 3'd3 && vcount[8:5] == 4'd3) begin
			en <= 1'b1;
			brick_address<=(hcount[10:8]+2'd2)*32*128+vcount[4:0]*128+hcount[7:1];
		end 
			else if(on[14] && hcount[10:8] == 3'd4 && vcount[8:5] == 4'd3) begin
			en <= 1'b1;
			brick_address<= (hcount[10:8]-3'd4) *32*128+vcount[4:0]*128+hcount[7:1];
		end 
		
		else if(on[15] && hcount[10:8] == 3'd0 && vcount[8:5] == 4'd4 ) begin
			en <= 1'b1;
			brick_address<=(2'd3+hcount[10:8])*32*128+vcount[4:0]*128+hcount[7:1];
		end
		else if(on[16] && hcount[10:8] == 3'd1 && vcount[8:5] == 4'd4 ) begin
			en <= 1'b1;
			brick_address<=(2'd3+hcount[10:8])*32*128+vcount[4:0]*128+hcount[7:1];
			end
		else if(on[17] && hcount[10:8] == 3'd2 && vcount[8:5] == 4'd4 ) begin
			en <= 1'b1;
			brick_address<=(2'd3+hcount[10:8])*32*128+vcount[4:0]*128+hcount[7:1];
			end
		else if(on[19] && hcount[10:8] == 3'd4 && vcount[8:5] == 4'd4) begin
			en <= 1'b1;
			brick_address<= (hcount[10:8]-2'd3) *32*128+vcount[4:0]*128+hcount[7:1];
		end 
		
		else if(on[20] && hcount[10:8] == 3'd0 && vcount[8:5] == 4'd5 ) begin
			en <= 1'b1;
			brick_address<=(3'd3+hcount[10:8])*32*128+vcount[4:0]*128+hcount[7:1];
		end
		else if(on[21] && hcount[10:8] == 3'd1 && vcount[8:5] == 4'd5 ) begin
			en <= 1'b1;
			brick_address<=(hcount[10:8])*32*128+vcount[4:0]*128+hcount[7:1];
			end
		else if(on[22] && hcount[10:8] == 3'd2 && vcount[8:5] == 4'd5 ) begin
			en <= 1'b1;
			brick_address<=(hcount[10:8]-3'd2)*32*128+vcount[4:0]*128+hcount[7:1];
			end
		else if(on[23] && hcount[10:8] == 3'd3 && vcount[8:5] == 4'd5) begin
			en <= 1'b1;
			brick_address<= (hcount[10:8]+3'd1) *32*128+vcount[4:0]*128+hcount[7:1];
		end 
			else if(on[24] && hcount[10:8] == 3'd4 && vcount[8:5] == 4'd5) begin
			en <= 1'b1;
			brick_address<= (hcount[10:8]-3'd2) *32*128+vcount[4:0]*128+hcount[7:1];
		end 
		
		else begin
			en <= 1'b0;
		end
		
	end
	
	logic [23:0] paddle_output;
	logic  [11:0] paddle_address;
	logic paddle_en;
	
	paddle paddle_unit(
	.clock(clk50),
	.address(paddle_address),
	.q(paddle_output)
	);
	
logic columbia_en;
	always_ff @(posedge clk50) begin
	
		if(vcount[8:5]==4'd13 && (hcount[10:1]-px)<=127 && (hcount[10:1]-px>=0)) begin
			paddle_en<=1'b1;
			paddle_address<= vcount[4:0]*128+ (hcount[10:1]-px)%128; //hcount[7:1];
		end
		else if(on[9] && hcount[10:8] == 3'd4 && vcount[8:5] == 4'd2) begin
			columbia_en<=1'b1;
			paddle_address<= vcount[4:0]*128+ (hcount[10:1]-px)%128; //hcount[7:1];
		end 
		else if(on[18] && hcount[10:8] == 3'd3 && vcount[8:5] == 4'd4) begin
			columbia_en<=1'b1;
			paddle_address<= vcount[4:0]*128+ (hcount[10:1]-px)%128; //hcount[7:1];
		end 
		else if(on[12] && hcount[10:8] == 3'd2 && vcount[8:5] == 4'd3 ) begin
			columbia_en<=1'b1;
			paddle_address<= vcount[4:0]*128+ (hcount[10:1]-px)%128; //hcount[7:1];
		end
		else begin
			paddle_en<=1'b0;
			columbia_en <= 1'b0;
		end
	end
	
	
	logic [23:0] ball_output;
	logic  [9:0] ball_address;
	logic ball_en;
	logic [4:0] ball_pixel_y;
	//logic [4:0] ball_pixel_y;
	logic [5:0] ball_pixel_x;
	ball ball_unit(
		.clock(clk50),
		.address(ball_address),
		.q(ball_output)
	);
	


logic ball_reset;
assign ball_reset=(hcount[10:1]==10'd0)&&(vcount==10'd0);
	always_ff @(posedge clk50 or posedge ball_reset) begin
		if(ball_reset) begin
			ball_en<=1'b0;
			ball_pixel_y<=5'd0;
			ball_pixel_x<=6'd0;
		end
		else if ((vcount<=(y_coords+10'd16) && vcount>(y_coords-10'd16))  && (hcount[10:1]<=(x_coords+10'd16) && hcount[10:1]>(x_coords-10'd16) )) begin
			ball_en<=1'b1;
			ball_address<= ball_pixel_y*32+ball_pixel_x[5:1];
			
			if(ball_pixel_x==6'b111111) begin
				ball_pixel_y<=ball_pixel_y+1'b1;
				ball_pixel_x<=6'd0;
			end
			else 
				ball_pixel_x<=ball_pixel_x+1'b1;
		end
		else begin
			ball_en<=1'b0;
			
		end
	end
	
	assign rom_output=char_rom_data[~hcount[3:1]];//hcount LSB => char_rom_data MSB
	
	always_comb begin	  
		character_address=6'o40;// space by default
		if (vcount<8) begin
			case (hcount[10:4])
				7'd1: character_address=6'o14;
				7'd2: character_address=6'o11;
				7'd3: character_address=6'o26;
				7'd4: character_address=6'o05;
				7'd5: character_address=6'o23;
				
				7'd7: begin
						case(life)
						32'd0: character_address=6'o60; //ten
						32'd1: character_address=6'o61; //ten
						32'd2: character_address=6'o62; //ten
						32'd3: character_address=6'o63; //ten
						32'd4: character_address=6'o64; //ten
						32'd5: character_address=6'o65; //ten
						32'd6: character_address=6'o66; //ten
						32'd7: character_address=6'o67; //ten
						32'd8: character_address=6'o70; //ten
						32'd9: character_address=6'o71; //ten
						endcase
						
						end 
//				7'd77: character_address=6'd48+(score/100); //hundred
				7'd77: begin 	
						case(score/10)
						32'd0: character_address=6'o60; //ten
						32'd1: character_address=6'o61; //ten
						32'd2: character_address=6'o62; //ten
						32'd3: character_address=6'o63; //ten
						32'd4: character_address=6'o64; //ten
						32'd5: character_address=6'o65; //ten
						32'd6: character_address=6'o66; //ten
						32'd7: character_address=6'o67; //ten
						32'd8: character_address=6'o70; //ten
						32'd9: character_address=6'o71; //ten
						endcase
						end
				7'd78: begin 	
						case(score%10)
						32'd0: character_address=6'o60; //ten
						32'd1: character_address=6'o61; //ten
						32'd2: character_address=6'o62; //ten
						32'd3: character_address=6'o63; //ten
						32'd4: character_address=6'o64; //ten
						32'd5: character_address=6'o65; //ten
						32'd6: character_address=6'o66; //ten
						32'd7: character_address=6'o67; //ten
						32'd8: character_address=6'o70; //ten
						32'd9: character_address=6'o71; //ten
						endcase
						end
				7'd71: character_address=6'o23; //s
				7'd72: character_address=6'o03; //c
				7'd73: character_address=6'o17; //o
				7'd74: character_address=6'o22; //r
				7'd75: character_address=6'o05; //e
			
			endcase
				
				
		end
	
	end
	
/*	
   always_comb begin
      {VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00}; // Black
		if(en) 
			{VGA_R, VGA_G, VGA_B} = brick_output;
		if(paddle_en)
			{VGA_R, VGA_G, VGA_B} = paddle_output;
		if(ball_en)
			{VGA_R, VGA_G, VGA_B} = ball_output;
		//if (isBall)
			//{VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};
		if (rom_output)
			{VGA_R, VGA_G, VGA_B} = {8'hff, 8'hff, 8'h00};
  end  
*/   




always_comb begin

    if (hcount[10:1] > 19 && hcount[10:1] < 80 || hcount[10:1] > 139 && hcount[10:1] < 200 || hcount[10:1] > 259 && hcount[10:1] < 320 )
         mid_sprite = 0;
    else if (hcount[10:1] > 79 && hcount[10:1] < 140 || hcount[10:1] > 199 && hcount[10:1] < 260)
         mid_sprite = 1;
    else if(hcount[10:1] > 319 && hcount[10:1] < 380 || hcount[10:1] > 439 && hcount[10:1] < 500 || hcount[10:1] > 559 && hcount[10:1] < 620 )
         mid_sprite = 2;
    else 
         mid_sprite = 3;
end

assign mid_address = mid_sprite * 120 + line;

always_comb begin
    if (hcount[10:1] < 20)
       background = left_sprite_data[19 - hcount[10:1]];
    else if (hcount[10:1] > 619)
       background = right_sprite_data[639 - hcount[10:1]];
//    else if (hcount[10:1] > 19 && hcount[10:1] < 80)
//	    background = mid_sprite_data[79 - hcount[10:1]];
//	 else if (hcount[10:1] > 79 && hcount[10:1] < 140)
//	    background = mid_sprite_data[139 - hcount[10:1]];
//	 else if (hcount[10:1] > 139 && hcount[10:1] < 200)
//	    background = mid_sprite_data[199 - hcount[10:1]];
//	 else if (hcount[10:1] > 199 && hcount[10:1] < 260)
//	    background = mid_sprite_data[259 - hcount[10:1]];
//	 else if (hcount[10:1] > 259 && hcount[10:1] < 320)
//	    background = mid_sprite_data[319 - hcount[10:1]];
//	 else if (hcount[10:1] > 319 && hcount[10:1] < 380)
//	    background = mid_sprite_data[379 - hcount[10:1]];
//	 else if (hcount[10:1] > 379 && hcount[10:1] < 440)
//	    background = mid_sprite_data[439 - hcount[10:1]];
//	 else if (hcount[10:1] > 439 && hcount[10:1] < 500)
//	    background = mid_sprite_data[499 - hcount[10:1]];
//	 else if (hcount[10:1] > 499 && hcount[10:1] < 560)
//	    background = mid_sprite_data[559 - hcount[10:1]];
//	 else 
//	    background = mid_sprite_data[619 - hcount[10:1]];
    else
       background = mid_sprite_data[59 - (hcount[10:1] - 21)%60];
end


	always_comb begin
/*	
	if(peng)
	{VGA_R, VGA_G, VGA_B} = {8'h00, 8'h00, 8'h00}; // Black
	else
	{VGA_R, VGA_G, VGA_B} = {8'hff, 8'hff, 8'hff}; // whilte
*/	
    if(over==32'd0) begin
		if (background && (changeColor[26])) begin
	        {VGA_R, VGA_G, VGA_B} = {8'd60, 8'd210, 8'd144}; // Light Green
      end else if (background && (~changeColor[26]))
          {VGA_R, VGA_G, VGA_B} = {8'd40, 8'd145, 8'd100}; // Dark Green
	   else if ((~background) && (changeColor[26]))
	        {VGA_R, VGA_G, VGA_B} = {8'd40, 8'd145, 8'd100}; // Dark Green
      else //if ((~background) && (~changeColor[26]))
          {VGA_R, VGA_G, VGA_B} = {8'd60, 8'd210, 8'd144}; // Light Green
	   
		
      if(en) begin	
			{VGA_R, VGA_G, VGA_B} = brick_output;
		end
		if(columbia_en)
			{VGA_R,VGA_G,VGA_B} = paddle_output;
		if(paddle_en)
			{VGA_R, VGA_G, VGA_B} = paddle_output;
		if(ball_en&&ball_output)
			{VGA_R, VGA_G, VGA_B} = ball_output;
		//if (isBall)
			//{VGA_R, VGA_G, VGA_B} = {8'h00, 8'hff, 8'h00};
		if (rom_output)
			{VGA_R, VGA_G, VGA_B} = {8'hff, 8'hff, 8'h00};
		
		end else begin
		 {VGA_R, VGA_G, VGA_B} = {8'd0, 8'd0, 8'd0}; // Black
      if (gameover_output && inG)
	        {VGA_R, VGA_G, VGA_B} = {8'd0, 8'd191, 8'd255}; // DeepSkyBlue
      else if (gameover_output && inA)
          {VGA_R, VGA_G, VGA_B} = {8'd0, 8'd255, 8'd0}; // Lime
	    else if (gameover_output && inM)
	        {VGA_R, VGA_G, VGA_B} = {8'd255, 8'd165, 8'd0}; // Orange
      else if (gameover_output && inE1)
          {VGA_R, VGA_G, VGA_B} = {8'd240, 8'd128, 8'd128}; // LightCoral
      else if (gameover_output && inO)
          {VGA_R, VGA_G, VGA_B} = {8'd255, 8'd182, 8'd193}; // LightPink
      else if (gameover_output && inV)
          {VGA_R, VGA_G, VGA_B} = {8'd255, 8'd105, 8'd180}; // HotPink
      else if (gameover_output && inE2)
          {VGA_R, VGA_G, VGA_B} = {8'd147, 8'd112, 8'd219}; // MidiumPurple
      else if (gameover_output && inR)
          {VGA_R, VGA_G, VGA_B} = {8'd0, 8'd255, 8'd127}; // SpringGreen
			 end
	   
   end  
	
	
	////////////////////////////////////////////////////////////////
logic[2:0] letter_address;
logic inG;
logic inA;
logic inM;
logic inE1;
logic inO;
logic inV;
logic inE2;
logic inR;

logic[63:0] gameover_data;
logic gameover_output;

gameover_rom gameover_unit(
      .clock(clk50),
      .address({letter_address, vcount[5:0]}),
      .q(gameover_data)
);


always_comb begin
    inG = 0;
    inA = 0;
    inM = 0;
    inE1 = 0;
    inO = 0;
    inV = 0;
    inE2 = 0;
    inR = 0;
	 letter_address=0;
    if (hcount[10:1] > 191 && hcount[10:1] < 256 && vcount[9:0] > 127 && vcount[9:0] < 192) begin
         inG = 1;
         letter_address = 0;
    end
    else if (hcount[10:1] > 255 && hcount[10:1] < 320 && vcount[9:0] > 127 && vcount[9:0] < 192) begin
         inA = 1;
         letter_address = 1;
    end
    else if (hcount[10:1] > 319 && hcount[10:1] < 384 && vcount[9:0] > 127 && vcount[9:0] < 192) begin
         inM = 1;
         letter_address = 2;
    end
    else if (hcount[10:1] > 383 && hcount[10:1] < 448 && vcount[9:0] > 127 && vcount[9:0] < 192) begin
         inE1 = 1;
         letter_address = 3;
    end
    else if (hcount[10:1] > 191 && hcount[10:1] < 256 && vcount[9:0] > 191 && vcount[9:0] < 256) begin
         inO = 1;
         letter_address = 4;
    end
    else if (hcount[10:1] > 255 && hcount[10:1] < 320 && vcount[9:0] > 191 && vcount[9:0] < 256) begin
         inV = 1;
         letter_address = 5;
    end
    else if (hcount[10:1] > 319 && hcount[10:1] < 384 && vcount[9:0] > 191 && vcount[9:0] < 256) begin
         inE2 = 1;
         letter_address = 3;
    end
    else if (hcount[10:1] > 383 && hcount[10:1] < 448 && vcount[9:0] > 191 && vcount[9:0] < 256) begin
         inR = 1;
         letter_address = 6;
    end


end

assign gameover_output = gameover_data[~hcount[6:1]];
/*
always_comb begin
 if(over==32'd1) begin
     
	end		 

end
*/
////////////////////////////////////////////////////////////////
	
endmodule // VGA_LED_Emulator
