module DISTR(input logic clk, init, clear, ker, laplace, read,
					input logic [11:0] addr, 
					output logic fin, 
					output logic infm, 
					output logic [41:0] Bph, 
					output logic [6:0] vph);

logic [15:0]	wData, ovrData, rdData;
logic [5:0]              addr_m;
logic [5:0]   nxroll, roll;
logic        ovfl;

// mem control state machine
logic [3:0]   nstate, state;
logic         rd, wr, sqAddr, incr, z, p;

// laplace op state machine
logic [2:0]      lpstate, nlpstate;
logic [15:0]   buff [2:0];
logic [15:0]   min, nxmin;
logic [5:0]   bound, nxbound;
logic  push_is, push_seg, en_min;
logic in_is;

parameter IDLE = 4'b0000, CL = 4'b0001, RSH = 4'b0010, WSH = 4'b0011, CSH = 4'b0100, RP = 4'b0101, WP = 4'b0110, RLP = 4'b0111, FIN = 4'b1001, CSHF = 4'b1010, FIN2 = 4'b1011, FLP = 4'b1100, RH = 4'b1101, WH = 4'b1110;
parameter STANDBY = 3'b000, FT = 3'b001, FP = 3'b010, TR = 3'b011, CRS = 3'b100, PK = 3'b101;


assign ovrData = (z)?16'b0000000000000000:(rdData >> 1);
assign wData = (sqAddr)?ovrData:(rdData+1);
assign addr_m = (sqAddr)?roll:((p)?addr[11:6]:addr[5:0]);
assign ovfl = &rdData[15:1];
// laplace
assign in_is = ((buff[1]<<1) > (buff[2] + buff[0] + 16'b0000000000001000));

MEMD memkde(.clk(clk), .rd(rd), .wr(wr&(~clk)), .addr(addr_m), .wData(wData), .rdData(rdData));
FIS featureInfSmp(.clk(clk), .addr(addr), .push(push_is), .in(in_is), .infm(infm));
SEGS segments(.clk(clk), .push(push_seg), .in(bound), .boundary(Bph), .vph(vph));

always_ff @(posedge clk)
begin
  case (state)
    IDLE:
    begin
      rd = 1'b0;
      wr = 1'b0;
      incr = 1'b0;
      sqAddr = 1'b0;
      z = 1'b0;
      fin = 1'b0;
      push_is = 1'b0;
      p = 1'b0;
    end
    CL:
    begin
      rd = 1'b0;
      wr = 1'b1;
      incr = 1'b1;
      sqAddr = 1'b1;
      z = 1'b1;
      fin = 1'b0;
		push_is = 1'b0;
      p = 1'b0;
    end
    RSH:
    begin
      rd = 1'b1;
      wr = 1'b0;
      incr = 1'b0;
      sqAddr = 1'b1;
      z = 1'b0;
      fin = 1'b0;
      push_is = 1'b0;
      p = 1'b0;
    end
    WSH:
    begin
      rd = 1'b0;
      wr = 1'b1;
      incr = 1'b0;
      sqAddr = 1'b1;
      z = 1'b0;
      fin = 1'b0;
      push_is = 1'b0;
      p = 1'b0;
    end
    CSH:
	 begin
      rd = 1'b0;
      wr = 1'b0;
      incr = 1'b1;
      sqAddr = 1'b1;
      z = 1'b0;
      fin = 1'b0;
      push_is = 1'b0;
      p = 1'b0;
    end
    CSHF:
    begin
      rd = 1'b0;
      wr = 1'b0;
      incr = 1'b1;
      sqAddr = 1'b1;
      z = 1'b0;
      fin = 1'b0;
      push_is = 1'b0;
      p = 1'b0;
    end
    RP:
    begin
      rd = 1'b1;
      wr = 1'b0;
      incr = 1'b0;
      sqAddr = 1'b0;
      z = 1'b0;
      fin = 1'b0;
      push_is = 1'b0;
      p = 1'b1;
    end
    WP:
    begin
      rd = 1'b0;
      wr = 1'b1;
      incr = 1'b0;
      sqAddr = 1'b0;
      z = 1'b0;
      fin = 1'b0;
      push_is = 1'b0;
      p = 1'b1;
    end
    RH:
    begin
      rd = 1'b1;
      wr = 1'b0;
      incr = 1'b0;
      sqAddr = 1'b0;
      z = 1'b0;
      fin = 1'b0;
      push_is = 1'b0;
      p = 1'b0;
    end
    WH:
    begin
      rd = 1'b0;
      wr = 1'b1;
      incr = 1'b0;
      sqAddr = 1'b0;
      z = 1'b0;
      fin = 1'b0;
      push_is = 1'b0;
      p = 1'b0;
    end
    RLP:
    begin
      rd = 1'b1;
      wr = 1'b0;
      incr = 1'b1;
      sqAddr = 1'b1;
      z = 1'b0;
      fin = 1'b0;
      push_is = 1'b1;
      p = 1'b0;
    end
    FIN:
    begin
      rd = 1'b0;
      wr = 1'b0;
      incr = 1'b0;
      sqAddr = 1'b1;
      z = 1'b0;
      fin = 1'b1;
      push_is = 1'b0;
      p = 1'b0;
    end
    FIN2:
    begin
      rd = 1'b0;
      wr = 1'b0;
      incr = 1'b0;
      sqAddr = 1'b1;
      z = 1'b0;
      fin = 1'b1;
      push_is = 1'b0;
      p = 1'b0;
    end
    FLP:
    begin
      rd = 1'b0;
      wr = 1'b0;
      incr = 1'b0;
      sqAddr = 1'b1;
      z = 1'b0;
      fin = 1'b1;
      push_is = 1'b1;
      p = 1'b0;
    end
    default:
    begin
      rd = 1'b0;
      wr = 1'b0;
      incr = 1'b0;
      sqAddr = 1'b0;
      z = 1'b0;
      fin = 1'b0;
      push_is = 1'b0;
      p = 1'b0;
    end
  endcase
end
always @(posedge clk) begin
  if (init) begin
    roll <= 6'b000000;
    bound <= 6'b000000;
    min <= 16'b1111111111111111;
    state = IDLE;
    lpstate = STANDBY;
    buff[2] <= 16'b0000000000000000;
    buff[1] <= 16'b0000000000000000;
    buff[0] <= 16'b0000000000000000;
  end else begin
    roll <= nxroll;
    bound <= nxbound;
    min <= nxmin;
    state = nstate;
    lpstate = nlpstate;
    if (push_is) begin
      buff[2] <= rdData;
      buff[1] <= buff[2];
      buff[0] <= buff[1];
    end
  end
end

always_comb 
begin
    nxroll <= (fin)?(6'b000000):((incr)?(roll + 1):roll);
    nxbound <= (buff[2]<min)?(addr_m - 6'b000011):bound;
    nxmin <= (en_min)?((buff[2]<min)?buff[2]:min):(16'b1111111111111111);

   case (state)
      IDLE:
        if (ker)
          nstate = RP;
        else if (laplace)
          nstate = RLP;
        else
          nstate = IDLE;
      RSH:
        nstate = WSH;
      WSH:
        if (&roll)
          nstate = CSHF;
        else
          nstate = CSH;
      CSH:
        nstate = RSH;
      CSHF:
        nstate = FIN;
      RP:
        nstate = WP;
      WP:
        if (ovfl)
          nstate = RSH;
        else
          nstate = RH;
      RH:
        nstate = WH;
      WH:
        if (ovfl)
          nstate = RSH;
        else
          nstate = FIN;
      RLP:
        if (&roll)
          nstate = FLP;
        else
          nstate = RLP;
      FIN:
      if (read)
			nstate = IDLE;
		else
			nstate = FIN;
      FIN2:
		if (read)
        nstate = IDLE;
		else
		  nstate = FIN2;
      FLP:
		if (read)
        nstate = IDLE;
		else
			nstate = FLP;
      default:
        nstate = IDLE;
    endcase
	 
    case (lpstate)
      STANDBY:
        if (push_is)
          nlpstate = FT;
        else
          nlpstate = STANDBY;
      FT:
        if (in_is)
          nlpstate = FP;
        else
          nlpstate = FT;
      FP:
        if (in_is)
          nlpstate = FP;
        else
          nlpstate = TR;
      TR:
        if (in_is)
          nlpstate = CRS;
        else
          nlpstate = TR;
      CRS:
        if (&roll)
          nlpstate = STANDBY;
        else if (in_is)
          nlpstate = PK;
        else
         nlpstate = TR;
      PK:
        if (&roll)
          nlpstate = STANDBY;
        else if (in_is)
          nlpstate = PK;
        else
          nlpstate = TR;
      default:
        nlpstate = STANDBY;
    endcase
end
// laplace states
always_ff @(posedge clk)
begin
  case (lpstate)
    STANDBY:
    begin
      push_seg = 1'b0;
      en_min = 1'b0;
    end
    FT:
    begin
      push_seg = 1'b0;
      en_min = 1'b0;
    end
    FP:
    begin
      push_seg = 1'b0;
      en_min = 1'b0;
    end
    TR:
    begin
      push_seg = 1'b0;
      en_min = 1'b1;
    end
    CRS:
    begin
      push_seg = 1'b1;
      en_min = 1'b0;
    end
    PK:
    begin
      push_seg = 1'b0;
      en_min = 1'b0;
    end
    default:
    begin
      push_seg = 1'b0;
      en_min = 1'b0;
    end
  endcase
end

endmodule


// memery for distribution
module MEMD(input logic clk, rd, wr, 
				input logic [5:0] addr, 
				input logic [15:0] wData, 
				output logic [15:0] rdData);

logic [15:0]               kde [63:0];
//logic [9:0]                       rdData;

always_ff @(negedge clk) begin
        if (rd) begin
                rdData <= kde[addr];
        end
        if (wr) begin
                kde[addr] <= wData;
        end
end

endmodule

// shift registers for informative features
module FIS(input logic clk, push, in, 
				input logic [11:0] addr,
				output logic infm);
				
logic [63:0]              informative_sample;

assign infm = informative_sample[addr[11:6]]&informative_sample[addr[5:0]];

always_ff @(negedge clk) begin
        if (push) begin
                informative_sample[62] <= in;
                informative_sample[61:0] <= informative_sample[62:1];
        end
end

endmodule

// shift registers for boundary information
module SEGS(input logic clk, push, 
				input logic [5:0] in, 
				output logic [41:0] boundary, 
				output logic [6:0] vph);

logic [5:0]       regfile [6:0];

assign boundary = {regfile[6],regfile[5],regfile[4],regfile[3],regfile[2],regfile[1],regfile[0]};

always_ff @(negedge clk) begin
        if (push) begin
                regfile[6] <= in;
                regfile[5] <= regfile[6];
                regfile[4] <= regfile[5];
                regfile[3] <= regfile[4];
                regfile[2] <= regfile[3];
                regfile[1] <= regfile[2];
                regfile[0] <= regfile[1];
                vph <= {1'b1, vph[6:1]};
        end
end

endmodule
