open Ast
open Bytecode

module StringMap = Map.Make(String)

(* Symbol table: Information about all the names in scope *)
type env = {
    function_index : int StringMap.t; (* Index for each function *)
    global_index   : int StringMap.t; (* "Address" for global variables *)
    local_index    : int StringMap.t; (* FP offset for args, locals *)
  }

(* val enum : int -> 'a list -> (int * 'a) list *)
(* This creates "addresses"*)
(* enum 1 1 [8; 7; 6];;
- : (int * int) list = [(1, 8); (2, 7); (3, 6)]
*)
let rec enum stride n = function
    [] -> []
  | hd::tl -> (n, hd) :: enum stride (n+stride) tl

(* val string_map_pairs StringMap 'a -> (int * 'a) list -> StringMap 'a *)
(* This creates "Symbol table" *)
(* when (int * int) list = [(1, 8); (2, 7); (3, 6)] is passed,
then it says that 8 is present at location 1, 7 is present at location 2 and so on *)
let string_map_pairs map pairs =
  List.fold_left (fun m (i, n) -> StringMap.add n i m) map pairs

let translate (globals, functions) =

  (* Allocate "addresses" for each global variable *)
  let global_indexes = string_map_pairs StringMap.empty (enum 1 0 globals) in

  (* Assign indexes to function names; built-in "printtoscreen" is special *)
  let funcmap1 = StringMap.add "printtoscreen" (-1) StringMap.empty in
  let funcmap2 = StringMap.add "fetchvalue" (-2) funcmap1 in
  let funcmap3 = StringMap.add "compare" (-3) funcmap2 in
  let built_in_functions = StringMap.add "sendrequest" (-4) funcmap3 in
  let function_indexes = string_map_pairs built_in_functions
      (enum 1 1 (List.map (fun f -> f.fname) functions)) in

  (* Translate a function in AST form into a list of bytecode statements *)
  let translate env fdecl =
    (* Bookkeeping: FP offsets for locals and arguments *)
    let num_formals = List.length fdecl.formals
    and num_locals = List.length fdecl.locals
    and formal_offsets = enum 1 1 fdecl.formals
    (* locals begin immediately after formals. Consider formals as local variables in code*)
    and local_offsets = enum 1 ((List.length fdecl.formals)+1) fdecl.locals in
    let env = { env with local_index = string_map_pairs
		  StringMap.empty (formal_offsets @ local_offsets) } in
(* Each operation would save its value in a temporary array. As a result, we should be able to execute statements like "printtoscreen (Compare something)" *)
(* Save operation should also be able to use the temporary array. Therefore, we could execute statements like "var t1=Compare something" *)
(* SendReq operation is where temporary array is most useful. Since, sendrequest uses 4 arguments, we would store the local *)
    let rec expr = function
	Text s -> [STemp (s)]
       | Id s -> (* Check if s begins with double quotes. If it does, then add it to the temporary array otherwise, it is a variable*)
	if s.[0] = '"' then [STemp (s)] else
	  (try [LLocal (StringMap.find s env.local_index)]
          with Not_found -> try [LGlobal (StringMap.find s env.global_index)]
          with Not_found -> raise (Failure ("undeclared variable " ^ s)))
      |	FetchVal (e1,s) -> (match s with
		  "\"papertonscollected\"" -> expr e1 @ [Gnf 0] @ [Bin FetchValue]
		| "\"mgptonscollected\"" -> expr e1 @ [Gnf 1] @ [Bin FetchValue]
		| "\"communitydistrict\"" -> expr e1 @ [Gnf 2] @ [Bin FetchValue]
		| "\"borough\"" -> expr e1 @ [Gnf 3] @ [Bin FetchValue]
		| _ -> raise (Failure ("Unrecognized field name" ^ s))
		)
      | AddVal (e1,e2) -> expr e1 @ expr e2 @ [Bin Add]
      |	CompareVal (e1, e2) -> expr e1 @ expr e2 @ [Bin Compare]
      | PrintVal (op, e1) -> (match e1 with
		 Text s -> [STemp(s)] @ [PTemp]
		| Id s -> if s.[0] = '"' then [STemp(s)] @ [PTemp] else (try [PLocal (StringMap.find s env.local_index)]
          	with Not_found -> try [PGlobal (StringMap.find s env.global_index)]
          	with Not_found -> raise (Failure ("undeclared variable " ^ s)))
		| CompareVal (e1, e2) -> expr e1 @ expr e2 @ [Bin Compare] @ [PTemp]
		| AddVal (e1,e2) -> expr e1 @ expr e2 @ [Bin Add] @ [PTemp]
		| FetchVal (e1, s) -> (match s with
		  "\"papertonscollected\"" -> expr e1 @ [Gnf 0] @ [Bin FetchValue] @ [PTemp]
		| "\"mgptonscollected\"" -> expr e1 @ [Gnf 1] @ [Bin FetchValue] @ [PTemp]
		| "\"communitydistrict\"" -> expr e1 @ [Gnf 2] @ [Bin FetchValue] @ [PTemp]
		| "\"borough\"" -> expr e1 @ [Gnf 3] @ [Bin FetchValue] @ [PTemp]
		| _ -> raise (Failure ("Unrecognized field name"^s))
		)
		| _ -> raise (Failure ("Cannot call print on non-returning expression"))
	) 
      | Assign (s, e) -> expr e @
	  (try [SLocal (StringMap.find s env.local_index)]
  	  with Not_found -> try [SGlobal (StringMap.find s env.global_index)]
	  with Not_found -> raise (Failure ("undeclared variable " ^ s)))
      | SendWebRequest (op, e1) -> (List.concat (List.map expr (List.rev e1))) @ [SendReq]
      | Call (fname, actuals) -> (try
	  (List.concat (List.map expr (List.rev actuals))) @
	  [Jsr (StringMap.find fname env.function_index) ]   
        with Not_found -> raise (Failure ("undefined function " ^ fname)))
      | Noexpr -> []

(* This returns the bytecodes for a forin loop. Example :- if the list is (60 80), then the bytecode would be :-
	STemp 60
	LSkip 3 // 3 because we have an additional skip statement at the end of list in forin
	STemp 80
	LSkip 2 *)	
    in let rec loop_ignore_cmd = function
	[] -> []
      | hd :: [] -> expr hd @ [LSkip 1] 
      | hd :: tl -> expr hd @ [LSkip (1 + (2 * List.length tl))] @ (loop_ignore_cmd tl)

    in let rec stmt = function
	Block sl     ->  List.concat (List.map stmt sl)
      | Expr e       -> expr e
      | Return e     -> expr e (* The returned value would automatically be saved in temporary array *)
      | If (p, t, f) -> let t' = stmt t and f' = stmt f in
	expr p @ [Skip(1 + List.length t')] @
	t' @ [Skip(List.length f')] @ f'
      | Repeat(sl, e) -> let t' = stmt sl in
	[Loop] @ t' @ expr e @ [Skip(1)] @ [GoBackLoop] @ [EndLoop]
      | Forin(s,el,sl) -> let t'=stmt sl in
	(* Adding 2 to the skip so that it can skip over EndForLoop and SLocal/SGlobal also *)
	(loop_ignore_cmd (List.rev el)) @ [Skip (2 + (List.length t'))] @
	  (try [SLocal (StringMap.find s env.local_index)]
  	  with Not_found -> try [SGlobal (StringMap.find s env.global_index)]
	  with Not_found -> raise (Failure ("undeclared variable " ^ s)))
	@ t' @ [EndForLoop]
    
    in let bytecodearray = stmt (Block fdecl.body) 
    in [Ent (num_locals+num_formals)] @      (* Entry: allocate space for locals *)
     bytecodearray @         (* Body *)
    [Rts (num_formals+num_locals)]   (* Default = return 0 *)

  in let env = { function_index = function_indexes;
		 global_index = global_indexes;
		 local_index = StringMap.empty } in

  (* Code executed to start the program: Jsr main; halt *)
  let entry_function = try
    [Jsr (StringMap.find "main" function_indexes); Hlt]
  with Not_found -> raise (Failure ("no \"main\" function"))
  in
    
  (* Compile the functions *)
  let func_bodies = entry_function :: List.map (translate env) functions in

  (* Calculate function entry points by adding their lengths *)
  let (fun_offset_list, _) = List.fold_left
      (fun (l,i) f -> (i :: l, (i + List.length f))) ([],0) func_bodies in
  let func_offset = Array.of_list (List.rev fun_offset_list) in

  { num_globals = List.length globals;
    (* Concatenate the compiled functions and replace the function
       indexes in Jsr statements with PC values *)
    text = Array.of_list (List.map (function
	Jsr i when i > 0 -> Jsr func_offset.(i)
      | _ as s -> s) (List.concat func_bodies))
  }
