type bstmt =
    	  LLocal of int (* Load from local *)
	| LGlobal of int (* Load from global *)
	| Bin of Ast.op (* Perform binary operation *)
	| Jsr of int (* Jump to location *)
	| Gnf of int (* Gets next field at offset *)
	| PLocal of int      (* Prints the value present on local array location *)
	| PGlobal of int (* Prints the value present on global array location *)
	| PTemp     (* Prints the last value present in temporary *)
	| SendReq  (* Sends the request using the values present in call array *)
	| SLocal of int (* Saves the value at specified location in local array *)
	| SGlobal of int (* Saves the value at specified location in global array *)
	| STemp of string (* Saves the string to the temporary array *)
	| Hlt (* Halt the program *)
	| Ent of int    (* Tells the number of local variables present *)
  	| Rts of int    (* Tells the total number of variables (local+formals) present. This helps in resetting the local array to its original state *)
	| Skip of int   (* Skips over 'i' instructions present in instructions array*)
	| Loop		(* A loop would start. This would push the location of the next statement onto the stack and treat a loop as an anonymous inner function *)
	| GoBackLoop	(* Go back to the last loop. Get the location from the stack and start execution again. Reset temporary array *)
	| EndLoop	(* This corrects the entries in the stack so the anonymous function (loop) is no longer accessible*) 
	| LSkip of int	(* This skips the i instructions while iterating the for loop. These skipped instructions are executed in the next iteration *)
	| EndForLoop	(* This corrects the entries in the stack so that the for loop would restart at the correct location *)

type prog = {
    num_globals : int;   (* Number of global variables *)
    text : bstmt array; (* Code for all the functions *)
  }

let string_of_stmt = function
	LLocal(i) -> "LLocal " ^ string_of_int i
	| LGlobal(i) -> "LGlobal " ^ string_of_int i
	| PTemp -> "PTemp"
	| Jsr(i) -> "Jsr " ^ string_of_int i
	| Bin(Ast.Compare) -> "Compare"
	| Bin(Ast.FetchValue) -> "Fetch"
	| Bin(Ast.Add) -> "Add"
	| Gnf(i) -> "Gnf " ^ string_of_int i
	| PLocal(i) -> "PLocal " ^ string_of_int i
	| PGlobal(i) -> "PGlobal " ^ string_of_int i
	| SendReq -> "SendReq"
	| SLocal(i) -> "SLocal " ^ string_of_int i
	| SGlobal(i) -> "SGlobal " ^ string_of_int i
	| STemp(s) -> "STemp " ^ s
	| Hlt -> "Hlt"
	| Ent(i) -> "Ent " ^ string_of_int i
	| Rts(i) -> "Rts " ^ string_of_int i
	| Skip(i) -> "Skip " ^ string_of_int i
	| Loop -> "Loop"
	| GoBackLoop -> "GoBackLoop"
	| EndLoop -> "EndLoop"
	| LSkip(i) -> "LSkip " ^ string_of_int i
	| EndForLoop -> "EndForLoop"

let string_of_prog p =
  string_of_int p.num_globals ^ " global variables\n" ^
  let funca = Array.mapi
      (fun i s -> string_of_int i ^ " " ^ string_of_stmt s) p.text
  in String.concat "\n" (Array.to_list funca)
