open Ast

exception FunctionNotFoundException of string
exception ClassConstructorException of string
exception SemanticFailure of string


(* ******************* Syntax File Parsing variables ******************* *)
exception Syntax_error of string
let current_parse_line = ref 0
let current_include_parse_file = ref ""
let main_parse_file = ref ""
let get_parse_file_name = (fun x -> if !current_include_parse_file = "" then x else !current_include_parse_file)

(* ******************* Literal String Functions ******************* *)

(* return the number of tab chars specified *)
let rec get_tabs count =
	if count = 0 then
		""
	else
		"\t" ^ (get_tabs (count - 1)) 

let string_of_literal lit =
	let rec rec_string_of_literal embed_count = function
		IntLiteral(lit)    -> string_of_int lit
		| BoolLiteral(lit) -> string_of_bool lit
		| StringLiteral(lit) -> lit
		| DoubleLiteral(lit) -> string_of_float lit
		| ClassLiteral(cls_map) -> 
			NameMap.fold (fun key value accum -> accum ^ " \n" ^ (get_tabs (embed_count)) 
				^ key ^ " " ^ (rec_string_of_literal (embed_count + 1) value)) cls_map ""
	  	| LhsLiteral(var, lit) -> (rec_string_of_literal embed_count lit)
		| ListLiteral(items) ->  "[" ^ (String.concat ", " (List.map (rec_string_of_literal embed_count) items)) ^ "]"	
		| KVPLiteral(key, lit) -> key ^ ": " ^ (rec_string_of_literal embed_count lit)
	in
	(rec_string_of_literal 0 lit)

let rec is_list_literal = function
	ListLiteral(items) ->  true	
 	| LhsLiteral(var, lit) -> (is_list_literal lit)
	| _ -> false

let rec get_list_items = function
	ListLiteral(items) ->  items	
 	| LhsLiteral(var, lit) -> (get_list_items lit)
	| _ -> raise(Failure("Type is not a list"))


let string_of_var = function
    Int    -> "Int"
  | Bool -> "Bool"
  | List(var) -> "List"
  | String -> "String"
  | Double -> "Double"
  | ClassType(classname) -> classname

let string_of_var_decl vdecl =
    string_of_var vdecl.vtype ^ " " ^ vdecl.vname ^ ";\n"

let rec string_of_expr = function
    Literal(lit) -> string_of_literal lit
  | Id(s) -> s
  | Binop(e1, o, e2) ->
      string_of_expr e1 ^ " " ^
      (match o with
	Add -> "+" | Sub -> "-" | Mult -> "*" | Div -> "/"
      | Equal -> "==" | Neq -> "!="
      | Less -> "<" | Leq -> "<=" | Greater -> ">" | Geq -> ">=") ^ " " ^
      string_of_expr e2
  | Assign(v, e) -> string_of_expr v ^ " = " ^ string_of_expr e
  | Call(f, el) ->
      f ^ "(" ^ String.concat ", " (List.map string_of_expr el) ^ ")"
  | ClassCall(class_instance, f, el) ->
      string_of_expr class_instance ^ " . " ^ f ^ "(" ^ String.concat ", " (List.map string_of_expr el) ^ ")"
  | Access(e, member) -> string_of_expr e ^ " . " ^ member
  | Noexpr -> ""
  | Negate(e) -> "Negation of " ^ string_of_expr e
  | Cast(vari, e) -> "Cast of " ^ (string_of_expr e) ^ " to " ^ (string_of_var vari)
  | ListItems(items) -> "List Items: " ^ String.concat "\n" 
						(List.map (fun item -> item.lkey ^ ": " ^ string_of_expr item.lvalue) items)


let rec string_of_stmt = function
    Block(stmts) ->
      "{\n" ^ String.concat "" (List.map string_of_stmt stmts) ^ "}\n"
  | Expr(expr) -> string_of_expr expr ^ ";\n";
  | Return(expr) -> "return " ^ string_of_expr expr ^ ";\n";
  | If(e, s, Block([])) -> "if (" ^ string_of_expr e ^ ")\n" ^ string_of_stmt s
  | If(e, s1, s2) ->  "if (" ^ string_of_expr e ^ ")\n" ^
      string_of_stmt s1 ^ "else\n" ^ string_of_stmt s2
  | For(e1, e2, e3, s) ->
      "for (" ^ string_of_expr e1  ^ " ; " ^ string_of_expr e2 ^ " ; " ^
      string_of_expr e3  ^ ") " ^ string_of_stmt s
  | While(e, s) -> "while (" ^ string_of_expr e ^ ") " ^ string_of_stmt s

let string_of_fdecl fdecl =
  fdecl.fname ^ "(" ^ 
  String.concat "" (List.map string_of_var_decl fdecl.locals) ^ ")\n{\n" ^
  String.concat "" (List.map string_of_var_decl fdecl.locals) ^
  String.concat "" (List.map string_of_stmt fdecl.body) ^
  "}\n"

let string_of_odecl odecl =
  odecl.class_name ^ "\n{\n" ^
  String.concat "" (List.map string_of_var_decl odecl.data_members) ^
  String.concat "" (List.map string_of_fdecl odecl.function_members) ^
  "}\n"

let string_of_program (vars, funcs, classes) =
  String.concat "" (List.map string_of_var_decl vars) ^ "\n" ^
  String.concat "\n" (List.map string_of_fdecl funcs) ^ "\n" ^
  String.concat "\n" (List.map string_of_odecl classes)


(* ******************* Literal Utility Functions ******************* *)

(* convert literal into a boolean *)
let rec bool_of_literal = function
	IntLiteral(lit) -> (lit != 0)
	| BoolLiteral(lit) -> lit
	| ListLiteral(lit) -> (List.length lit) != 0
	| StringLiteral(lit) -> (lit <> "")
	| DoubleLiteral(lit) -> ((abs_float lit) > epsilon_float)
	| LhsLiteral(var, lit) -> bool_of_literal lit
    | _ -> raise (Failure("Class boolean operation not supported"))

(* perform the passed in binary operation on the literal type *)
let rec do_literal_operation int_op double_op string_op (v1, v2) = 
  match v1,v2 with
	(IntLiteral(lit1), IntLiteral(lit2)) ->  IntLiteral(int_op lit1 lit2)
	| (DoubleLiteral(lit1), DoubleLiteral(lit2)) ->  DoubleLiteral(double_op lit1 lit2)
	| (StringLiteral(lit1), StringLiteral(lit2)) ->  StringLiteral(string_op lit1 lit2)
	| (LhsLiteral(var, lit), (_ as second)) -> do_literal_operation (int_op) (double_op) (string_op) (lit, second)
	| ((_ as first), LhsLiteral(var, lit)) -> do_literal_operation (int_op) (double_op)  (string_op) (first, lit)
	| (_ as first), (_ as second) -> raise (Failure("Binary operations not supported for: " ^ 
												(string_of_literal first) ^ " and " ^ (string_of_literal second)))

(* perform the passed in comparison operation on the literal type *)
let rec do_literal_compare int_op double_op string_op bool_op (v1, v2) = 
	match v1,v2 with
		(IntLiteral(lit1), IntLiteral(lit2)) ->  BoolLiteral(int_op lit1 lit2)
		| (DoubleLiteral(lit1), DoubleLiteral(lit2)) ->  BoolLiteral(double_op lit1 lit2)
		| (StringLiteral(lit1), StringLiteral(lit2)) ->  BoolLiteral(string_op lit1 lit2)
		| (BoolLiteral(lit), (_ as second)) -> BoolLiteral(bool_op lit (bool_of_literal second))
		| ((_ as first), BoolLiteral(lit)) -> BoolLiteral(bool_op lit (bool_of_literal first))
		| (LhsLiteral(var, lit), (_ as second)) -> do_literal_compare (int_op) (double_op) (string_op) (bool_op) (lit, second)
		| ((_ as first), LhsLiteral(var, lit)) -> do_literal_compare (int_op) (double_op) (string_op) (bool_op) (first, lit)
		| _ -> raise (Failure("Comparison not supported for this type"))

let rec negate_of_literal = function
	IntLiteral(lit)    -> IntLiteral(-lit)
	| DoubleLiteral(lit) -> DoubleLiteral(-.lit)
	| LhsLiteral(var, lit) -> negate_of_literal lit
	| _ -> raise (Failure("Negation not defined for this type"))

let rec cast_of_literal (v_type, v1) =
	match v_type, v1 with
		String, IntLiteral(lit) -> StringLiteral(string_of_int lit)
		| String, DoubleLiteral(lit) -> StringLiteral(string_of_float lit)
		| _, LhsLiteral(var, lit) -> cast_of_literal (v_type, lit)
		| _, _ -> raise (Failure("Casting not defined for this type"))

let rec double_of_literal = function
	DoubleLiteral(lit) -> lit
	| LhsLiteral(var, lit) -> double_of_literal lit
	| _ -> raise (Failure("Not a double type"))



(* ******************* List FUNCTIONS ******************* *)

(* a list item can be a value or a key value pair *)
let get_list_literal item eval env =
	let v1, env = eval env item.lvalue in
	if item.lkey = "_" then
		v1, env
	else
		KVPLiteral(item.lkey, v1), env

let get_list = function
	ListLiteral(items) -> items
	| _ -> raise (Failure("Class constructors must be initialized with key value pairs"))


(* ******************* Utility FUNCTIONS ******************* *)

let get_lhs_varname = function
  LhsLiteral(var, lit) -> var
    | _ as lit -> raise (Failure("Variable is NOT a LHS:" ^ string_of_literal lit))

let is_lhs_literal = function
  LhsLiteral(var, lit) -> true
    | _ -> false



(* return a list of strings that were separated by the delimiter *)
let rec rec_split_string delim str_test id_list = 
  try 
    let delim_index = String.index str_test delim in
    let part1 = (String.sub str_test 0 delim_index) in
    let part2 = (String.sub str_test (delim_index + 1) ((String.length str_test) - (delim_index + 1))) in
	(rec_split_string delim part2 (part1::id_list))
  with Not_found -> if (String.length str_test) != 0 then (List.rev (str_test::id_list)) else (List.rev id_list)

let rec insert_item (item, index) = function
	[] -> 
		if index = 0 then 
			[item]
		else
			raise(Failure("Insert index is invalid"))
	| hd::tl as lst -> 	
		if index = 0 then 
			item::lst
		else
			hd::(insert_item (item, (index - 1)) tl)

let rec remove_item index = function
	[] -> 
		raise(Failure("Remove index is invalid"))
	| hd::tl -> 	
		if index = 0 then 
			tl
		else
			hd::(remove_item (index - 1) tl)

let rec pop_item ret_lst = function
	[] -> 
		raise(Failure("Cannot pop an empty list"))
	| hd::[] -> 	
		([], hd)
	| hd::tl -> 	
		if List.length tl = 1 then 
			(List.rev (hd::ret_lst), List.hd tl)
		else
			pop_item (hd::ret_lst) tl




