import os
import sys

class ParseError(RuntimeError):
	def __init__(self, error_msg, line_num):
		self.error_msg = error_msg
		self.line_num = line_num

	def GetString(self):
		err_msg = "Preparsing error on line: " + str(self.line_num + 1)
		err_msg += "\n"
		return err_msg + self.error_msg


def ValidateLine(line, line_num):
	invalid_chars = "{}"
	
	if any(elem in line for elem in invalid_chars):
		raise ParseError("Invalid character found on line: " + line, line_num)

def RemoveComment(in_comment, line, line_num):

	if in_comment:
		begin_of_comment = -1
	else:
		begin_of_comment = line.find('/*')

		# started line uncommented and ended uncommented (SU, EU)
		if begin_of_comment < 0:
			return False, line

		in_comment = True


	end_of_comment = line.find('*/')

	if end_of_comment < 0:
		# started line commented, and ended commented (SC, EC)
		if begin_of_comment < 0:
			return True, ""

		# started line uncommented, but ended commented (SU, EC)
		parsed_line = line[:begin_of_comment]
		return True, parsed_line

	# started line commented, but ended uncommented (SC, EU)
	if begin_of_comment == -1:
		parsed_line = line[(end_of_comment + 2):]
		return RemoveComment(False, parsed_line, line_num)

	# begin and end comment in this line
	if end_of_comment < begin_of_comment:
		raise ParseError("Begin comment found before end of comment on line: " + line, line_num)
	
	parsed_line = line[:begin_of_comment]
	parsed_line += line[(end_of_comment + 2):]
	return RemoveComment(False, parsed_line, line_num)

class ParseEnv(RuntimeError):
	def __init__(self, directory_path):
		self.directory_path = directory_path		
		self.in_comment = False
		self.prev_line_indent = False
		self.indent_stack = []
		self.parsed_files = []

def handle_include_line(source_file, parsed_line, line_num, parsed_file, env):
	include_string = "include \""
	if parsed_line.lstrip()[:(len(include_string))] != include_string:
		return False

	include_file_name = parsed_line.lstrip()[(len(include_string)):]
	
	include_dir = os.path.dirname(source_file.name)
	include_dir = os.path.normpath(include_dir) + os.sep
	include_file_name = include_dir + include_file_name.strip('\"')

	if include_file_name in env.parsed_files:
		raise ParseError("Recursive includes are not allowed: " + include_file_name, line_num)
		
	try:
		include_file = open(include_file_name, "r")
	except IOError:
		raise ParseError("Cannot find include file: " + include_file_name, line_num)
	
	parsed_file.write('__BEGIN_OF_INCLUDE__ ' + include_file_name + "!@<" + '\n')
	parse_source(include_file, parsed_file, env)
	parsed_file.write('__END_OF_INCLUDE__ '  + '\n')
	return True

def parse_source(input_file, parsed_file, env):
	env.parsed_files.append(input_file.name) 

	for i, line in enumerate(input_file):
		ValidateLine(line, i)
		env.in_comment, parsed_line = RemoveComment(env.in_comment, line, i)

		parsed_line = parsed_line.rstrip()
		if len(parsed_line) == 0:
			parsed_file.write('\n')
			continue

		if handle_include_line(input_file, parsed_line, i, parsed_file, env):
			continue
		
		tab_count  = len(parsed_line) - len(parsed_line.lstrip('\t'))
		indent_marker_on_line = parsed_line[-1] == ':'

		if env.prev_line_indent:
			if (not env.indent_stack) or (tab_count > env.indent_stack[-1]):
				env.indent_stack.append(tab_count)
				env.prev_line_indent = False
			elif indent_marker_on_line:
				raise ParseError("Empty functions are not allowed.", i)
			else:
				raise ParseError("You must increase indentation on line: " + line, i)

		if env.indent_stack and (tab_count > env.indent_stack[-1]):
			raise ParseError("Cannot increase indentation on line: " + line, i)
		else:
			while(env.indent_stack and (tab_count < env.indent_stack[-1])):
				parsed_line = "}" + parsed_line
				env.indent_stack.pop()

			if env.indent_stack and (tab_count != env.indent_stack[-1]):
				raise ParseError("Indentation must be the same as the previous group: " + line, i)

		if indent_marker_on_line:
			parsed_line = parsed_line + '{\n'
			env.prev_line_indent = True
		else:
			parsed_line = parsed_line + ';\n'

		# print parsed_line
		parsed_file.write(parsed_line)

	# can not reach end the file still in a comment
	if env.in_comment:
		raise ParseError("Reached end of file before the end of comment was found.", i)
	# can not reach end the file with a single line in a function/control statement
	elif env.prev_line_indent:
		raise ParseError("Empty statement not allowed.", i)

	while env.indent_stack:
		parsed_file.write("}")
		env.indent_stack.pop()

if __name__ == "__main__":

	if len(sys.argv) != 2:
		print "The first argument must be the source file name."
		sys.exit(2)
	
	source_name = sys.argv[1]

	try:
		source_file = open(source_name, "r")

		directory_path = os.path.dirname(source_file.name)
		directory_path = os.path.normpath(directory_path) + os.sep

		parsed_file = open(directory_path + os.path.basename(source_file.name) + ".tmp", 'w')
		error_file_name = directory_path + os.path.basename(source_file.name) + ".err"

		if os.path.isfile(error_file_name):
			os.remove(error_file_name)

		parsed_file.write('__START_MAIN_FILE__ ' + source_name + "!@<" + '\n')
		parse_source(source_file, parsed_file, ParseEnv(directory_path))
	except IOError:
		print "Unable to read source file: %s\n" % source_name
		sys.exit(1)
	except ParseError as exception:
		error_file = open(error_file_name, 'w')
		error_file.write(exception.GetString())
		sys.exit(1)


