%{ 
	open Ast 
	open Utility

	(* Used to output error for unknown syntax *)
	let parse_error s = (* Called by the parser function on error *)
	  print_endline ("***** " ^ s ^ " in file: " ^ (get_parse_file_name !main_parse_file) ^ ", on line: " ^ (string_of_int (!current_parse_line)));
	  flush stdout
%}

%token SEMI LPAREN RPAREN LBRACE RBRACE COMMA
%token PLUS MINUS TIMES DIVIDE ASSIGN
%token EQ NEQ LT LEQ GT GEQ
%token RETURN IF ELSE FOR WHILE INT
%token COLON LSQBRACKET RSQBRACKET CLASS INT STRING DOUBLE BOOL LIST TRUE FALSE ACCESS AT
%token <bool> BOOLLITERAL
%token <int> INTLITERAL
%token <string> ID STRINGLITERAL CLASSTYPE
%token <float> DOUBLELITERAL
%token EOF

%nonassoc NOELSE
%nonassoc ELSE
%right ASSIGN
%left EQ NEQ
%left LT GT LEQ GEQ
%left PLUS MINUS
%left TIMES DIVIDE
%left ACCESS
%nonassoc UMINUS

%start program
%type <Ast.program> program

%%

/* Program consists of variable, function, and class declarations */
program:
   /* nothing */ { [], [], [] }
 | program vdecl { let (variables, functions, classes) = $1 in
   ($2 @ variables), functions, classes }
 | program fdecl { let (variables, functions, classes) = $1 in
    variables, ($2 :: functions), classes }
 | program cdecl { let (variables, functions, classes) = $1 in
    variables, functions, ($2 :: classes) }

fdecl_list:
    /* nothing */     { [] }
  | fdecl_list fdecl { $2 :: $1 }

fdecl:
	ID LPAREN formals_opt RPAREN COLON LBRACE vdecl_list stmt_list RBRACE
		{ { fname = $1; formals = $3; locals = List.rev $7; body = List.rev $8 } }


formals_opt:
    /* nothing */ { [] }
  | formal_list   { List.rev $1 }

formal_list:
    var ID                   { [{ vtype = $1;
	                          vname = $2 }] }
  | formal_list COMMA var ID { { vtype = $3;
	                         vname = $4 } :: $1 }

vdecl:
    var ID { [{ vtype = $1; vname = $2 }] }
 |  vdecl COMMA ID { { vtype = ((List.hd $1).vtype); vname = $3 } :: $1 }
 |  vdecl SEMI { $1 }

vdecl_list:
     { [] }
  | vdecl_list vdecl { $2 @ $1 }


var:
    INT      { Int }
  | STRING   { String }
  | BOOL     { Bool }
  | LIST LSQBRACKET var RSQBRACKET { List($3) }
  | DOUBLE   { Double }
  | CLASS ID { ClassType($2) } 

cdecl:
   CLASS ID COLON LBRACE vdecl_list fdecl_list RBRACE
     { { class_name = $2;
	 data_members = List.rev $5;
	 function_members = List.rev $6 } }

stmt_list:
      { [] } 
  | stmt_list stmt { $2 :: $1 }

stmt:
    expr SEMI { Expr($1) }
  | RETURN expr SEMI { Return($2) }
  | LBRACE stmt_list RBRACE { Block(List.rev $2) }
  | IF expr COLON stmt %prec NOELSE { If($2, $4, Block([])) }
  | IF expr COLON stmt ELSE COLON stmt    { If($2, $4, $7) }
  | FOR LPAREN expr_opt SEMI expr_opt SEMI expr_opt RPAREN COLON stmt
     { For($3, $5, $7, $10) }
  | WHILE LPAREN expr RPAREN COLON stmt { While($3, $6) }

expr_opt:
    /* nothing */ { Noexpr }
  | expr          { $1 }

expr:
    INTLITERAL       { Literal(IntLiteral($1)) }
  | BOOLLITERAL      { Literal(BoolLiteral($1)) }
  | TRUE             { Literal(BoolLiteral(true)) }
  | FALSE            { Literal(BoolLiteral(false)) }
  | STRINGLITERAL    { Literal(StringLiteral($1)) }
  | DOUBLELITERAL    { Literal(DoubleLiteral($1)) }
  | ID               { Id($1) } 
  | expr PLUS   expr { Binop($1, Add,   $3) }
  | expr MINUS  expr { Binop($1, Sub,   $3) }
  | expr TIMES  expr { Binop($1, Mult,  $3) }
  | expr DIVIDE expr { Binop($1, Div,   $3) }
  | expr EQ     expr { Binop($1, Equal, $3) }
  | expr NEQ    expr { Binop($1, Neq,   $3) }
  | expr LT     expr { Binop($1, Less,  $3) }
  | expr LEQ    expr { Binop($1, Leq,   $3) }
  | expr GT     expr { Binop($1, Greater,  $3) }
  | expr GEQ    expr { Binop($1, Geq,   $3) }
  | expr ASSIGN expr   { Assign($1, $3) }
  | ID LPAREN actuals_opt RPAREN { Call($1, $3) }
  | expr ACCESS ID LPAREN actuals_opt RPAREN { ClassCall($1, $3, $5) }
  | expr ACCESS ID   { Access($1, $3) }
  | LPAREN expr RPAREN { $2 }  
  | MINUS expr %prec UMINUS { Negate($2) }
  | LT var GT LPAREN expr RPAREN { Cast($2, $5) }
  | LSQBRACKET list_opt RSQBRACKET { ListItems($2) }

actuals_opt:
    /* nothing */ { [] }
  | actuals_list  { List.rev $1 }

actuals_list:
    expr                    { [$1] }
  | actuals_list COMMA expr { $3 :: $1 }

list_opt:
    /* nothing */ { [] }
  | list_items  { $1 }
  | list_nvp_items  { $1 }

list_items:
  |  expr { [ { lkey = "_"; lvalue = $1 } ] }
  | list_items COMMA expr { { lkey = "_"; lvalue = $3 } :: $1 }

list_nvp_items:
    ID COLON expr { [{ lkey = $1; lvalue = $3 }] }
  | list_nvp_items COMMA ID COLON expr { { lkey = $3; lvalue = $5 } :: $1 }

