open Ast
open Utility
open Class
open Environment
open Lists


let call_fdecl (call, fdecl, actuals, env) =
	try
		let env = { (call fdecl actuals env) with env_locals = NameMap.empty }
		in IntLiteral(0), env
	with ReturnException(v, env) -> v, { env with env_locals = NameMap.empty }

let eval_actual_list (actuals, env, eval) = 
	List.fold_left
		(fun (actuals, env) actual -> let v, env = eval env actual in v :: actuals, env)
		([], env) (List.rev actuals)


(* Main entry point: run a program *)


(* ******************* Runner/Exec/Eval FUNCTIONS ******************* *)

let run (vars, funcs, classes) =
	(* Put function declarations in a symbol table *)
	let func_decls = map_funcs (funcs, classes)
	in

	(* Invoke a function and return an updated global symbol table *)
	let rec call fdecl actuals call_env =
		(* Evaluate an expression and return (value, updated environment) *)
		let rec eval env = function
			Literal(i) -> i, env
			| Noexpr -> IntLiteral(1), env (* must be non-zero for the for loop predicate *)
			| Id(var) -> id_variable_lookup (var, env, "ID: undeclared identifier ")
			| Cast(v_type, e1) ->
				let v1, env = eval env e1 in
				(cast_of_literal (v_type, v1)), env 
			| Negate(e1) ->
				let v1, env = eval env e1 in
				(negate_of_literal v1), env
			| Binop(e1, op, e2) ->
				let v1, env = eval env e1 in
				let v2, env = eval env e2 in
				let bool_compare_exception = (fun x y -> raise (Failure("Boolean greater/less than operators not supported"))) in
				(match op with
					Add -> 
						if ((is_list_literal v1) && (is_list_literal v2)) then
							list_concat (v1, v2)
						else
							do_literal_operation (+) (+.) (^) (v1,v2)
					| Sub -> do_literal_operation (-) (-.) (fun x y -> raise (Failure("String subtract not supported"))) (v1,v2)
					| Mult -> do_literal_operation ( * ) ( *. ) (fun x y -> raise (Failure("String multiply not supported"))) (v1,v2)
					| Div -> do_literal_operation ( / ) ( /. ) (fun x y -> raise (Failure("String divide not supported"))) (v1,v2)
					| Equal -> do_literal_compare (==) (=) (=) (==) (v1,v2)
					| Neq -> do_literal_compare (!=) (<>) (<>) (!=) (v1,v2)
					| Less -> do_literal_compare (<) (<) (<) (bool_compare_exception) (v1,v2)
					| Leq -> do_literal_compare (<=) (<=) (<=) (bool_compare_exception) (v1,v2)
					| Greater -> do_literal_compare (>) (>) (>) (bool_compare_exception) (v1,v2)
					| Geq -> do_literal_compare (>=) (>=) (>=) (bool_compare_exception) (v1,v2)), env
			| ListItems(items) -> 
				let list_with_env = List.fold_left (fun env_accum item -> 
					let lit_and_env = get_list_literal item eval (fst env_accum) in
					((snd lit_and_env), (fst lit_and_env)::(snd env_accum))) (env, []) items in
				ListLiteral((snd list_with_env)), (fst list_with_env)
			| Assign(e_var, e_val) ->
				let v_var, env = eval env e_var in
				let v_val, env = eval env e_val in
				let var_name = (get_lhs_varname v_var) in
				assign_variable_lookup (var_name, v_val, env, "Assign: undeclared identifier ")
			| Call("print", [e]) ->
				let v, env = eval env e in
				print_endline (string_of_literal v);
				IntLiteral(0), env
			| Call("cos", [e]) ->
				let v, env = eval env e in
				DoubleLiteral(cos (double_of_literal v)), env
			| Call("sin", [e]) ->
				let v, env = eval env e in
				DoubleLiteral(sin (double_of_literal v)), env
			| Call("sqrt", [e]) ->
				let v, env = eval env e in
				DoubleLiteral(sqrt (double_of_literal v)), env
			| ClassCall (e, cls_func, actuals) ->
				let cls_lit, env = eval env e in
				let actuals, env = eval_actual_list (actuals, env, eval) in
				if (is_list_literal cls_lit) = true then
					eval_list_func (cls_lit, cls_func, actuals, env)
				else
					let class_name = (get_class_name cls_lit) in  
					let func_name = class_name ^ "." ^ cls_func in
					cls_call_func (call, env, func_decls, func_name, actuals, cls_lit, call_fdecl)
			| Access (e, member) ->
				let cls_lit, env = eval env e in
				id_cls_variable_lookup ((get_lhs_varname cls_lit), member, env, "undeclared access identifier: "), env
			| Call(func, actuals) ->
				let get_context_cls_func =
					if NameMap.cardinal (snd env.env_context) = 0 then
						""
					else
						let cls_name = (get_class_name (ClassLiteral(snd env.env_context))) in
						let cls_func_name = cls_name ^ "." ^ func in
						if NameMap.mem cls_func_name func_decls then
							cls_func_name
						else
							"" in
				let cls_func_name = get_context_cls_func in	
				if String.length cls_func_name > 0 then
					let cls_lit = LhsLiteral((fst env.env_context), ClassLiteral(snd env.env_context)) in
					let actuals, env = eval_actual_list (actuals, env, eval) in
					cls_call_func (call, env, func_decls, cls_func_name, actuals, cls_lit, call_fdecl)
			  	else 
					let actuals, env = eval_actual_list (actuals, env, eval) in	
					
					try
						let fdecl = find_fdecl (func, func_decls) in
						let call_env = { env with env_locals = NameMap.empty; env_context = get_def_env_context } in
						let call_lit, post_call_env = call_fdecl (call, fdecl, actuals, call_env) in
						call_lit, {env with env_globals = post_call_env.env_globals}
					with FunctionNotFoundException s -> 
						try
							try_func_as_class_constructor (actuals, env, classes, func, init_var)
						with Failure s1 ->
							raise (FunctionNotFoundException(s))
		in
		(* Execute a statement and return an updated environment *)
		let rec exec env = function
			Block(stmts) -> List.fold_left exec env stmts
			| Expr(e) -> let _, env = eval env e in env
			| If(e, s1, s2) ->
				let v, env = eval env e in
				exec env (if (bool_of_literal v) then s1 else s2)
			| While(e, s) ->
				let rec loop env =
					let v, env = eval env e in
					if (bool_of_literal v) then loop (exec env s) else env
					in loop env
			| For(e1, e2, e3, s) ->
				let _, env = eval env e1 in
				let rec loop env =
					let v, env = eval env e2 in
					if (bool_of_literal v) then
						let _, env = eval (exec env s) e3 in
						loop env
					else
						env
				in loop env
			| Return(e) ->
				let v, env = eval env e in
				raise (ReturnException(v, env))
		in

		(* Enter the function: bind actual values to formal arguments *)
		let locals =
			try List.fold_left2
				(fun locals formal actual -> NameMap.add formal.vname actual locals)
				NameMap.empty fdecl.formals actuals
			with Invalid_argument(_) -> raise (Failure ("wrong number of arguments passed to " ^ fdecl.fname))
		in
		(* Initialize local variables to 0 *)
		let locals = List.fold_left
			(fun accum local -> NameMap.add local.vname (init_var classes local.vtype) accum) 
			locals fdecl.locals
		in
		let locals = NameMap.fold
			(fun key value accum -> NameMap.add key value accum) 
			call_env.env_locals locals
		in
		(* Execute each statement in sequence, return updated global symbol table *)
		(List.fold_left exec { call_env with env_locals = locals } fdecl.body)

	(* Run a program: initialize global variables to 0, find and run "main" *)
	in let globals = List.fold_left
		(fun globals vdecl -> NameMap.add vdecl.vname (init_var classes vdecl.vtype) globals) 
		NameMap.empty vars
	in try
		call (NameMap.find "main" func_decls) [] 
			{ env_globals = globals; 
			  env_locals = NameMap.empty; 
			  env_context = get_def_env_context }
	with Not_found -> raise (Failure ("did not find the main() function"))
