open Ast
open Utility
open Environment

(* ******************* List Functions ******************* *)
let get_index index actuals =
	if (List.length actuals) <= index then
		raise (Failure("List function does not have enough parameters")) 
	else
		(List.nth actuals index)

let check_paramter_size func_name size actuals =
	if (List.length actuals) != size then
		let err_func_name = "List function " ^ func_name ^ " should have " in
		raise (SemanticFailure(err_func_name ^ (string_of_int size) ^ " parameters")) 
	else
		List.length actuals


let check_list_func (cls_lit, cls_func, actuals, env) = 
	let rec get_int_of_literal = function
		IntLiteral(lit)    -> lit
		| LhsLiteral(var, lit) -> get_int_of_literal lit
		| _ -> raise (SemanticFailure("List function parameter must be an integer")) in
	let items = get_list_items cls_lit in
	(* Can reuse most of the code for push and insert into a list *)
	let list_generic_insert (new_item, index) =
		let new_list = ListLiteral(insert_item (new_item, index) items) in
		let var_name = get_lhs_varname cls_lit in
		let err_msg = "Unable to perform " ^ cls_func ^ " on the list" in
		let ret_val, env = assign_variable_lookup (var_name, new_list, env, err_msg) in
		new_item, env in
	match cls_func with
		"at" -> 
			let _ = (check_paramter_size cls_func 1 actuals) in
			let index = (get_int_of_literal (get_index 0 actuals)) in
			if (index < 0) then
				raise(SemanticFailure("List At index cannot be negative: " ^ (string_of_int index)))
			else if (List.length items == 0) then
				raise(SemanticFailure("Semantic Parsing Error: Cannot index an empty list"))
			else
				let item = List.hd items in
				item, env
		| "count" -> 
			let _ = (check_paramter_size cls_func 0 actuals) in
			IntLiteral(List.length items), env
		| "insert" -> 
			let _ = (check_paramter_size cls_func 2 actuals) in
			let new_item = (get_index 1 actuals) in
			let _ = (get_int_of_literal (get_index 0 actuals)) in
			list_generic_insert (new_item, 0)
		| "pop" -> 
			let _ = (check_paramter_size cls_func 0 actuals) in
			if (List.length items == 0) then
				raise(SemanticFailure("Semantic Parsing Error: Cannot pop an empty list"))
			else			
				List.hd items, env
		| "push" -> 
			let _ = (check_paramter_size cls_func 1 actuals) in
			let new_item = (get_index 0 actuals) in
			let index = List.length items in
			list_generic_insert (new_item, index)
		| "remove" -> 
			let _ = (check_paramter_size cls_func 1 actuals) in
			let index = (get_int_of_literal (get_index 0 actuals)) in
			if (index < 0) then
				raise(SemanticFailure("List Remove index cannot be negative: " ^ (string_of_int index)))
			else if (List.length items == 0) then
				raise(SemanticFailure("Semantic Parsing Error: Cannot remove an empty list"))
			else			
				List.hd items, env
		| "reverse" -> 
			let _ = (check_paramter_size cls_func 0 actuals) in
			cls_lit, env
		| _ as func -> raise (SemanticFailure("List function not definied: " ^ func))


let eval_list_func (cls_lit, cls_func, actuals, env) = 
	let rec get_int_of_literal = function
		IntLiteral(lit)    -> lit
		| LhsLiteral(var, lit) -> get_int_of_literal lit
		| _ -> raise (Failure("List function parameter must be an integer")) in
	let items = get_list_items cls_lit in
	(* Can reuse most of the code for push and insert into a list *)
	let list_generic_insert (new_item, index) =
		let new_list = ListLiteral(insert_item (new_item, index) items) in
		let var_name = get_lhs_varname cls_lit in
		let err_msg = "Unable to perform " ^ cls_func ^ " on the list" in
		let ret_val, env = assign_variable_lookup (var_name, new_list, env, err_msg) in
		new_item, env in
	match cls_func with
		"at" -> 
			let index = (get_int_of_literal (get_index 0 actuals)) in
			if ((List.length items) <= index) then
				raise(Failure("List At index is out of range: " ^ (string_of_int index)))
			else
				let item = List.nth items (get_int_of_literal (get_index 0 actuals))  in
				item, env
		| "count" -> 
			IntLiteral(List.length items), env
		| "insert" -> 
			let new_item = (get_index 1 actuals) in
			let index = (get_int_of_literal (get_index 0 actuals)) in
			list_generic_insert (new_item, index)
		| "pop" -> 
			let (new_list, last_item) = pop_item [] items in
			let var_name = get_lhs_varname cls_lit in
			let err_msg = "Unable to pop from list" in
			let ret_val, env = assign_variable_lookup (var_name, ListLiteral(new_list), env, err_msg) in
			last_item, env
		| "push" -> 
			let new_item = (get_index 0 actuals) in
			let index = List.length items in
			list_generic_insert (new_item, index)
		| "remove" -> 
			let index = (get_int_of_literal (get_index 0 actuals)) in
			if ((List.length items) <= index) then
				raise(Failure("List At index is out of range: " ^ (string_of_int index)))
			else
				let removed_item = List.nth items (get_int_of_literal (get_index 0 actuals))  in
				let new_list = ListLiteral(remove_item index items) in
				let var_name = get_lhs_varname cls_lit in
				let err_msg = "Unable to remove from list" in
				let ret_val, env = assign_variable_lookup (var_name, new_list, env, err_msg) in
				removed_item, env
		| "reverse" -> 
			let new_list = ListLiteral(List.rev items) in
			let var_name = get_lhs_varname cls_lit in
			let err_msg = "Unable to reverse list" in
			assign_variable_lookup (var_name, new_list, env, err_msg)
		| _ as func -> raise (Failure("List function not definied: " ^ func))


let list_concat (lit1, lit2) = 
	let items1 = get_list_items lit1 in
	let items2 = get_list_items lit2 in
	ListLiteral(items1 @ items2)

