open Ast
open Utility

type action = Ast | Ncgenerator | Sast
exception ArgumentFailure of string
exception ParseFailure of string

let read_file file_name =
	let lines = ref [] in
	let channel = open_in file_name in
	try
		while true; do
			lines := input_line channel :: !lines
	  	done; !lines
	with End_of_file ->
		close_in channel;
		List.rev !lines ;;

let _ =
	if Array.length Sys.argv == 2 then
		raise (ArgumentFailure("There must be two arguments: mode (-nc,-s, -a) and file name."))
	else
		let pre_file_name = Sys.argv.(2) in
		let file_name = (pre_file_name ^ ".tmp") in
		let err_file_name = (pre_file_name ^ ".err") in
		let pre_parser_return = Sys.command ("python pre_parser.py " ^ pre_file_name) in
		if Sys.file_exists err_file_name then
			let lines = read_file err_file_name in
 			List.iter (fun line -> print_endline line) lines
		else
			if pre_parser_return != 0 then
				print_endline ("***** Error running python pre_parser.py on file: " ^ pre_file_name ^ 
								"\nSystem command returned " ^ (string_of_int pre_parser_return))
			else
				let action = List.assoc Sys.argv.(1) [ ("-a", Ast); ("-nc", Ncgenerator); ("-s", Sast) ] in
				let lexbuf = Lexing.from_channel (open_in file_name) in
				try
					let program = Parser.program Scanner.token lexbuf in
					match action with
						Ast -> let listing = string_of_program program
				   			in print_string listing
						| Ncgenerator -> 
							ignore (Sast.semantic_check program); 
							ignore (Ncgenerator.run program)
						| Sast -> 
							ignore (Sast.semantic_check program);
							let listing = string_of_program program
				   			in print_string listing
				with 
					| Parsing.Parse_error ->
						let token = Lexing.lexeme lexbuf in
						print_endline ("Parsing error on token: " ^ token)
					| Failure s ->
						print_endline ("***** Syntax error: " ^ s)
					| FunctionNotFoundException s ->
						print_endline ("***** Syntax error: " ^ s)
					| Scanner.Syntax_error s ->
						print_endline ("***** Syntax error: " ^ s)
					| ClassConstructorException s ->
						print_endline ("***** Syntax error: " ^ s)
					| SemanticFailure s ->
						print_endline ("***** Semantic error: " ^ s)



