open Ast
open Utility
open Class

type environment = {
    env_globals  : literal NameMap.t;
    env_locals   : literal NameMap.t;
    env_context : string * literal NameMap.t; (* context (if inside a class function call *)
  }

exception ReturnException of literal * environment

(* ******************* envirnoment FUNCTIONS ******************* *)

(* by default, there is no context.  Only set when inside class function call *)
let get_def_env_context = ("", NameMap.empty)

(* look at locals first, then at class scope, finally look in globals *)
let variable_lookup (var_name, env, err_msg) =
  if NameMap.mem var_name env.env_locals then
    env.env_locals
  else if NameMap.mem var_name (snd env.env_context) then
    (snd env.env_context)
  else if NameMap.mem var_name env.env_globals then
    env.env_globals
  else raise (Failure (err_msg ^ var_name))

let id_variable_lookup (var_name, env, err_msg) =
	let map_match = variable_lookup (var_name, env, err_msg) in
	try
		LhsLiteral(var_name, (NameMap.find var_name map_match)), env
	with Not_found -> raise (Failure ("Unable to look up variable" ^ var_name))

(* recurse through a list of class variables and find the nested member variable *)
let rec rec_lst_cls_variable_lookup cls_data_map = function
	[] -> raise (Failure ("Internal error: Variable lookup failure"))
	| hd::[] -> raise (Failure ("Internal error: Variable lookup failure"))
	| hd::tl -> 
		let cls_member_name = (List.hd tl) in
		if NameMap.mem cls_member_name cls_data_map then
			try
				let match_data = (NameMap.find cls_member_name cls_data_map) in
				if (List.length tl) = 1 then
					match_data
				else
					rec_lst_cls_variable_lookup (get_class_value_map match_data) tl
			with Not_found -> raise (Failure ("Unable to find class member " ^ cls_member_name))
	  	else raise (Failure ("WHAAAT"))

let id_cls_variable_lookup (in_cls_var_name, cls_member_name, env, err_msg) =
	(* split the variable name on the access operator (.) *)
	let split_var_name = (rec_split_string '.' in_cls_var_name []) in
	let cls_var_lst = List.rev (cls_member_name::(List.rev split_var_name)) in
	let cls_var_name = (List.hd cls_var_lst) in	
	let map_match = variable_lookup (cls_var_name, env, err_msg) in
	try
		(* recurse through the class members to find the matching variables *)
		let cls_data_map = get_class_value_map (NameMap.find cls_var_name map_match) in
		let long_var_name = in_cls_var_name ^ "." ^ cls_member_name in
		try
			let cls_lit = (rec_lst_cls_variable_lookup cls_data_map cls_var_lst) in
			let lit = LhsLiteral(long_var_name, cls_lit) in
			lit
		with Failure s -> raise (Failure (err_msg ^ long_var_name)) 
	with Not_found -> raise (Failure ("Unable to find member in class: " ^ cls_var_name))

(* recurse through a list of class variables and find the nested member variable *)
let rec rec_cls_assign_variable_lookup (cls_data_map, v_val) = function
	[] -> raise (Failure ("Internal error: Variable assignment failure"))
	| hd::[] ->
		if NameMap.mem hd cls_data_map then
			ClassLiteral(NameMap.add hd v_val cls_data_map)
		else raise (Failure ("Internal error: Variable assignment failure"))
	| hd::tl -> 
		if NameMap.mem hd cls_data_map then
			let child_cls_data_map = get_class_value_map (NameMap.find hd cls_data_map) in
			let cls_lit = (rec_cls_assign_variable_lookup (child_cls_data_map, v_val) tl) in
			ClassLiteral(NameMap.add hd cls_lit cls_data_map)
		else raise (Failure ("Internal error: Variable assignment failure"))


let rec assign_variable_lookup (var_name, v_val, env, err_msg) =
	(* split the variable name on the access operator (.) *)
	let split_var_name = (rec_split_string '.' var_name []) in
	if (List.length split_var_name) > 1 then
		(* recurse through the class members to find the matching variables *)
		let class_var_name = (List.hd split_var_name) in
		let map_match = variable_lookup (class_var_name, env, err_msg) in
		try
			let cls_data_map = get_class_value_map (NameMap.find class_var_name map_match) in
			let cls_val = rec_cls_assign_variable_lookup (cls_data_map, v_val) (List.tl split_var_name) in
			let _, env = assign_variable_lookup(class_var_name, cls_val, env, err_msg) in
			v_val, env
		with 
			| Not_found -> raise (Failure ("Could not find class variable: " ^ class_var_name))
			| Failure s -> raise (Failure ("Could not find class variable: " ^ class_var_name))
	else
		(* find the class member and update value *)
		if NameMap.mem var_name env.env_locals then
			v_val, {env with env_locals = NameMap.add var_name v_val env.env_locals }
		else if NameMap.mem var_name (snd env.env_context) then
			let context = ((fst env.env_context), (NameMap.add var_name v_val (snd env.env_context))) in
			v_val, {env with env_context = context}
		else if NameMap.mem var_name env.env_globals then
			v_val, {env with env_globals = NameMap.add var_name v_val env.env_globals}
		else raise (Failure (err_msg ^ var_name))

(* update the environment after a class function call *)
let update_env_class_call (pre_call_env, post_call_env, err_msg) =
	(* split the function name on the access operator (.) *)
	let split_var_name = (rec_split_string '.' (fst post_call_env.env_context) []) in
	let var_name = (List.hd split_var_name) in
	let rec rec_update_local_map (cls_map, original_cls_map) = function
		[] -> raise (Failure ("Internal error: Update env contains invalid data"))
		| hd::[] -> 
			(* Reached the member variable, so update the class map *)
			let updated_cls_value = ClassLiteral((snd post_call_env.env_context))  in
			ClassLiteral(NameMap.add hd updated_cls_value original_cls_map) 
		| hd::tl -> 
			(* Update member class object *)
			let child_cls_data_map = get_class_value_map (NameMap.find hd cls_map) in
			let cls_lit = (rec_update_local_map (child_cls_data_map, child_cls_data_map) tl) in
			(* return the new object with the updated class member map *)
			ClassLiteral(NameMap.add hd cls_lit cls_map)
	in
	let update_local_map var_map = 
		(* if this is not a member class object update *)
		if (List.length split_var_name) = 1 then
			let cls_map = get_class_value_map (NameMap.find var_name var_map) in 
			let updated_cls_value = ClassLiteral(NameMap.fold 
				(fun key value accum -> NameMap.add key value accum)
				(snd post_call_env.env_context) cls_map) in
			NameMap.add var_name updated_cls_value var_map
		else
			let cls_map = get_class_value_map (NameMap.find var_name var_map) in 
			let updated_cls_value = (rec_update_local_map (cls_map, cls_map) (List.tl split_var_name)) in
			NameMap.add var_name updated_cls_value var_map
	in
	try
		(* Update of a local variable *)
		if NameMap.mem var_name pre_call_env.env_locals then
			{ pre_call_env with 
				env_locals = (update_local_map pre_call_env.env_locals);
				env_globals = post_call_env.env_globals }
		(* Update of a global variable *)
		else if NameMap.mem var_name pre_call_env.env_globals then
			{ pre_call_env with env_globals = (update_local_map pre_call_env.env_globals) }
		(* if the updated object is the calling object *)
		else if (fst pre_call_env.env_context) = (fst post_call_env.env_context) then
			{ pre_call_env with env_globals = post_call_env.env_globals;
				env_context = post_call_env.env_context }
		(* if the updated object is a member of the calling object *)
		else 
			let update_implict_map = update_local_map (snd pre_call_env.env_context) in
			{ pre_call_env with env_globals = post_call_env.env_globals;
				env_context = ((fst pre_call_env.env_context), update_implict_map)}
	with Not_found -> raise (Failure ("Did not find the class variable " ^ var_name))


(* ******************* Function Mapping FUNCTIONS ******************* *)



(* return class functions with the format class.function_name *)
let class_func_decls cls_decl = 
	List.map (fun fun_decl -> ((cls_decl.class_name ^ "." ^ fun_decl.fname) , fun_decl))
	cls_decl.function_members


(* return a map with all the functions *)
let map_funcs (funcs, classes) =
	let func_decls_first = List.fold_left
		(fun funcs fdecl -> NameMap.add fdecl.fname fdecl funcs)
		NameMap.empty funcs
	in
	let func_decls = List.fold_left 
		(fun accum cls_decl -> List.fold_left
			(fun accum name_fun_pair -> NameMap.add (fst name_fun_pair) (snd name_fun_pair) accum)
			accum (class_func_decls cls_decl))
		func_decls_first classes
	in
		func_decls


let find_fdecl (func, func_decls) =
	try NameMap.find func func_decls
	with Not_found -> raise (FunctionNotFoundException ("undefined function " ^ func))

let cls_call_func (call, env, func_decls, func_name, actuals, cls_lit, call_fdecl) =
	let fdecl = find_fdecl (func_name, func_decls) in
	let cls_context = ((get_lhs_varname cls_lit), (get_class_value_map cls_lit)) in
	let call_env = {env with env_locals = NameMap.empty; env_context = cls_context} in
	let call_lit, post_call_env = call_fdecl (call, fdecl, actuals, call_env) in
	let err_msg = "Unknown context: " ^ (fst call_env.env_context) in
	let return_env = update_env_class_call (env, post_call_env, err_msg) in 
	call_lit, return_env

