open Ast
open Utility


(* ******************* CLASS FUNCTIONS ******************* *)

(* Initalize a class object *)
let get_class_default classes classname init_var =
	try
	  let cls_decl = List.find (fun cls_decl -> cls_decl.class_name = classname) classes in
	  let cls_data = List.fold_left 
		(fun accum member -> NameMap.add member.vname (init_var classes member.vtype) accum) 
		NameMap.empty  cls_decl.data_members in
	  ClassLiteral(NameMap.add "_cls.Name" (StringLiteral(classname)) cls_data)
	with Not_found -> raise (Failure ("Class not found " ^ classname))

let rec init_var class_map vtype =
  match vtype with
    Int -> IntLiteral(0)
    | Bool -> BoolLiteral(false)
    | List(var) -> ListLiteral([])
    | String -> StringLiteral("")
    | Double -> DoubleLiteral(0.0)
    | ClassType(classname) ->  get_class_default class_map classname init_var

(* find the class name *)
let rec get_class_name = function
	IntLiteral(lit) -> raise (Failure ("Int is not a class object: " ^ string_of_int lit))
	| BoolLiteral(lit) -> raise (Failure ("Bool is not a class object: " ^ string_of_bool lit))
	| StringLiteral(lit) -> raise (Failure ("String is not a class object: " ^ lit))
	| DoubleLiteral(lit) -> raise (Failure ("Double is not a class object: " ^ string_of_float lit))
	| LhsLiteral(var, lit) -> get_class_name lit
	| ListLiteral(items) ->  raise (Failure ("List is not a class object"))
	| KVPLiteral(key, lit)-> raise (Failure ("KVP is not a class object"))
	| ClassLiteral(cls_map) as cls_lit -> 
		try 
			let cls_name = NameMap.find "_cls.Name" cls_map in
			string_of_literal cls_name
		with Not_found -> raise (Failure ("Class not intialized correctly: " ^ (string_of_literal cls_lit) ))

(* get a class member map *) 
let rec get_class_value_map = function
  IntLiteral(lit) -> raise (Failure ("Int is not a class object: " ^ string_of_int lit))
  | BoolLiteral(lit) -> raise (Failure ("Bool is not a class object: " ^ string_of_bool lit))
  | StringLiteral(lit) -> raise (Failure ("String is not a class object: " ^ lit))
  | DoubleLiteral(lit) -> raise (Failure ("Double is not a class object: " ^ string_of_float lit))
  | LhsLiteral(var, lit) -> get_class_value_map lit 
  | ListLiteral(items) ->  raise (Failure ("List is not a class object"))
  | KVPLiteral(key, lit)-> raise (Failure ("KVP is not a class object"))
  | ClassLiteral(cls_map) -> cls_map

(* get a class member value *) 
let get_class_value (member, lit) =
  try 
    NameMap.find member (get_class_value_map lit)
  with Not_found -> raise (Failure ("Class does not contain member: " ^ member))


(* ******************* CLASS fdecl FUNCTIONS ******************* *)

(* See if the function is a class type and create an instance of that class *)
let try_func_as_class_constructor (actuals, env, classes, func, init_var) =
	let cls_lit = (get_class_default classes func init_var) in 
	if (List.length actuals) = 0 then
		cls_lit, env
	else if (List.length actuals) != 1 then
		raise (ClassConstructorException("Class constructors must be initialized with key value pairs"))
	else
		(* parse key (member name) value pair list and set class data *) 
		let get_kvp = function
			KVPLiteral(key, lit) -> (key, lit)
			| _ -> raise (ClassConstructorException("Class constructors must be initialized with key value pairs")) in
		let tmp_items = (get_list (List.hd actuals)) in
		let key_value_pairs = List.map get_kvp tmp_items in
		let cls_map = get_class_value_map cls_lit in		
		let cls_map = List.fold_left 
						(fun accum item -> 
							if NameMap.mem (fst item) accum then
								NameMap.add (fst item) (snd item) accum
							else
								raise (ClassConstructorException("Class constructor key does not exist: " ^ (fst item))) )
						cls_map key_value_pairs in
		ClassLiteral(cls_map), env








