{ open Parser }

let number = ['0'-'9']
let letter = ['a'-'z' 'A'-'Z']
let integer = number+
let exponent =  'e' ['+' '-']? integer
let fraction = '.' integer

rule token = parse
  [' ' '\t' '\r' '\n']  { token lexbuf }
| "//"                  { comment lexbuf }
| '('                   { LPAREN }
| ')'                   { RPAREN }
| '{'                   { LBRACE }
| '}'                   { RBRACE }
| '['                   { LBRACKET }
| ']'                   { RBRACKET }
| ';'                   { SEMICOLON }
| ','                   { COMMA }
| '='                   { ASSIGN }
| "=="                  { EQ }
| "!="                  { NE }
| '<'                   { LT }
| "<="                  { LTE }
| '>'                   { GT }
| ">="                  { GTE }
| '+'                   { PLUS }
| '-'                   { MINUS }
| '*'                   { TIMES }
| '/'                   { DIVIDE }
| '!'                   { NOT }
| "&&"                  { AND }
| "||"                  { OR }
| '%'                   { MODULO }
| ':'                   { COLON }
| "if"                  { IF }
| "elseif"              { ELSEIF }
| "else"                { ELSE }
| "for"                 { FOR }
| "while"               { WHILE }
| "foreach"             { FOREACH }
| "return"              { RETURN }
| "function"            { FUNCTION }
| "kernel"              { KERNEL }
| "channel"             { CHANNEL }
| "in"                  { IN }
| "out"                 { OUT }
| "true" as s           { BOOL(Pervasives.bool_of_string s) }
| "false" as s          { BOOL(Pervasives.bool_of_string s) }
| "null"                { NULL }
| (integer? fraction exponent?) | (integer '.'? exponent) | (integer '.')
    as num              { FLOAT(float_of_string num) }
| number+ as s          { INT(int_of_string s) }
| (letter | '_') (letter | number | '_')* as s    { ID(s) }
| '"'                   { STRING(string_literal "" lexbuf) }
| eof                   { EOF }
| _ as char { raise (Failure("illegal character " ^ Char.escaped char)) }

(* Ignore everything from // to end of line *)
and comment = parse
  ['\r' '\n' ]  { token lexbuf }
| _             { comment lexbuf }

(* Match string literals.
 * TODO: other escape sequences?
 * TODO: this is maybe super-inefficient. *)
and string_literal str = parse
  ([^ '"' '\\']*) as s  { string_literal (str ^ s) lexbuf }
| '\\' '"'              { string_literal (str ^ "\"") lexbuf }
| '"'                   { str }
