#!/bin/bash
PLS=./pls
PLS_OPTIONS=

DIFF=/usr/bin/diff
DIFF_TARGET=/dev/null

TESTS_DIR=./tests
TEST_SCRIPTS=${TESTS_DIR}/*.pls

NUM_FAILED=0

VERBOSE=0

#GREEN='\e[0;32m'
#YELLOW='\e[1;33m'
#RED='\e[0;31m'
#NO_COLOR='\e[0m'

LOG_FILE=testlog.log
LOG_OUTPUT="Test started at $(date)"

print_usage() {
  echo "Usage: $0 [-h] [-v] [-o file]"
  echo "    -h: show this help"
  echo "    -v: verbose output"
  echo "    -o <file>: log output to 'file' (default: ./testlog.log)"
  exit 0
}

log_string() {
  LOG_OUTPUT="${LOG_OUTPUT}\n$@"
}

output() {
  echo -e "$@"
  log_string "$@"
}

debug() {
  if [ $VERBOSE -eq 1 ]; then
    output "$@"
  else
    log_string "$@"
  fi
}

run_test() {
  local script_name=$1;
  local input_file=${script_name/%pls/in}
  local output_file=${script_name/%pls/out}

  local test_name=$(basename ${script_name})
  test_name=${test_name%.pls}

  # gtest? never heard of it!
  output  "[ RUN       ] ${test_name}"

  if [ ! -e ${input_file} ]; then
    output "[      WARN ] ${test_name}: input file \"$(basename $input_file)\" does not exist - skipping"
    return 0
  fi

  if [ ! -e ${output_file} ]; then
    output "[      WARN ] ${test_name}: output file \"$(basename $output_file)\" does not exist - skipping"
    return 0
  fi

  local script_output=$(${PLS} ${PLS_OPTIONS} ${script_name} < ${input_file})

  debug "output:\n$script_output"
  local diff_output=$(${DIFF} ${output_file} <(echo "${script_output}"))
  #echo "${diff_output}"
  if [ -z "${diff_output}" ]; then
    output  "[        OK ] ${test_name}"
  else
    debug "diff failed:\n${diff_output}"
    output  "[      FAIL ] ${test_name}"
    NUM_FAILED=$(($NUM_FAILED + 1))
  fi
}

#####

while getopts hvo: c; do
  case $c in
    v) VERBOSE=1;;
    h) print_usage;;
    o) LOG_FILE=${OPTARG}
  esac
done

for script in ${TEST_SCRIPTS}; do
  run_test ${script}
done

if [ $NUM_FAILED -eq 0 ]; then
  output "All tests passed!"
else
  output "Error: $NUM_FAILED tests failed!"
  echo "See ${LOG_FILE} for details"
fi

echo -e "$LOG_OUTPUT" > ${LOG_FILE}

exit $NUM_FAILED
