open Compiler

let _ =
    let dump_ast      = ref false
    and dump_bytecode = ref false
    and run           = ref true
    and debug         = ref false
    and filename      = ref ""
    in

    let arg_spec = [
        ("-a", Arg.Set(dump_ast),      "Dump program AST");
        ("-b", Arg.Set(dump_bytecode), "Dump program bytecode");
        ("-d", Arg.Clear(run),         "Dry run mode (don't execute)");
        ("-g", Arg.Set(debug),         "Print extra debugging information")
    ]
    in

    let usage_message = String.concat "\n" [
        "PipeLine Script compiler";
        "Usage: " ^ Sys.argv.(0) ^ " [options] <filename>";
        "";
        "Compile and run the PLS script <filename>";
        "";
        "Options:"
    ]
    in

    Arg.parse arg_spec (fun s -> filename := s) usage_message;

    if (!filename) = ""
    then
        Arg.usage arg_spec usage_message
    else
        let script_file =  open_in (!filename) in
        let lexbuf = Lexing.from_channel script_file in
        let program_ast = List.rev (Parser.program Scanner.token lexbuf) in
        if !dump_ast then
            ignore (print_endline (Ast.string_of_program program_ast));
        let prog = Compiler.compile program_ast in (
            if !dump_bytecode then
                ignore (print_endline (Bytecode.string_of_program prog));
            if (!run) = true then
                ignore (Vm.run_program prog !debug);
            close_in script_file
        )
