%{ open Ast %}

%token ASSIGN PLUS MINUS TIMES DIVIDE MODULO
%token AND OR NOT
%token EOF
%token LPAREN RPAREN LBRACE RBRACE LBRACKET RBRACKET
%token SEMICOLON COMMA COLON
%token EQ NE LT LTE GT GTE
%token IF ELSEIF ELSE FOR WHILE FOREACH RETURN
%token KERNEL FUNCTION
%token IN OUT CHANNEL
%token <int> INT
%token <float> FLOAT
%token <string> STRING
%token <string> ID
%token <bool> BOOL
%token NULL

%right ASSIGN
%left OR
%left AND
%left EQ NE
%left LT GT LTE GTE
%left PLUS MINUS
%left TIMES DIVIDE MODULO
%right NOT
%left LBRACKET

%start program
%type <Ast.program> program

%%

program:
    /* nothing */           { [] }
  | program func_decl       { FuncDecl($2) :: $1 }
  | program kernel_decl     { KernelDecl($2) :: $1 }
  | program stmt            { Stmt($2) :: $1 }
  | program channel_decl    { Channels($2) :: $1 }

channel_decl:
    CHANNEL id_list SEMICOLON   { $2 }

id_list:
    ID                  { [$1] }
  | id_list COMMA ID    { $3 :: $1 }

func_decl:
    FUNCTION ID LPAREN func_param_list_opt RPAREN block
        { { name = $2; args = $4; body = $6} }

func_param_list_opt:
    /* empty args */    { [] }
  | id_list             { List.rev $1 }

kernel_decl:
    KERNEL ID LPAREN kernel_arg_list RPAREN block
        { { kname = $2; kargs = List.rev $4; kbody = $6 } }

kernel_arg_list:
    kernel_arg                          { [$1] }
  | kernel_arg_list COMMA kernel_arg    { $3 :: $1 }

kernel_arg:
    IN ID   { Input($2) }
  | OUT ID  { Output($2) }
  | ID      { BasicArg($1) }

stmt:
    expr SEMICOLON                              { ExprStmt($1) }
  | RETURN expr SEMICOLON                       { Return($2) }
  | block                                       { Block($1) }
  | IF LPAREN expr RPAREN block else_opt        { If($3, Block($5), $6) }
  | FOR LPAREN expr_opt SEMICOLON expr_opt SEMICOLON expr_opt RPAREN block
        { For(ExprStmt($3), $5, ExprStmt($7), Block($9)) }
  | WHILE LPAREN expr RPAREN block              { While($3, Block($5)) }
  | FOREACH LPAREN ID COLON expr RPAREN block   { ForEach($3, $5, Block($7)) }

else_opt:
    /* nothing */   { Block([]) }
  | ELSE block      { Block($2) }
  | elseif          { $1 }

elseif:
    ELSEIF LPAREN expr RPAREN block else_opt { If($3, Block($5), $6) }

stmt_list:
    /* nothing */   { [] }
  | stmt_list stmt  { $2 :: $1 }

block:
    LBRACE stmt_list RBRACE { List.rev $2 }

expr_opt:
    /* nothing */   { EmptyExpr }
  | expr            { $1 }

assignable:
    expr LBRACKET expr RBRACKET       { ListIndex($1, $3) }
  | ID                                { Id($1) }

primary_expr:
    assignable                          { Assignable($1) }
  | literal                             { Literal($1) }
  | dlist                               { $1 }
  | LPAREN expr RPAREN                  { $2 }
  | ID LPAREN expr_list_opt RPAREN      { Call($1, $3) }

expr:
    primary_expr                        { $1 }
  | ID ASSIGN expr                      { Assign(Id($1), $3) }
  | expr PLUS expr                      { Binop($1, Add, $3) }
  | expr MINUS expr                     { Binop($1, Subtract, $3) }
  | expr TIMES expr                     { Binop($1, Multiply, $3) }
  | expr DIVIDE expr                    { Binop($1, Divide, $3) }
  | expr MODULO expr                    { Binop($1, Modulo, $3) }
  | expr EQ expr                        { Binop($1, Equal, $3) }
  | expr NE expr                        { Binop($1, Neq, $3) }
  | expr LT expr                        { Binop($1, Less, $3) }
  | expr LTE expr                       { Binop($1, Leq, $3) }
  | expr GT expr                        { Binop($1, Greater, $3) }
  | expr GTE expr                       { Binop($1, Geq, $3) }
  | MINUS expr                          { Binop(Literal(Int(0)), Subtract, $2) }
  | NOT expr                            { Binop(Literal(Bool(true)), Neq, $2) }
  | expr AND expr                       { Binop($1, And, $3) }
  | expr OR expr                        { Binop($1, Or, $3) }

expr_list:
    expr                    { [$1] }
  | expr_list COMMA expr    { $3 :: $1 }

expr_list_opt:
    /* nothing */   { [] }
  | expr_list       { $1 }

literal:
    INT     { Int($1) }
  | FLOAT   { Float($1) }
  | STRING  { String($1) }
  | BOOL    { Bool($1) }
  | NULL    { Null }

dlist:
    LBRACKET expr_list_opt RBRACKET { DList(List.rev $2) }
