(* This type represents any value that can be operated on at runtime *)
type value =
    Int of int
  | Float of float
  | String of string
  | Bool of bool
  | VList of value list
  | Channel of int
  | Null

(*
 *  Bytecode | Description                                     | SP   | PC
 * ----------+-------------------------------------------------+------+--------
 *   Halt    | End program                                     | --   | --
 *   Push v  | Push 'v' on to stack                            | +1   | +1
 *   Pop     | Remove value from top of stack                  | +1   | +1
 *   Add     | Pop two values, push sum                        | -1   | +1
 *   Sub     | Pop two values, push difference                 | -1   | +1
 *   Mul     | Pop two values, push product                    | -1   | +1
 *   Div     | Pop two values, push quotient                   | -1   | +1
 *   Mod     | Pop two values, push remainder (sp-2) % (sp-1)  | -1   | +1
 *   Ceq     | Pop two values, push bool (sp-2) == (sp-1)      | -1   | +1
 *   Cne     | Pop two values, push bool (sp-2) != (sp-1)      | -1   | +1
 *   Clt     | Pop two values, push bool (sp-2) < (sp-1)       | -1   | +1
 *   Cle     | Pop two values, push bool (sp-2) <= (sp-1)      | -1   | +1
 *   Cgt     | Pop two values, push bool (sp-2) > (sp-1)       | -1   | +1
 *   Cge     | Pop two values, push bool (sp-2) >= (sp-1)      | -1   | +1
 *   And     | Pop two values, push (sp-2) && (sp-1)           | -1   | +1
 *   Or      | Pop two values, push (sp-2) || (sp-1)           | -1   | +1
 *   Strg n  | Store top of stack to global index              | --   | +1
 *   Ldg n   | Load global from arg index and push on stack    | +1   | +1
 *   Strl n  | Store top of stack to local index relative FP   | --   | +1
 *   Ldl n   | Load relative FP and push on stack              | +1   | +1
 *   Call n  | Call function at absolute offset 'n'          $ | +1   | =n
 *   Ret n   | Return from function of arity 'n'               |FP-n  | *
 *   Bz n    | Branch relative if top of stack is 0            | -1   | +1 or n
 *   Bnz n   | Branch relative if top of stack is not 0        | -1   | +1 or n
 *   Ba n    | Unconditional branch relative                   | --   | +n
 *   Mlst n  | Pop 'n' values, construct list, push list       | +1-n | +1
 *   Ilst    | Pop list, pop index, push value from list index | -1   | +1
 *   Alst    | Pop list at (sp-3), set idx (sp-2) to (sp-1)    | -2   | +1
 *   Read    | Pop channel index, push value from chan head #  | +1   | +1
 *   Write   | Pop val, pop chan index, write val to chan      | -1   | +1
 *   Ent n   | Push current FP, advance SP by 'n'              | +n+1 | +1
 *   Par n m | Create kernel instance at PC 'n' with arity 'm' | -n   | +1
 *   Kent n  | Advance SP by 'n'                               | +n   | +1
 *   Term    | Terminate kernel instance                       | --   | --
 *   Run     | Start kernel instances                          | --   | --
 *
 *  * = Retrieve old values from stack
 *  # = Will yield execution if channel is empty
 *  $ = Builtin functions are "inlined" and just consume arguments and push
 *      results without branching.
 *)

type bcode =
    Halt (* End program *)
  | Push of value (* Push a literal on the stack *)
  | Pop (* Pop a value off the stack *)
  | Add (* add *)
  | Sub (* subtract *)
  | Mul (* multiply *)
  | Div (* divide *)
  | Mod (* modulus *)
  | Ceq (* compare equal *)
  | Cne (* compare not equal *)
  | Clt (* compare less than *)
  | Cle (* compare less than or equal to *)
  | Cgt (* compare greater than or *)
  | Cge (* compare greater than or equal to *)
  | And (* logical and *)
  | Or (* logical or *)
  | Strg of int (* Store global var *)
  | Ldg of int (* Load var *)
  | Strl of int (* Store local var *)
  | Ldl of int (* Load local var *)
  | Call of int (* call func with specified index *)
  | Ret of int (* Return with number of args to the function *)
  | Bz of int (* Branch relative if top if stack zero *)
  | Bnz of int (* Branch relative if top of stack is not zero *)
  | Ba of int (* Branch relative always *)
  | Mlst of int (* Make a list with the top n values on the stack *)
  | Ilst (* Index a list *)
  | Alst (* Assign a value to a list *)
  | Read (* Push a value from the channel on the top of the stack *)
  | Write (* Write a value to a channel *)
  | Ent of int (* allocate room for locals, shift sp and fp for call *)
  | Par of int * int (* Spawn of kernel instance with num locals *)
  | Kent of int (* Allocate space for kernel locals *)
  | Term (* Terminate a kernel instance *)
  | Run (* Start the parallel processing *)

type program = {
    num_globals          : int;
    text                 : bcode array;
    num_channels         : int;
    num_kernel_instances : int;
}

(* Convert an Ast literal to a value for the stack *)
let rec value_of_literal = function
    Ast.Int(i)      -> Int(i)
  | Ast.Float(f)    -> Float(f)
  | Ast.String(s)   -> String(s)
  | Ast.Bool(b)     -> Bool(b)
  | Ast.Null        -> Null

let bool_of_value = function
    Int(i)      -> i != 0
  | Float(f)    -> f <> 0.0
  | String(_)   -> true
  | Bool(b)     -> b
  | VList(_)    -> true
  | Channel(_)  -> true
  | Null        -> false

(* Pretty Printing Functions *)
let rec string_of_value = function
    Int(i)      -> string_of_int i
  | Float(f)    -> string_of_float f
  | String(s)   -> s
  | Bool(b)     -> string_of_bool b
  | VList(l)    -> "[" ^ String.concat ", " (List.map string_of_value l) ^ "]"
  | Channel(c)  -> "[Channel " ^ string_of_int c ^ "]"
  | Null        -> "Null"

let string_of_bcode = function
    Halt        -> "Halt"
  | Push(value) -> "Push " ^ string_of_value value
  | Pop         -> "Pop"
  | Add         -> "Add"
  | Sub         -> "Sub"
  | Mul         -> "Mul"
  | Div         -> "Div"
  | Mod         -> "Mod"
  | Ceq         -> "Ceq"
  | Cne         -> "Cne"
  | Clt         -> "Clt"
  | Cle         -> "Cle"
  | Cgt         -> "Cgt"
  | Cge         -> "Cge"
  | And         -> "And"
  | Or          -> "Or"
  | Strg(idx)   -> "Strg " ^ string_of_int idx
  | Ldg(idx)    -> "Ldg " ^ string_of_int idx
  | Strl(idx)   -> "Strl " ^ string_of_int idx
  | Ldl(idx)    -> "Ldl " ^ string_of_int idx
  | Call(idx)   -> "Call " ^ string_of_int idx
  | Ret(nargs)  -> "Ret " ^ string_of_int nargs
  | Bz(idx)     -> "Bz " ^ string_of_int idx
  | Bnz(idx)    -> "Bnz " ^ string_of_int idx
  | Ba(idx)     -> "Ba " ^ string_of_int idx
  | Mlst(n)     -> "Mlst " ^ string_of_int n
  | Ilst        -> "Ilst"
  | Alst        -> "Alst"
  | Read        -> "Read"
  | Write       -> "Write"
  | Ent(num)    -> "Ent " ^ string_of_int num
  | Par(idx, nargs) -> "Par " ^ string_of_int idx ^ " " ^ string_of_int nargs
  | Kent(n)     -> "Kent " ^ string_of_int n
  | Term        -> "Term"
  | Run         -> "Run"

let string_of_program prog =
    "num_globals: " ^ string_of_int prog.num_globals ^ "\n" ^
    "num_channels: " ^ string_of_int prog.num_channels ^ "\n" ^
    "num_kernel_instances: " ^ string_of_int prog.num_kernel_instances ^ "\n" ^
    (String.concat "\n"
        (Array.to_list (Array.mapi
            (fun idx bcode ->
                Printf.sprintf "%-4d: %s" idx (string_of_bcode bcode))
            prog.text)))
