(****************************************************************************
 *
 * File: symbol.ml
 *
 * Purpose: defines and contains the multiple symbol tables:
 *  functions, globals, local variables
 *
 *)

type sym_var  = {vtype: Ast.expr_type; address: int; scope: int} 
type sym_func = {args : Ast.expr_type list; rtn: Ast.expr_type; fid: string}
type environment = {
  scope: int; 
  label: int; 
  mutable vars: string list; 
  mutable address: int;
  mutable rec_address: int;
  rtype: Ast.expr_type 
}

exception Duplicate_id of string

(* 
 * Seperate namespace for functions and variables 
 *  All variable are stack variables, there are no global variables 
 *)

let functions = Hashtbl.create 109
let global_vars = Hashtbl.create 109
let symbol_table = Hashtbl.create 109

(****************************************************************************
 * Return the next address for stored variables
 *)
let next_address env =
  let addr = env.address in 
  ignore(env.address <- succ env.address);
  addr

let init () = 
  let var = {vtype = Ast.Int; address = 0; scope = 0} in
  Hashtbl.add symbol_table "__force_table_setup__" var;
  Hashtbl.remove symbol_table "__force_table_setup__";

  Hashtbl.add functions "int_to_string"   
    {args=[Ast.Int]; rtn=Ast.String; fid = "Rtl/int_to_string"};
  Hashtbl.add functions "int_to_float"    
    {args=[Ast.Int]; rtn=Ast.Float; fid = "Rtl/int_to_float"};
  Hashtbl.add functions "int_to_char"     
    {args=[Ast.Int]; rtn=Ast.Char; fid = "Rlt/int_to_char"};
  Hashtbl.add functions "char_to_string"  
    {args=[Ast.Char]; rtn=Ast.String; fid = "Rtl/char_to_string"};
  Hashtbl.add functions "char_to_int"     
    {args=[Ast.Char]; rtn=Ast.Int; fid = "Rtl/char_to_int"};
  Hashtbl.add functions "string_to_int"   
    {args=[Ast.String]; rtn=Ast.Int; fid = "Rtl/string_to_int"};
  Hashtbl.add functions "string_to_float" 
    {args=[Ast.String]; rtn=Ast.Float; fid = "Rtl/string_to_float"};
  Hashtbl.add functions "string_to_char"  
    {args=[Ast.String]; rtn=Ast.Char; fid = "Rtl/string_to_char"};
  Hashtbl.add functions "float_to_string" 
    {args=[Ast.Float]; rtn=Ast.String; fid = "Rtl/float_to_string"};
  Hashtbl.add functions "float_to_int"    
    {args=[Ast.Float]; rtn=Ast.Int; fid = "Rtl/float_to_int"};
  Hashtbl.add functions "substr"          
    {args=[Ast.String; Ast.Int; Ast.Int]; rtn=Ast.String; fid = "Rtl/substr"};
  Hashtbl.add functions "concat"          
    {args=[Ast.String; Ast.String]; rtn=Ast.String; fid = "Rtl/concat"};
  Hashtbl.add functions "print"           
    {args=[Ast.String]; rtn=Ast.Void; fid = "Rtl/print"}


(*
 *  We have a list of symbol tables(Hash tables), we add a new one each time we enter
 *  a new block of code, which creates a new scope.  When we leave the that
 *  scope, we remove the added symbol table
 *)

(****************************************************************************
 * Return an empty environment
 *)
let create_initial_env () = 
  init ();
  {scope = -1; vars = []; label = -1; address = 0; rec_address = -1; rtype = Ast.Void}


(****************************************************************************
 * Create an empty environment at the top of the list of environments
 *)
let push_env env label naddr rtype =
  {
    scope = succ env.scope; 
    vars = []; 
    label = env.label; 
    address = naddr; 
    rec_address = env.rec_address;
    rtype = rtype
  }
 
let pop_env env =
  ignore(List.iter (fun elt -> Hashtbl.remove symbol_table elt) env.vars)

  
let add_local_var env name typ = 
  try 
    let var = Hashtbl.find symbol_table name in 
    if var.scope = env.scope then
      raise (Duplicate_id name)   
    else
      var
  with Not_found -> begin
    let addr = next_address env in
    let var = {vtype = typ; address = addr; scope = env.scope} in
    
    ignore(env.vars <- name :: env.vars);
    Hashtbl.add symbol_table name var;
    var
  end


(****************************************************************************
 * Add an variable to the curent environment 
 *)
let add_var env name typ = 
  try 
    let var = if env.scope = 0 then
      begin
        ignore(Hashtbl.find global_vars name);
        raise (Duplicate_id name)   
      end
    else
      Hashtbl.find symbol_table name in
    if var.scope = env.scope then
      raise (Duplicate_id name)   
    else 
      var
  with Not_found -> begin
    if env.scope = 0 then
      let var = {vtype = typ; address = 0; scope = env.scope} in
      Hashtbl.add global_vars name var;
      var
    else
      let addr = next_address env in
      let var = {vtype = typ; address = addr; scope = env.scope} in
      
      ignore(env.vars <- (name :: env.vars));
      Hashtbl.add symbol_table name var;
      var
  end

(****************************************************************************
 * Return the variable if it exists in the immediate scope only
 *)
let find_local_var env name =
  let var = if env.scope = 0 then
    Hashtbl.find global_vars name
  else
    Hashtbl.find symbol_table name in

  if var.scope = env.scope then
    var
  else 
    raise Not_found

(****************************************************************************
 *  Return variable if it exists some where in the layers of scope
 *)
let find_var env name =
  if env.scope = 0 then
    Hashtbl.find global_vars name
  else 
    Hashtbl.find symbol_table name

(****************************************************************************
 * find the function record 
 *)
let find_function name = 
  Hashtbl.find functions name 

let add_func name args rtn = 
  try 
    ignore(Hashtbl.find functions name);
    raise (Duplicate_id name)
  with Not_found ->
    Hashtbl.add functions name {args = args; rtn = rtn; fid = name}



