(****************************************************************************
 *
 * File: scanner.mll
 *
 * Purpose: language lex specification 
 *
 *)

{ 
open Parser 

let update_linenum lexbuf = 
  let pos = lexbuf.Lexing.lex_curr_p in
    lexbuf.Lexing.lex_curr_p <- { pos with
      Lexing.pos_lnum = pos.Lexing.pos_lnum + 1;
      Lexing.pos_bol = pos.Lexing.pos_cnum;
    }
}

let white_space = [' ' '\012' '\t' '\n' '\r']
let digit       = ['0'-'9']
let exponent    = "e" ['-''+']? digit+
let int_lit     = '-'? digit+
let float_lit   = (int_lit "." digit+) exponent? 
let string_lit  = '"' ('\\' '"' | [^ '"'])* '"'
let char_lit    = '\'' _ '\''
let true_lit    = "true"
let false_lit   = "false"
let id_start    = ['a'-'z' 'A'-'Z' '_' ]
let id_other    = ['a'-'z' 'A'-'Z' '_' '0'-'9' ]
let id          = id_start id_other*
let comment     = '#' [^ '\n']* 

rule token = parse
  [' ' '\t' '\r'] { token lexbuf } (* Whitespace *)
| '\n'          { update_linenum lexbuf; token lexbuf }
| comment       { update_linenum lexbuf; token lexbuf }           (* Comments *)
| '('           { LPAREN }
| ')'           { RPAREN }
| '{'           { LBRACE }
| '}'           { RBRACE }
| '['           { LBRACK }
| ']'           { RBRACK }
| ';'           { SEMI }
| ':'           { COLON }
| ','           { COMMA }
| '+'           { PLUS }
| '-'           { MINUS }
| '*'           { TIMES }
| '/'           { DIVIDE }
| '='           { ASSIGN }
| "=="          { EQ }
| "!="          { NEQ }
| '<'           { LT }
| "<="          { LEQ }
| ">"           { GT }
| ">="          { GEQ }

| "break"       { BREAK }
| "check"       { CHECK }
| "else"        { ELSE }
| "file"        { FILE }
| "filter"      { FILTER }
| "for"         { FOR }
| "function"    { FUNCTION }
| "if"          { IF }
| "output"      { OUTPUT }
| "print"       { PRINT }
| "process"     { PROCESS }
| "sep"         { SEP }
| "return"      { RETURN }
| "to"          { TO }
| "while"       { WHILE }

| "char"        { CHAR }
| "float"       { FLOAT }
| "int"         { INT }
| "string"      { STRING }

| float_lit as value            { LIT_FLOAT(value) }
| int_lit as value              { LIT_INT(value) }
| string_lit as value           { LIT_STR(value) }
| char_lit as value             { LIT_CHAR(value) }
| true_lit as value             { LIT_BOOL(value) }
| false_lit as value            { LIT_BOOL(value) }
| id as value                   { ID(value) }
| eof                           { EOF }

| _ as char { raise (Failure("illegal character " ^ Char.escaped char)) }


