#!/usr/bin/env python3

import argparse 
import filecmp
import sys
import re
import pdb
import os.path

from subprocess import Popen,PIPE,STDOUT
from glob import glob
from datetime import datetime
from os import path

g_logFile = None
g_traceFile = None
g_quiet = False
g_passedFiles = []
g_failedFiles = []


def trace_log(msg):
    pass


def write_log(msg):
    global g_logFile

    if g_logFile:
        g_logFile.write('{} {}\n'.format(datetime.today(), msg))


def make_file(filename):
    return open(filename, 'w')


def fail_msg(file):
    global g_quiet
    global g_failedFiles
    
    g_failedFiles.append(file)

    msg = '{} FAILED'.format(file)
    write_log(msg)

    if not g_quiet:
        print('FAILED')


def passed_msg(file):
    global g_quiet
    global g_passedFiles
        
    g_passedFiles.append(file)

    msg = '{} passed'.format(file)
    write_log(msg)

    if not g_quiet:
        print('PASSED')


def error_msg(msg):
    full_output = 'ERROR: {}'.format(msg)
    write_log(full_output)
    
    print(full_output)


def info_msg(msg):
    write_log('INFO: {}'.format(msg))
    

def trace_msg(msg):
    write_log('TRACE: {}'.format(msg))


def find_files(directory, filter, extension):
    """return a list of files with a given extension based on the supplied filter"""
    if filter != None:
        return glob('{}/*{}*.{}'.format(directory, filter, extension ))
    else:
        return glob('{}/*.{}'.format(directory, extension ))


def create_log_filename(directory, log_file):
    if log_file:
        return log_file

    return '{}/validate_{}.log'.format(directory, datetime.today().strftime('%Y%m%d_%H%M%S'))


def create_ref_filename(rtl_file, extension):
    name, ext = path.splitext(rtl_file)
    return name + extension

def generate_ast(bin_path, file, options):
    if not path.exists(file):
        fail_msg(file, 'failed, file does not exist')
        g_failedFiles.append((file,'rtl file missing'))
        return False

    trace_log('generating AST for {}'.format(file))

    args = [bin_path + 'rtl', '--ast', file]
    with Popen(args, stdout=PIPE, stderr=STDOUT) as proc:
        (stdoutdata, stderrdata) = proc.communicate()

    output = stdoutdata.decode('utf-8')

    of = make_file(create_ref_filename(file, '.output'))
    of.write(oarithmeticutput)
    of.close()

    write_log('finished {}'.format(file))
    return True

def compile_file(bin_path, file, options):
    if not path.exists(file):
        fail_msg(file, 'compilation failed, file does not exist')
        g_failedFiles.append((file,'rtl file missing'))
        return False

    trace_log('compiling {}'.format(file))

    args = [bin_path + 'rtl', file]
    g_traceFile.write('running:[{}]\n'.format(' '.join(args)))

    with Popen(args, stdout=PIPE, stderr=STDOUT) as proc:
        (stdoutdata, stderrdata) = proc.communicate()
        
    g_traceFile.write(stdoutdata.decode('utf-8'))

    pdb.set_trace()

    if re.search(r"fail_.*.rtl", file):
      oname = create_ref_filename(file, '.output')
      if stderrdata != None:
          ofile = open(oname, 'w')
          ofile.write(stderrdata.decode('utf-8'))

    write_log('compiled {}'.format(file))
    return True

def run_file(jar_file):
    ref_file = create_ref_filename(jar_file, '.output')
    bad_file = create_ref_filename(jar_file, '.bad')

    if not path.exists(jar_file):
        fail_msg(file, 'compilation failed, jar file {} does not exist'.format(jar_file))
        g_failedFiles.append((create_ref_filename(jar_file, '.rtl'),'jar file missing'))
        return False

    trace_log('running {}'.format(jar_file))

    args = ['java', '-jar', jar_file]
    g_traceFile.write('running:[{}]\n'.format(' '.join(args)))

    with Popen(args, stdout=PIPE, stderr=STDOUT) as proc:
        (stdoutdata, stderrdata) = proc.communicate()
        
    g_traceFile.write(stdoutdata.decode('utf-8'))
    out_file = open(ref_file, 'w')
    out_file.write(stdoutdata.decode('utf-8'))

    if stderrdata != None:
        bad_file = open(bad_file, 'w')
        bad_file.write(stderrdata.decode('utf-8'))


def compare_file(rtl_file, extension):
    global g_passedFiles
    global g_failedFiles

    #pdb.set_trace()

    output_file = create_ref_filename(rtl_file, '.output')
    ref_file = create_ref_filename(rtl_file, extension)

    if not path.exists(ref_file):
        # if there is no ref file, assume the output should be an empty file
        ref_file = './tests/empty' + extension

    if not path.exists(ref_file):
        fail_msg(rtl_file) 
        trace_msg('{} missing'.format(output_file))

        return False

    if filecmp.cmp(output_file, ref_file, shallow=False):
        passed_msg(rtl_file)
    else:
        fail_msg(rtl_file)
        trace_msg('{} did not match {}'.format(output_file, ref_file))

    return True


def parse_args():
    parser = argparse.ArgumentParser(description='run validation scripts agains rtl compiler')
    parser.add_argument('-l', '--log', dest='log_path', metavar='DIR',
            help='write logs to this drectory', default='.')
    parser.add_argument('-t', '--tests', dest='tests_path', metavar='DIR',
            help='path to validation tests')
    parser.add_argument('-b', '--bin', dest='bin_path', metavar='DIR', default='./',
            help='path to executable files')
    parser.add_argument('-f', '--filter', dest='filter', metavar='FILTER',
            help='filter files to check')
    parser.add_argument('-c', '--compiler', dest='compiler_options', 
            help='comma seperated list of compiler options for rtl', metavar='CARGS')
    parser.add_argument('-q', '--quiet', action='store_false',default=False, dest='quiet',
            help='don\'t print status messages to stdout')
    parser.add_argument('-A', '--ast', action='store_true',default=False, dest='ast',
            help='dump generated AST, do not compile')
    parser.add_argument('-B', '--bytecode', action='store_true',default=False, dest='bytecode',
            help='dump bytecode, do not compile')
    parser.add_argument('-S', '--disassemble', action='store_true',default=False, dest='disassemble',
            help='dump assembly, do not compile')
    parser.add_argument('--logfile', dest='log_file', metavar='LOG', 
            help='validation results are logged to this file')

    return parser.parse_args()


def main():
    global g_logFile
    global g_traceFile
    global g_passedFiles
    global g_failedFiles

    args = parse_args() 

    args.log_file = create_log_filename(args.log_path, args.log_file) 
    g_logFile = make_file(args.log_file)
    g_traceFile = open(args.log_path + '/trace.log', 'w')

    info_msg('run arguments are: {}'.format(args))
    info_msg('starting validation run')

    rtl_files = sorted(find_files(args.tests_path, args.filter, 'rtl'))

    if len(rtl_files) == 0:
        if args.filter == None: 
            error_msg('error: unable to find any rtl files in {}'.format(args.tests_path))
        else:
            error_msg('error: unable to find any rtl files for filter "{}" in {}'.format(
              args.filter, args.tests_path))
        return 1

    #if rtl_files:
    #    print('rtl files are:', rtl_files)
    #if ref_files:
    #    print('ref files are:', ref_files)

    for rtl_file in rtl_files:
        print( '{}...'.format(rtl_file), end='')
        if args.ast:
            if generate_ast(args.bin_path, rtl_file, args.compiler_options):
                compare_file(rtl_file, '.ast') 
        elif args.bytecode:
            if generate_bytecode(args.bin_path, rtl_file, args.compiler_options):
                compare_file(rtl_file, '.bytecode') 
        elif args.disassemble:
            if generate_assemble(args.bin_path, rtl_file, args.compiler_options):
                compare_file(rtl_file, '.S') 
        else:
            jar_file = create_ref_filename(rtl_file, '.jar')
            #pdb.set_trace()
            
            if compile_file(args.bin_path, rtl_file, args.compiler_options):
                if re.search(r'test_.*.rtl', rtl_file) and os.path.isfile(jar_file):
                    run_file(jar_file)
                    compare_file(rtl_file, '.ref')
                elif re.search(r"fail_.*.rtl", rtl_file):
                    compare_file(rtl_file, '.bad')
                else:
                    fail_msg(rtl_file)
                    trace_msg('no jar file created for {}'.format(rtl_file))
                    

    if len(g_failedFiles):
        print('** failed {} tests, passed {} tests **'.format(len(g_failedFiles), len(g_passedFiles)))
    else:
        print('** passed {} tests'.format(len(g_passedFiles)))


    return 0

if __name__ == "__main__":
    exit(main())
