(****************************************************************************
 *
 * File: rtl.ml
 *
 * Purpose: main application 
 *
 *)

type action_type = PrintAst | PrintBC | Compile | Disassemble | Help | Ignore

let show_help () = 
  let msg = "rtl {options} files\n" ^
    "    -h   --help            this message\n" ^ 
    "    -a   --ast             dump abstract syntax tree\n" ^
    "    -b   --bytecode        generate bytecode listing\n" ^
    "    -S   --disassemble     generate disassembly listing\n" 
  in
    print_string msg;
    exit 1 

let action = ref Compile 
let file_count = ref 0
let has_error = ref false

let current_line lb p = 
  if String.contains_from lb.Lexing.lex_buffer p.Lexing.pos_bol '\n' then
    String.sub lb.Lexing.lex_buffer p.Lexing.pos_bol 
      (String.index_from lb.Lexing.lex_buffer p.Lexing.pos_bol '\n' - p.Lexing.pos_bol) 
  else
    String.sub lb.Lexing.lex_buffer p.Lexing.pos_bol 
      (String.length lb.Lexing.lex_buffer - p.Lexing.pos_bol)

let create_error_caret i =
  String.make i ' ' ^ "^"


let () =
  let specs = [
    ('a', "ast",        (Getopt.set action PrintAst), None);
	  ('b', "bytecode",   (Getopt.set action PrintBC), None);
    ('S', "disasemble", (Getopt.set action Disassemble), None);
    ('h', "help",       (Getopt.set action Help), None)
  ] in

  (* 
   * Attempt to open a file
   * 
   * Return: option(channel)
   *)
  let open_file filename = 
    try 
      Some(open_in filename) 
    with Sys_error(msg) -> 
      Printf.fprintf stderr "error: %s\n" msg;
      None
  in

  (*
   * Parse file
   * 
   * Returns: AST
   *)
  let parse_file filename channel = 
    let empty_prog = Ast.Expr(Ast.Noexpr) :: [] in
    let lexbuf = Lexing.from_channel channel in
    
    lexbuf.Lexing.lex_curr_p <- {
      Lexing.pos_fname = filename; 
      Lexing.pos_lnum = 1;
      Lexing.pos_bol = 0; 
      Lexing.pos_cnum = 0
    };

    try
      let program = List.rev (Parser.program Scanner.token lexbuf) in
      close_in channel;
      program

    with e ->
      close_in channel;

      match e with 
      | Parsing.Parse_error ->
        let p = Lexing.lexeme_start_p lexbuf in
        let cline = current_line lexbuf p in  
        let bad_ch = p.Lexing.pos_cnum - p.Lexing.pos_bol in
        let bline = create_error_caret bad_ch in
          Printf.fprintf stderr "%s\n" cline;
          Printf.fprintf stderr "%s\n" bline;
          Printf.fprintf stderr "%s(%d) error: character %d: syntax error.\n"
            p.Lexing.pos_fname p.Lexing.pos_lnum (bad_ch + 1);

          action := Ignore;
          empty_prog
      | _ -> raise e
  in

  let set_exit flag =
    if flag then
      exit 0
    else
      exit 1
  in

  (*
   * Peform *action* on the AST
   *
   * Return: ()
   *)
  let do_action filename program = 
    file_count := !file_count + 1;
    match ! action with
        Help -> show_help ()
      | PrintAst -> 
          Ast.string_of_program program |> List.iter print_endline
      | PrintBC -> 
          Ir.translate program |> Bytecode.string_of_program |> List.iter print_endline
      | Compile -> 
          ignore(Ir.translate program |> Opt_ir.optimize |> Jvm.to_jvm_code |> Opt_jvm.optimize |> 
          Jvm.assemble filename |> Jvm.compile filename |> set_exit)
      | Disassemble ->
          ignore(Ir.translate program |> Jvm.to_jvm_code |> Jvm.assemble filename)
      | Ignore -> ()
  in

  let handle_file filename = 
    match (open_file filename) with
        Some(channel) -> do_action filename (parse_file filename channel)
      | None -> action := Ignore; do_action filename [] 

  in
    Getopt.parse_cmdline specs handle_file; 
    if !file_count = 0 then begin
      print_endline "error: no source files specified\n";
      show_help ()
    end else
      exit 0

