(****************************************************************************
 *
 * File: ir.ml
 *
 * Purpose: convert the abstract syntax tree representation of the program
 * into bytecode.  Syntax checking and type checking are also done at this
 * stage.
 *
 *)

let dump_program = function 
  | Ast.Expr(expr) -> "Expr"
  | Ast.Break -> "Break"
  | Ast.Return(expr) -> "Return"
  | Ast.Block(stmts) -> "Block"
  | Ast.If(e, s1, s2) -> "If" 
  | Ast.For(e1, e2, e3, s) -> "For"
  | Ast.While(e, s) -> "While"
  | Ast.Function(n,vl,rt,b) -> "Function"
  | Ast.FilterFunc(n,b) -> "FilterFunc"
  | Ast.Process(f,p,t) -> "Process"
  | Ast.Declare(v) -> "Declare"
  | Ast.File(n,p,o) -> "File"
  | Ast.Print(el) -> "Print"



(****************************************************************************
 * Get the name of the statement, used for error messages
 *)
let string_of_stmt = function 
  | Ast.Block(_) -> "code block {...}"
  | Ast.Break -> "break"
  | Ast.Expr(_) -> "expression"
  | Ast.Return(_) -> "return"
  | Ast.If(_,_,_) -> "if"
  | Ast.For(_,_,_,_) -> "for"
  | Ast.While(_,_) -> "while"
  | Ast.Function(_,_,_,_) -> "function"
  | Ast.FilterFunc(_,_) -> "filter"
  | Ast.Process(_,_,_) -> "process"
  | Ast.Declare(_) -> "declare"
  | Ast.File(_,_,_) -> "file"
  | Ast.Print(_) -> "print"

let string_of_expr_eval (code, typ) =
  let module BC = Bytecode in
  match typ with 
    | Ast.String -> code
    | Ast.Int -> code @ [BC.Call("rtl/int_to_string(I)Ljava/lang/String;", false)]
    | Ast.Float -> code @ [BC.Call("rtl/float_to_string(F)Ljava/lang/String;", false)]
    | Ast.Char -> code @ [BC.Call("rtl/char_to_string(C)Ljava/lang/String;", false)]
    | _ as typ ->
        (Error.internal_error ("cannot convert a '" ^ Ast.string_of_type typ ^ "' to a string");
        [BC.Halt])

let calc_stack_limit code =
  let module BC = Bytecode in
  let instr_stack_usage = function 
    | BC.Dup -> 1 
    | BC.Pushb(_) -> 1
    | BC.Pushc(_) -> 1
    | BC.Pushf(_) -> 1
    | BC.Pushi(_) -> 1
    | BC.Pushs(_) -> 1
    | BC.Pop -> -1
    | BC.Binop(_,_) -> 2 
    | BC.Cmp(_,_,_) -> 2
    | BC.Load(_,_)  -> 1
    | BC.Store(_,_) -> -1
    | BC.GetGlobal(_,_)  -> 1
    | BC.PutGlobal(_,_) -> -1
    | BC.Call(_,_) -> 0 (* this should use pop values from the stack, but for now ignore that *)
    | BC.CallV(_,_) -> 0 (* this should use pop values from the stack, but for now ignore that *)
    | BC.Bfunc(_,_,_,_,_) -> 0
    | BC.Efunc(_) -> 0
    | BC.Return(_) -> -1
    | BC.Beq(_) -> -1
    | BC.Bne(_) -> -1
    | BC.Jump(_) -> 0 
    | BC.Label(_) -> 0
    | BC.Halt -> 0
    | BC.Comment(_) -> 0
    | BC.Swap -> 0
    | BC.Nop -> 0
  in
  List.fold_left (fun a e -> 
      let v = (fst a) + e in 
      ((v + e), (max (v + e) (snd a)))
      ) (0,0) (List.map instr_stack_usage code) 

let calc_local_limit code = 
  let module BC = Bytecode in
  let local_index = function
    | BC.Load(i,_)  -> i
    | BC.Store(i,_) -> i
    | _ -> 0
  in
  List.fold_left max 0 (List.map local_index code) + 1

(****************************************************************************
 * Keep track of the next lable number, and return it incrementing the 
 * counter.
 *)
let label_cntr = ref 0

let next_label () = 
  label_cntr := succ !label_cntr;
  !label_cntr

(****************************************************************************
 * Return the instruction to push the value onto the stack by var type 
 *)
let push_var_instr typ value = 
  let module BC = Bytecode in
  match typ with
  | Ast.Char    -> BC.Pushc(value)
  | Ast.Float   -> BC.Pushf(value)
  | Ast.Int     -> BC.Pushi(value)
  | Ast.String  -> BC.Pushs(value)

  (* internal types *)
  | Ast.Map     -> 
    Error.internal_error "unable to push type map";
    BC.Halt
  | Ast.Void -> 
    Error.internal_error "unable to push type void";
    BC.Halt
  | Ast.StringArray -> 
    Error.internal_error "unable to push type string[]";
    BC.Halt
  | Ast.FileIn -> 
    Error.internal_error "unable to push type FileIn";
    BC.Halt
  | Ast.FileOut -> 
    Error.internal_error "unable to push type FileOut";
    BC.Halt
  | Ast.Bool    ->
    Error.internal_error "unable to push type bool";
    BC.Halt
      

let int_of_bool_str = function
  | "true" -> 1
  | _ -> 0

let bool_of_bool_str = function
  | "true" -> true
  | _ -> false

(****************************************************************************
 * Return the default value for the variable by type
 *)
let var_default = function
  | Ast.Char -> " "
  | Ast.Float -> "0.0"
  | Ast.Int -> "0"
  | Ast.String -> ""
  
  (* internal types *)
  | Ast.Map -> ""
  | Ast.Bool -> "0"
  | Ast.Void -> ""
  | Ast.StringArray -> ""
  | Ast.FileIn -> ""
  | Ast.FileOut -> ""

let params_to_types params = 
  List.map (fun p -> match p with Ast.Variable(et,_,_) -> et) params
        
let rec has_return code = 
  let module BC = Bytecode in
  match code with
    | [] -> false
    | [BC.Return(_)] -> true
    | hd :: tl -> has_return tl
  

(****************************************************************************
 * Check if the variable has been defined in the local scope
 *)
let is_already_local env name =
  try
    ignore(Symbol.find_local_var env name);
    Error.report ("'" ^ name  ^ "' already defined in local scope");
    true
  with Not_found -> false

(****************************************************************************
 * Add a variable to the symbol table and return the instructions to 
 * store the default value for the variable
 *)
let add_variable_default env typ name = 
  let module BC = Bytecode in
  if is_already_local env name then
    [BC.Halt]
  else
    let value = var_default typ in
    let var = Symbol.add_var env name typ in 
    [push_var_instr var.Symbol.vtype value; BC.store_of name var]

(****************************************************************************
 * Add the variable to the local symbol table and return the instruction 
 * to store the value on the stack 
 *
 *  IMPORTANT: It is expected that the value for the variable is on 
 *  top of the stack.
 *)
let add_variable env typ name = 
  let module BC = Bytecode in
  if env.Symbol.scope > 0 && (is_already_local env name) then
    [BC.Halt]
  else
    let var = Symbol.add_var env name typ in 
    [BC.store_of name var]

(****************************************************************************
 *  Convert the AST (abstract syntax tree) into a stack based 
 *  IR (intermediate representation)
 *)
let translate program =
  let module BC = Bytecode in
  let env = Symbol.create_initial_env () in

  (*
   * Converts the expression into a list of ir codes, and returns the ir code
   * list with the type of the expression
   *
   * Return: ([Bytecode.bstmt], Ast.expr_type)
     *)
  let rec translate_expr lenv = function
    | Ast.Literal_Int(v)    -> ([BC.Pushi(v)], Ast.Int)
    | Ast.Literal_Float(v)  -> ([BC.Pushf(v)], Ast.Float) 
    | Ast.Literal_Char(v)   -> ([BC.Pushc(String.sub v 1 1)], Ast.Char)
    | Ast.Literal_String(v) -> ([BC.Pushs(v)], Ast.String)
    | Ast.Literal_Bool(v)   -> ([BC.Pushi(string_of_int (int_of_bool_str v))], Ast.Bool)

    | Ast.Id(s) -> 
        (try 
          let var = Symbol.find_var lenv s in 
          ([BC.load_of s var], var.Symbol.vtype)
        with Not_found -> 
          Error.report ("'" ^ s ^ "' is undefined");
          ([BC.Halt], Ast.Void))

    (* an Ast.Binop can be either a math op or a comparison *)
    | Ast.Binop(e1,op,e2) -> 
        let expr1 = translate_expr lenv e1 in
        let expr2 = translate_expr lenv e2 in 

        if (snd expr1) != (snd expr2) then
          let type1 = Ast.string_of_type (snd expr1) in
          let type2 = Ast.string_of_type (snd expr2) in 
          let () = Error.report ("cannot convert type '" ^ type1 ^ "' to '" ^ type2 ^ "'") in
          ([BC.Halt], Ast.Void)
        else
          let code = match op with 
            | Ast.Add | Ast.Sub | Ast.Mult | Ast.Div -> BC.Binop(op, (snd expr1))
            | _ -> 
                let () = Error.internal_error "attempt to evaluate Cmp in generation translate_expr" in 
                BC.Cmp(op, Ast.Void, 0) in 
           
          ((fst expr1) @ (fst expr2) @ [code], (snd expr1))
    
    | Ast.Noexpr -> ([], Ast.Void)
    | Ast.Call(name, params) -> begin
        try 
          let func = Symbol.find_function name in
          let params' = List.map (translate_expr lenv) params in 
          let params_code = List.map fst params' in
          let params_type = List.map snd params' in
          let param_code = List.fold_left (@) [] params_code in

          let param_match = try List.for_all2 (=) params_type func.Symbol.args 
            with Invalid_argument(_) -> begin
              false
            end in

          if not param_match then 
            begin 
              Error.report ("number or type of parameters in call to '" ^ name ^ "' don't match");
              ([BC.Halt], Ast.Void)
            end
          else
            (BC.Comment("setting up call to " ^ name) :: 
              param_code @ [BC.Call(name, true)], func.Symbol.rtn)
        with Not_found -> begin
          Error.report ("cannot find function '" ^ name ^ "'");
          ([BC.Halt], Ast.Void)
        end
      end

    (*| Ast.Assign(e1,Ast.Assign(e2,e4)) -> *)
    | Ast.Assign(e1,e2) -> begin
        match e1 with 
          | Ast.Id(s) ->
            begin 
              try
                let var = Symbol.find_var lenv s in 
                let expr2 = translate_expr lenv e2 in

                if var.Symbol.vtype != (snd expr2) then
                  let type1 = Ast.string_of_type var.Symbol.vtype in
                  let type2 = Ast.string_of_type (snd expr2) in 
                  begin
                    Error.report ("cannot convert type '" ^ type1 ^ "' to '" ^ type2 ^ "'");
                    ([BC.Halt], Ast.Void)
                  end
                else
                  (BC.Comment("setting variable " ^ s) :: 
                    (fst expr2) @ [BC.store_of s var], var.Symbol.vtype)
              with Not_found -> 
                Error.report ("'" ^ s ^ "' is undefined");
                ([BC.Halt], Ast.Void)
            end
          | Ast.MapAccess(id,col) -> 
            begin
              try
                let var = Symbol.find_var lenv id in 
                let expr2 = translate_expr lenv e2 in
                let colexpr = translate_expr lenv col in

                if var.Symbol.vtype != Ast.Map then
                  let typ = Ast.string_of_type var.Symbol.vtype in
                  begin
                    Error.report ("cannot convert type '" ^ typ ^ "' to 'record'");
                    ([BC.Halt], Ast.Void)
                  end
                else if (snd colexpr) != Ast.String then
                  let typ = Ast.string_of_type (snd colexpr) in 
                  begin
                    Error.report ("cannot convert type '" ^ typ ^ "' to 'string' for record key");
                    ([BC.Halt], Ast.Void)
                  end
                else if (snd expr2) != Ast.String then
                  let typ = Ast.string_of_type (snd expr2) in 
                  begin
                    Error.report ("cannot convert type '" ^ typ ^ "' to 'string' for record value");
                    ([BC.Halt], Ast.Void)
                  end
                else
                  (BC.Comment("setting map " ^ id) :: 
                    [BC.load_of id var] @ (fst colexpr) @ (fst expr2) @ 
                    [BC.Call("java/util/Map/put(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;", false)], Ast.Map)
              with Not_found -> 
                Error.report ("'" ^ id ^ "' is undefined");
                ([BC.Halt], Ast.Void)
            end
          | Ast.Literal_Int(_) | Ast.Literal_Float(_)| Ast.Literal_String(_) | 
            Ast.Literal_Char(_) | Ast.Literal_Bool(_) ->
            begin
              Error.report "cannot assign a value to a literal";
              ([BC.Halt], Ast.Void)
            end
          | Ast.Binop(_,_,_) ->
            begin
              Error.report "cannot assign a value to the result of an operation";
              ([BC.Halt], Ast.Void)
            end
          | Ast.Call(_,_) ->
            begin
              Error.report "cannot assign a value to the result of a function call";
              ([BC.Halt], Ast.Void)
            end
          | Ast.Noexpr ->
            begin
              Error.report "assigning value to nothing ?";
              ([BC.Halt], Ast.Void)
            end
          | Ast.Assign(_,_) -> 
            begin
              Error.report "assignment chaining error ??";
              ([BC.Halt], Ast.Void)
            end
      end

    | Ast.MapAccess(id,col) -> begin 
        try
          let var = Symbol.find_var lenv id in 
          let colexpr = translate_expr lenv col in

          if var.Symbol.vtype != Ast.Map then
            let typ = Ast.string_of_type var.Symbol.vtype in
            begin
              Error.report ("cannot convert type '" ^ typ ^ "' to 'record'");
              ([BC.Halt], Ast.Void)
            end
          else if (snd colexpr) != Ast.String then
            let typ = Ast.string_of_type (snd colexpr) in 
            begin
              Error.report ("cannot convert type '" ^ typ ^ "' to 'string' for record key");
              ([BC.Halt], Ast.Void)
            end
          else
            (BC.Comment("getting map " ^ id) :: 
              [BC.load_of id var] @ (fst colexpr) @ 
              [BC.Call("java/util/Map/get(Ljava/lang/Object;)Ljava/lang/Object;", false)], Ast.Map)
        with Not_found -> 
          Error.report ("'" ^ id ^ "' is undefined");
          ([BC.Halt], Ast.Void)
      end
  in

  let reverse_cmp_op = function
    | Ast.Equal -> Ast.Neq
    | Ast.Neq -> Ast.Equal
    | Ast.Less -> Ast.Geq
    | Ast.Geq -> Ast.Less
    | Ast.Leq -> Ast.Greater
    | Ast.Greater -> Ast.Leq
    | _ -> begin
        Error.internal_error "attempt to reverse arithmetic op";
        Ast.Equal
      end
  in
        

  (**************************************************************************
   * Generate the code to evaluate the left and right expressions for
   * the comparision, and then do the compare op with a jump supplied 
   * for the false value
   *)
  let translate_cmp lenv label = function
    | Ast.Binop(e1,op,e2) ->
        let expr1 = translate_expr lenv e1 in
        let expr2 = translate_expr lenv e2 in 
        if (snd expr1) != (snd expr2) then
          let type1 = Ast.string_of_type (snd expr1) in
          let type2 = Ast.string_of_type (snd expr2) in 
          let () = Error.report ("cannot convert type '" ^ type1 ^ "' to '" ^ type2 ^ "'") in
          ([BC.Halt], Ast.Void)
        else
          (match op with 
            (* error message for invalid bin op not generated here *)
            | Ast.Add | Ast.Sub | Ast.Mult | Ast.Div -> 
              ([BC.Halt], Ast.Void)
            | _ ->  
              let code = (fst expr1) @ (fst expr2) @ [BC.Cmp(reverse_cmp_op op, (snd expr1), label)] in
              (code, Ast.Bool))

    | Ast.Literal_Bool(value) -> 
        if bool_of_bool_str value then
          ([BC.Jump(label)], Ast.Bool)
        else 
          ([], Ast.Void)

    | _ as expr -> 
      let () = Error.internal_error ("attempting to translate binop when expr is a '" ^ 
        (Ast.string_of_expr expr) ^ "'") in
      ([BC.Halt], Ast.Void)
  in
    
  let translate_cmp_rev lenv label = function
    | Ast.Binop(e1,op,e2) ->
        let expr1 = translate_expr lenv e1 in
        let expr2 = translate_expr lenv e2 in 
        if (snd expr1) != (snd expr2) then
          let type1 = Ast.string_of_type (snd expr1) in
          let type2 = Ast.string_of_type (snd expr2) in 
          let () = Error.report ("cannot convert type '" ^ type1 ^ "' to '" ^ type2 ^ "'") in
          ([BC.Halt], Ast.Void)
        else
          (match op with 
            (* error message for invalid bin op not generated here *)
            | Ast.Add | Ast.Sub | Ast.Mult | Ast.Div -> 
              ([BC.Halt], Ast.Void)
            | _ ->  
              let code = (fst expr1) @ (fst expr2) @ [BC.Cmp(op, (snd expr1), label)] in
              (code, Ast.Bool))

    | _ as expr -> 
      let () = Error.internal_error ("attempting to translate binop when expr is a '" ^ 
        (Ast.string_of_expr expr) ^ "'") in
      ([BC.Halt], Ast.Void)
  in

  let rec translate_stmt lenv = function
    | Ast.For(e1, e2, e3, s) -> 
        let label1 = next_label () in
        let label2 = next_label () in
        let label3 = next_label () in
        let expr1 = translate_expr lenv e1 in
        let expr2 = translate_cmp lenv label1 e2 in
        let expr3 = translate_expr lenv e3 in
        let nenv = Symbol.push_env lenv label3 lenv.Symbol.address lenv.Symbol.rtype in
        let stmts = translate_stmt nenv s in 
        let () = Symbol.pop_env nenv in 

        if (snd expr2) != Ast.Bool then
          begin
            Error.report " test expresion in 'while' statement must evaluate to a 'boolean'";
            [BC.Halt]
          end
        else  
          [BC.Comment("begin while loop")] @ (fst expr1) @ [BC.Jump(label2);BC.Label(label1)] @ stmts @ 
            (fst expr3) @ [BC.Label(label2)] @ (fst expr2) @ [BC.Label(label3); BC.Comment("end while loop")] 

    | Ast.While(e, s) -> 
        let label1 = next_label () in
        let label2 = next_label () in
        let label3 = next_label () in
        let expr = translate_cmp lenv label1 e in
        let nenv = Symbol.push_env lenv label3 lenv.Symbol.address lenv.Symbol.rtype in
        let stmts = translate_stmt nenv s in
        let () = Symbol.pop_env nenv in 

        if (snd expr) != Ast.Bool then
          begin
            Error.report "expression for 'while' statement must evaluate to a 'boolean'";
            [BC.Halt]
          end
        else  
          [BC.Comment("begin while loop"); 
           BC.Jump(label2);
           BC.Label(label1)] 
          @ stmts @ 
          [BC.Label(label2)]  @ (fst expr) @ [BC.Label(label3); BC.Comment("end while loop")]

    | Ast.If(e, s1, s2) -> 
        let label1 = next_label () in
        let label2 = next_label () in
        let expr = translate_cmp lenv label1 e in
        let nenv = Symbol.push_env lenv lenv.Symbol.label lenv.Symbol.address lenv.Symbol.rtype in
        let stmts1 = translate_stmt nenv s1 in
        let () = Symbol.pop_env nenv in 
        let nenv = Symbol.push_env lenv lenv.Symbol.label lenv.Symbol.address lenv.Symbol.rtype in
        let stmts2 = translate_stmt nenv s2 in
        let () = Symbol.pop_env nenv in 
       
        if (snd expr) != Ast.Bool then
          let () = Error.report "expression for 'if' statement must evaluate to a 'boolean'" in
          [BC.Halt]
        else  
          (fst expr) @ stmts1 @ 
            [BC.Jump(label2); BC.Label(label1)] @ stmts2 @ [BC.Label(label2)] 

    | Ast.Function(name, params, rtn_type, block) ->
        let lenv = Symbol.push_env lenv 0 0 rtn_type in
        let () =  
          ignore(List.map (fun elt -> 
            match elt with 
              | Ast.Variable(t,n,e) -> begin
                if is_already_local lenv n then
                  Error.report ("function parameter '" ^ n ^ "' already defined in '" ^ name ^ "'")
                else 
                  ignore(Symbol.add_local_var lenv n t)
                end
            ) params)
        in
        let cnt = List.length params in
        let block_code = translate_stmt lenv block in 
        let stack_limit = snd (calc_stack_limit block_code) + 2 in
        let local_limit = max (calc_local_limit block_code) cnt in

        ignore(Symbol.pop_env lenv);

        BC.Bfunc(name, (params_to_types params), rtn_type, stack_limit, local_limit) :: block_code @
        
        if has_return block_code then 
          [BC.Efunc(name)]
        else
          [(BC.push_of rtn_type); BC.Return(rtn_type); BC.Efunc(name)]

    | Ast.FilterFunc(name, stmts) -> 
      let func = Ast.Function(name, [Ast.Variable(Ast.Map, "rec", Ast.Noexpr)], Ast.Bool, stmts) in
      translate_stmt lenv func 
    | Ast.Declare(var) -> begin 
        try 
          trans_declare lenv var
        with Symbol.Duplicate_id(id) ->
          Error.report ("a variable named " ^ id ^ " already exists");
          [BC.Halt]
      end
    | Ast.Expr(expr) ->
      (match expr with 
      | Ast.Assign(_, _) | Ast.Call(_, _) | Ast.MapAccess(_, _) ->
        fst (translate_expr lenv expr)
      | _ -> 
          let () = Error.report ("'" ^ (Ast.string_of_expr expr) ^ "' is not a valid statement") in
          [BC.Halt])

    | Ast.Block(stmt_lst) -> trans_block lenv stmt_lst
    | Ast.Print(expr) -> trans_print lenv expr 

    | Ast.Return(expr) -> 
      let ecode = translate_expr lenv expr in 
      if snd ecode != lenv.Symbol.rtype then
        begin
          let type1 = Ast.string_of_type (snd ecode) in
          let type2 = Ast.string_of_type lenv.Symbol.rtype in 
          Error.report ("return type of '" ^ type1 ^ "' does not match function return type of '" ^ type2 ^ "'");
          [BC.Halt]
        end
      else
        (fst ecode) @ [BC.Return(snd ecode)]
  
    | Ast.File(name, path, opt) -> 
      begin 
        try 
          ignore(Symbol.find_var lenv name);
          Error.stmt_error ("'" ^ name ^ "' is already defined");
          [BC.Halt]
        with Not_found -> begin 
          let typ = 
            List.fold_left (fun acc elt -> if elt = Ast.Output then Ast.FileOut else acc) Ast.FileIn opt in
          let sep = 
            List.fold_left (fun acc elt -> match elt with Ast.Sep(s) -> s | _ -> acc) "\",\"" opt in
          let var = Symbol.add_var lenv name typ in
          let call = 
            if typ == Ast.FileIn then 
              [BC.Call("Rtl/create_in(Ljava/lang/String;Ljava/lang/String;)LRtlInFile;", false)]
            else 
              [BC.Call("Rtl/create_out(Ljava/lang/String;Ljava/lang/String;)LRtlOutFile;", false)]
          in
          [BC.Pushs(path);BC.Pushs(sep)] @ call @ [BC.store_of name var]
        end
      end

    | Ast.Process(fromf, steps, tof) ->
        begin
        try let f_var = Symbol.find_var lenv fromf in
        try let t_var = Symbol.find_var lenv tof in
          begin
            let label1 = next_label () in
            let label2 = next_label () in
            let label3 = next_label () in
            let map_name = "__map_" ^ (string_of_int (next_label ())) in
            let (map_var : Symbol.sym_var) = (Symbol.add_local_var lenv map_name Ast.Map) in 
            let nenv = Symbol.push_env lenv label1 lenv.Symbol.address lenv.Symbol.rtype in
            let () = ignore(nenv.Symbol.rec_address <- map_var.Symbol.address) in
            let step_code = List.map (translate_process_step nenv map_name map_var) steps in 
            let step_code' = List.fold_left (fun acc elt -> elt @ acc) [] (List.rev step_code) in 
            let () = Symbol.pop_env nenv in

            [
             BC.Label(label1);
             BC.load_of fromf f_var;
             BC.CallV("RtlInFile/read()Ljava/util/Map;", false);
             BC.Dup;
             BC.store_of map_name map_var;
             BC.Cmp(Ast.NotNull, Ast.FileIn, label2);
             BC.Jump(label3);
             BC.Label(label2)
            ]
            
            @ step_code' @ 
             
            [
             BC.load_of fromf f_var;
             BC.CallV("RtlInFile/getHeader()[Ljava/util/String;",false);
             BC.load_of map_name map_var;
             BC.load_of tof t_var;
             BC.CallV("RtlOutFile/write([Ljava/lang/String;Ljava/util/Map;)V;",false);
             BC.Jump(label1);
             BC.Label(label3);
            ]

          end
        with Not_found -> 
          (Error.stmt_error ("process from file '" ^ fromf ^ "' is not defined");
          [BC.Halt])
        with Not_found -> 
          (Error.stmt_error ("process from file '" ^ tof ^ "' is not defined");
          [BC.Halt])
        end


    | Ast.Break ->
        if lenv.Symbol.label = -1 then
          begin
            Error.stmt_error "break statement used in non-loop  context";
            [BC.Halt]
          end
        else
          [BC.Jump(lenv.Symbol.label)]

  and translate_process_step lenv map_name map_var = function
    (* Call the function for filter processing *)
    | Ast.Check_Function(id) -> 
        if lenv.Symbol.rec_address = -1 then
          begin
            Error.internal_error "Check_Function statement has no rec label";
            [BC.Halt]
          end
        else 
          begin
            [BC.Load(lenv.Symbol.rec_address, Ast.Map);
             BC.Call(id, true);
             BC.Cmp(Ast.Equal, Ast.Bool, lenv.Symbol.label)]
          end


  and trans_block lenv stmt_lst = 
    let local_env = Symbol.push_env lenv lenv.Symbol.label lenv.Symbol.address lenv.Symbol.rtype in
    let code = List.map (translate_stmt local_env) stmt_lst in 
    let () = Symbol.pop_env local_env in
    List.fold_left (fun acc elt -> elt @ acc) [] (List.rev code)

  and trans_declare lenv = function
    | Ast.Variable(typ, name, Ast.Noexpr) -> add_variable_default lenv typ name
    | Ast.Variable(typ, name, e1) -> 
        let expr = translate_expr lenv e1 in 
        if typ != (snd expr) then
          let type1 = Ast.string_of_type typ in
          let type2 = Ast.string_of_type (snd expr) in
          Error.stmt_error ("cannot convert type '" ^ type1 ^ "' to '" ^ type2 ^ "'");
          [BC.Halt]
        else
          (fst expr) @ (add_variable lenv typ name) 
          
  and trans_print lenv expr = 
    let module BC = Bytecode in 
    (* create formatted string to pass to print *)
    let (expr_code, typ) = translate_expr lenv expr in
    let setup = match typ with
      | Ast.String -> expr_code 
      | Ast.Int -> expr_code @ [BC.Call("Rtl/int_to_string(I)Ljava/lang/String;", false)]
      | Ast.Char -> expr_code @ [BC.Call("Rtl/char_to_string(C)Ljava/lang/String;", false)]
      | Ast.Float -> expr_code @ [BC.Call("Rtl/float_to_string(F)Ljava/lang/String;", false)]
      | _ -> begin
        Error.report ("cannot print type '" ^ (Ast.string_of_type typ) ^ "'");
        [BC.Halt]
      end in

     setup @ [BC.Call("Rtl/print(Ljava/lang/String;)V", false)]
  in

  let list_globals () = 
    Hashtbl.fold (fun k v acc -> (k,v) :: acc) Symbol.global_vars []
  in

  let extract_funcs acc = function
    | Ast.Function(n,p,r,_) as fb -> begin 
        try 
          Symbol.add_func n (Ast.var_list_to_type_list p) r
        with Symbol.Duplicate_id(id) -> 
          Error.report ("a function named " ^ id ^ " already exists")
        end;
        fb :: acc
    | Ast.FilterFunc(n,_) as ff -> begin
        try 
          Symbol.add_func n [Ast.Map] Ast.Bool
        with Symbol.Duplicate_id(id) -> 
          Error.report ("a function named " ^ id ^ " already exists")
        end;
        ff :: acc
    | _ -> acc
  in


  (*let prog_stmt = List.map dump_program program in*)
  let func_blocks = List.rev (List.fold_left extract_funcs [] program) in

  (*
   * Remove all functions statements
   *)
  let main_stmt_list = List.fold_left 
    (fun acc elt -> 
      match elt with 
        | Ast.Function(_,_,_,_) | Ast.FilterFunc(_,_) -> acc
        | _ as stmt -> stmt :: acc) [] program in

  let main_arg = Ast.Variable(Ast.StringArray,"args",Ast.Noexpr) in
  let main_func = [Ast.Function("main", [main_arg],Ast.Void,Ast.Block(List.rev main_stmt_list))] in

  (*  The environment tuple = use globals * symbol tables *)
  let main_code = List.map (translate_stmt env) main_func in
  let func_code = List.map (translate_stmt (Symbol.push_env env 0 0 env.Symbol.rtype)) func_blocks in
  let globals = list_globals () in

  let ir = List.fold_left (fun acc elt -> elt @ acc) [] (List.rev main_code @ func_code) in 

(*
  let dump = String.concat "\n" prog_stmt in
  print_endline dump;
*)

  (ir, globals)

