(****************************************************************************
 *
 * File: bytecode.ml
 *
 * Purpose: defines the intermediate representation operations and serveral
 * utility functions.
 *
 *)

type bstmt =
  | Dup
  | Swap
  | Pushb of string (* Push a literal boolean *)
  | Pushc of string (* Push a literal char *)
  | Pushf of string (* Push a literal float *)
  | Pushs of string (* Push a literal string index *)
  | Pushi of string (* Push a literal int *)
  | Pop             (* Discard a value *)
  | Binop of Ast.op * Ast.expr_type (* Perform arithmetic on top of stack *)
  | Cmp of Ast.op * Ast.expr_type * int   (* Perform comparison on top of stack *)
  | Load of int * Ast.expr_type
  | Store of int * Ast.expr_type
  | Call of string * bool (* bool is flag to signify need for fixup *)
  | CallV of string * bool (* bool is flag to signify need for fixup *)

    (* name, param list, return type, stack size, local var count *)
  | Bfunc of string * Ast.expr_type list * Ast.expr_type * int * int
  | Efunc of string 
  | Return of Ast.expr_type 
  | Beq of int      (* Branch relative if top-of-stack is zero *)
  | Bne of int      (* Branch relative if top-of-stack is non-zero *)
  | Jump of int     (* Branch relative *) 
  | Label of int    (* Jump/Branch to location *)
  | Halt            (* Terminate *)
  | Comment of string (* Comment *)
  | Nop
  | GetGlobal of string * Ast.expr_type
  | PutGlobal of string * Ast.expr_type

type field =
  | Int of string * string 
  | Char of string * string   
  | Float of string * string
  | String of string * string
  | FileIn of string * string
  | FileOut of string * string

let store_of name (var : Symbol.sym_var) =
  if var.Symbol.scope = 0 then
    PutGlobal(name, var.Symbol.vtype)
  else 
    Store(var.Symbol.address, var.Symbol.vtype)
      
let load_of name (var : Symbol.sym_var) =
  if var.Symbol.scope = 0 then
    GetGlobal(name, var.Symbol.vtype)
  else 
    Load(var.Symbol.address, var.Symbol.vtype)

let push_of = function
  | Ast.Bool -> Pushb("false")
  | Ast.Char -> Pushc(" ")
  | Ast.Float -> Pushf("0.0")
  | Ast.Int -> Pushi("0") 
  | Ast.Map -> Halt  (* not supported *)
  | Ast.String -> Pushs("")
  (* internal types *)
  | Ast.Void -> Nop
  | Ast.StringArray -> Nop
  | Ast.FileIn -> Nop
  | Ast.FileOut -> Nop


let string_of_instr = function
  | Dup -> "dup"
  | Swap -> "swap"
  | Pushb(s) -> "pushb " ^ s
  | Pushc(s) -> "pushc " ^ s 
  | Pushf(s) -> "pushf " ^ s
  | Pushs(s) -> "pushs " ^ s
  | Pushi(s) -> "pushi " ^ s
  | Pop -> "pop"
  | Binop(Ast.Add, typ) -> "add"
  | Binop(Ast.Sub, typ) -> "sub"
  | Binop(Ast.Mult, typ) -> "mul"
  | Binop(Ast.Div, typ) -> "div" 
  | Binop(op, _) -> "internal error, Binop used with unsupported op " ^ Ast.string_of_op op
  | Cmp(Ast.Equal, typ, l) -> "cmpEq " ^ string_of_int l 
  | Cmp(Ast.Neq, typ, l) -> "cmpNeq " ^ string_of_int l 
  | Cmp(Ast.Less, typ, l) -> "cmpLt " ^ string_of_int l
  | Cmp(Ast.Leq, typ, l) -> "cmpLeq " ^ string_of_int l
  | Cmp(Ast.Greater, typ, l) -> "cmpGt " ^ string_of_int l
  | Cmp(Ast.Geq, typ, l) -> "cmpGeq " ^ string_of_int l
  | Cmp(Ast.NotNull, typ, l) -> "notNull " ^ string_of_int l
  | Cmp(op, typ, _) -> "internal error, Cmp used with unsupported op " ^ Ast.string_of_op op
  | Load(i, _) -> "load " ^ string_of_int i
  | Store(i, _) -> "store " ^ string_of_int i
  | PutGlobal(name, typ) -> "putglobal " ^ name ^ ", " ^ Ast.string_of_type typ
  | GetGlobal(name, typ) -> "getglobal " ^ name ^ ", " ^ Ast.string_of_type typ
  | Call(s,_) -> "callstatic " ^ s
  | CallV(s,_) -> "callvirtual " ^ s
  | Bfunc(n,p,r,s,l) -> 
      Printf.sprintf "function %s(%s) %s <stack=%d, locals=%d>" n 
        (Ast.string_of_type_list p) (Ast.string_of_type r) s l
  | Efunc(s) -> "end " ^ s ^ "\n"
  | Return(typ) -> "return " ^ Ast.string_of_type typ
  | Bne(i) -> "bne L" ^ string_of_int i
  | Beq(i) -> "beq L" ^ string_of_int i
  | Jump(i) -> "jump L" ^ string_of_int i
  | Label(i) -> "L" ^ string_of_int i ^ ":"
  | Halt    -> "halt"
  | Comment(s) -> "; " ^ s
  | Nop -> "Nop"

let string_of_program (ir, globals) =
  List.map (fun instr -> "\t" ^ string_of_instr instr) ir

