(****************************************************************************
 *
 * File: ast.ml
 *
 * Purpose: defines the objects in the abstract syntax tree.  Also provides
 * utility methods to convert the ast objects into a string representation.
 *
 *)

type op = Add | Sub | Mult | Div | Equal | Neq | Less | Leq | Greater | Geq | LAnd | LOr | NotNull

type expr =
    Literal_Int of string
  | Literal_Float of string
  | Literal_String of string
  | Literal_Char of string
  | Literal_Bool of string
  | Id of string
  | Binop of expr * op * expr
  | Assign of expr * expr
  | Call of string * expr list
  | MapAccess of string * expr
  | Noexpr

and map_entry = MapEntry of expr * expr
and file_options = Sep of string | Output    

type expr_type = Bool | Char | Float | Int | Map | String | Void | StringArray | FileIn | FileOut 

(* Data Type * ID * value *)
type variable = 
    Variable of expr_type * string * expr

type process_steps = Check_Function of string

and stmt =
    Block of stmt list
  | Break
  | Expr of expr
  | Return of expr
  | If of expr * stmt * stmt
  | For of expr * expr * expr * stmt
  | While of expr * stmt
  | Function of string * variable list * expr_type * stmt 
  | FilterFunc of string * stmt 
  | Process of string * process_steps list * string 
  | Declare of variable
  | File of string * string * file_options list
  | Print of expr 

type program = stmt list 

let var_list_to_type_list lst =
  let extract_type = function
    | Variable(Char, n, _) -> Char
    | Variable(Float, n, _) -> Float
    | Variable(Int, n, _) -> Int
    | Variable(String, n, _) -> String
    | Variable(Bool, n, _) -> Bool
    | Variable(Map, n, _) -> Map
    | Variable(Void, n, _) -> Void
    | Variable(StringArray, n, _) -> StringArray
    | Variable(FileIn, n, _) -> FileIn
    | Variable(FileOut, n, _) -> FileOut
  in
  List.map extract_type lst


let string_of_op = function
    Add     -> "+" 
  | Sub     -> "-"
  | Mult    -> "*" 
  | Div     -> "/"
  | Equal   -> "==" 
  | Neq     -> "!="
  | Less    -> "<" 
  | Leq     -> "<=" 
  | Greater -> ">" 
  | Geq     -> ">="
  | LAnd    -> "&&"
  | LOr     -> "||"
  | NotNull -> "notnull"

let string_of_type = function
  | Char -> "char"
  | Float -> "float"
  | Int -> "int"
  | String -> "string"
  
  (* internal types *)
  | Bool -> "bool"
  | Map -> "map"
  | Void -> "void"
  | StringArray -> "String[]"
  | FileIn -> "FileIn"
  | FileOut -> "FileOut"
  
let string_of_type_list l = 
  String.concat ", " (List.map string_of_type l)


let string_of_formal = function
  | Variable(Char, n, _) -> "char " ^ n
  | Variable(Float, n, _) -> "float " ^ n
  | Variable(Int, n, _) -> "int " ^ n
  | Variable(String, n, _) -> "string " ^ n  
  | Variable(Bool, n, _) -> "string " ^ n  

  (* internal types, not used in rtl*)
  | Variable(Map, n, _) -> "internal-type-map " ^ n
  | Variable(Void, n, _) -> "internal-type-void " ^ n  
  | Variable(StringArray, n, _) -> "internal-type-string[] " ^ n  
  | Variable(FileIn, n, _) -> "internal-type-filein " ^ n  
  | Variable(FileOut, n, _) -> "internal-type-fileout " ^ n  

let string_of_formal_list l = 
  String.concat ", " (List.map string_of_formal l)

let rec string_of_expr = function
    Literal_Int(l) -> l
  | Literal_Float(l) -> l
  | Literal_String(l) -> l
  | Literal_Char(l) -> l
  | Literal_Bool(l) -> l
  | Id(s) -> s
  | Binop(lhs, o, rhs) -> string_of_expr lhs ^ " " ^ string_of_op o ^ " " ^ string_of_expr rhs
  | Assign(lhs, rhs) -> string_of_expr lhs ^ " = " ^ string_of_expr rhs
  | Call(f, el) -> f ^ "(" ^ String.concat ", " (List.map string_of_expr el) ^ ")"
  | MapAccess(name, prop) -> name ^ "[" ^ string_of_expr prop ^ "]"
  | Noexpr -> ""

let string_of_expr_list l = 
  String.concat ", " (List.map string_of_expr l)

let string_of_declare = function 
    Variable(Bool, n, Noexpr) -> "bool " ^ n ^ ";"
  | Variable(Bool, n, e) -> "bool " ^ n ^ " = " ^ string_of_expr e ^ ";"
  | Variable(Char, n, Noexpr) -> "char " ^ n ^ ";"
  | Variable(Char, n, e) -> "char " ^ n ^ " = " ^ string_of_expr e ^ ";"
  | Variable(Float, n, Noexpr) -> "float " ^ n ^ ";ocaml toplevel load file"
  | Variable(Float, n, e) -> "float " ^ n ^ " = " ^ string_of_expr e ^ ";"
  | Variable(Int, n, Noexpr) -> "int " ^ n ^ ";"
  | Variable(Int, n, e) -> "int " ^ n ^ " = " ^ string_of_expr e ^ ";"
  | Variable(String, n, Noexpr) -> "string " ^ n ^ ";"
  | Variable(String, n, e) -> "string " ^ n ^ " = " ^ string_of_expr e ^ ";"
  | Variable(Void,_,_) -> raise (Invalid_argument("Void type not supported"))
  | Variable(Map,_,_) -> raise (Invalid_argument("Map type not supported"))
  | Variable(StringArray,_,_) -> raise (Invalid_argument("String[] type not supported"))
  | Variable(FileIn,_,_) -> raise (Invalid_argument("File type not supported"))
  | Variable(FileOut,_,_) -> raise (Invalid_argument("File type not supported"))
  (*
  | Variable(Map, n, Noexpr) -> "map " ^ n ^ ";"
  | Variable(Map, n, e) -> "map " ^ n ^ " = " ^ string_of_expr e ^ ";"
  *)
let string_of_file_option = function
    Output -> "Output"
  | Sep(s) -> "Sep(" ^ s ^ ")"

let rec string_of_file_options = function
    [] -> ""
  | [v] -> " " ^ string_of_file_option v
  | hd :: tl -> " " ^ string_of_file_option hd ^ string_of_file_options tl 


let rec string_of_stmt = function
    Expr(expr) -> "Expr(" ^ string_of_expr expr ^ ")"
  | Break -> "Break"
  | Return(expr) -> "Return(" ^ string_of_expr expr ^ ")";
  | Block(stmts) -> "Block(" ^ string_of_stmt_list stmts ^ ")"
  | If(e, s1, s2) -> "If(" ^ string_of_expr e ^ ", " ^ string_of_stmt s1 ^ ", " ^ string_of_stmt s2 ^ ")" 
  | For(e1, e2, e3, s) ->   
      "For(" ^ string_of_expr e1  ^ ", " ^ string_of_expr e2 ^ ", " ^ string_of_expr e3 ^ ", " ^ string_of_stmt s ^ ")"
  | While(e, s) -> "While(" ^ string_of_expr e ^ ", " ^ string_of_stmt s ^ ")"
  | Function(n,vl,rt,b) -> 
      "Function(" ^ n ^ ", " ^ string_of_formal_list vl ^ ", " ^ string_of_type rt ^ ", " ^ string_of_stmt b ^ ")"
  | FilterFunc(n,b) -> "Filter(" ^ n ^ ", " ^ string_of_stmt b ^ ")"
  | Process(f,p,t) -> "Process(" ^ f ^ ", " ^ string_of_process_steps p ^ ", " ^ t ^ ")"
  | Declare(v) -> "Declare(" ^ string_of_declare v ^ ")"
  | File(n,p,o) -> "File(" ^ n ^ ", " ^ p ^ ", [" ^ string_of_file_options o ^ "])"
  | Print(expr) -> "Print(" ^ string_of_expr expr ^ ")"

and string_of_process_step = function
  | Check_Function(n) -> "Check_Function(" ^ n ^ ")"
  
and string_of_process_steps l = 
  "[" ^ String.concat ", " (List.map string_of_process_step l) ^ "]"
  
and string_of_stmt_list stmts = 
  "[" ^ String.concat ", " (List.map string_of_stmt stmts) ^ "]"
  
let string_of_program stmt_list  =
  List.map string_of_stmt stmt_list



