{ 
    open Parser
    open Char
    open Exception
}

let space = [' ' '\t' '\r' '\n']

rule token = parse 
| space                      { token lexbuf }
| "~~>"                      { comment lexbuf }
| "~~"                       { line_comment lexbuf }

| "defun"                    { DEFUN }
| "defvar"                   { DEFVAR }

| "int"                      { INT }
| "bit#" (['0'-'9']+ as lit) { BITS(int_of_string lit) }
| "string"                   { STR }
| "fun"                      { FUN }

| "let"                      { LET }
| "lambda"                   { LAMBDA }
| "make-vector"              { MAKE_VECTOR }

(* Prevent consecutive -- in identifier *)
| "--" as c                  { raise (Illegal_Id(c)) }
| ['A'-'Z' 'a'-'z'] ['A'-'Z' 'a'-'z' '0'-'9' '-']* as lit   { ID(lit) }

| "0b"(['0' '1']+ as lit)                 { BINARY(lit)}
| "0x"(['0'-'9' 'A'-'F' 'a'-'f']+ as lit) { HEX(lit)}
| "'"(['0' '1']+ as lit)                  { BIT_BINARY(lit)}
| "'x"(['0'-'9' 'A'-'F' 'a'-'f']+ as lit) { BIT_HEX(lit)}
| ['0'-'9']+ as lit             { INTEGER(lit) }
| '"' ([^ '"']* as content) '"'           { STRING(content) }

(* Special identifier used by built-in functions *)
| ">>>" as lit                            { ID(lit) }
| "<<<" as lit                            { ID(lit) }
| ">>"  as lit                            { ID(lit) }
| "<<"  as lit                            { ID(lit) }
| "&"   as lit                            { ID(escaped lit) }
| "|"   as lit                            { ID(escaped lit) } 
| "^"   as lit                            { ID(escaped lit) }
| "+"   as lit                            { ID(escaped lit) }
| "-"   as lit                            { ID(escaped lit) }
| "*"   as lit                            { ID(escaped lit) }
| "/"   as lit                            { ID(escaped lit) }

| '='                                  { ASSIGN }
| '@'(['0'-'9']+ as lit)               { VECDIMENSION(int_of_string lit) }

| ':'                     { COLON }
| ';'                     { SEMI }
| '('                     { LPAREN }
| ')'                     { RPAREN }
| '{'                     { LBRACE }
| '}'                     { RBRACE }
| '<'                     { LANGLE }
| '>'                     { RANGLE }
| '['                     { LBRACK }
| ']'                     { RBRACK }

| eof                     { EOF }
| _ as c                  { raise (Illegal_Char(c)) }
and comment = parse
| "<~~"                   { token lexbuf }
| _                       { comment lexbuf }
and line_comment = parse
| ['\n' '\r'] { token lexbuf }
| eof         { token lexbuf }
| _           { line_comment lexbuf }