%{
  open Ast
%}

%token DEFUN DEFVAR
%token INT STR
%token <int> BITS
%token FUN

%token LET
%token LAMBDA
%token MAKE_VECTOR

%token <string> ID

%token <string> BINARY
%token <string> HEX
%token <string> BIT_BINARY
%token <string> BIT_HEX
%token <string> INTEGER
%token <string> STRING

%token COLON SEMI LPAREN RPAREN LBRACK RBRACK LBRACE RBRACE LANGLE RANGLE  
%token ASSIGN
%token <int> VECDIMENSION
%token EOF

%start program
%type <Ast.program> program

%%

program:
| /* nothing */    { [] }
| clip_list        { List.rev $1 }

clip_list:
| clip             { [$1] } 
| clip_list clip   { $2 :: $1 }

clip:
| defvar           { $1 }
| defun            { $1 }
| expr             { Expr($1, true) }
| expr SEMI        { Expr($1, false) }

defvar:
| DEFVAR id_with_type ASSIGN expr SEMI    { Defvar ({ vname = $2;
                                                      vbody = $4 } )}

defun:
| DEFUN id_with_type arguments_opt ASSIGN expr SEMI { Defun ({ fname = $2;
                                                             fargu = $3;
                                                             fbody = $5 } )}

expr:
| constant                                { $1 }
| ID                                      { Id($1) }
| ID dimension_list                       { Idd($1, List.rev $2) }
| VECDIMENSION                            { Vec_Dimension($1) }
| LPAREN LET let_args expr RPAREN         { Let(List.rev $3, $4) }
| LPAREN LAMBDA lambda_arg expr RPAREN    { Lambda($3, $4) }
| LPAREN MAKE_VECTOR LANGLE c_type RANGLE expr RPAREN { Make_Vector($4, $6) }
| LPAREN expr expr_opt RPAREN             { Funcall($2, $3) }

lambda_arg:
| LANGLE arguments_opt RANGLE     { $2 }

let_args:
| let_arg                         { [$1] }
| let_args let_arg                { $2 :: $1 }

let_arg:
LANGLE id_with_type expr RANGLE   { ($2, $3) }

expr_opt:
| /* nothing */                   { [] }
| expr_list                       { List.rev $1 }

expr_list:
| expr                            { [$1] }
| expr_list expr                  { $2 :: $1 }

arguments_opt:
| /* nothing */                   { [] }
| argument_list                   { List.rev $1 }

argument_list:
| id_with_type                    { [$1] }
| argument_list id_with_type      { $2 :: $1 }

id_with_type:
| ID COLON c_type                 { { id = $1; t = $3 } }

/* Returns Type(basic_type, dimensions)
   e.g. parse int[8][5][4] as Vector(INT, [8, 5, 4])
              int          as INT
              bits#7[6][5] as Vector(Bits(7), [6, 5])
*/

c_type:
| sig_type                              { $1 }
| sig_type const_dimension_list         { Vector($1, List.rev $2) }
| FUN                                   { Fun(Indef, Wild_Card(0), []) }

sig_type:
| INT                                   { Int }
| BITS                                  { Bits($1) }
| STR                                   { String }

const_dimension_list:
| const_dimension                       { [$1] }
| const_dimension_list const_dimension  { $2::$1 }

const_dimension:
| LBRACK INTEGER RBRACK              { (int_of_string $2) }

dimension_list:
| dimension                       { [$1] }
| dimension_list dimension        { $2::$1 }

dimension:
| LBRACK expr RBRACK              { $2 }

constant:
| constant_int    { $1 }
| constant_bits   { $1 }
| STRING          { String_Lit($1) }
| vector          { Vector_Lit($1) }

vector:
| LBRACE expr_list RBRACE  { List.rev $2 }

constant_int:
| INTEGER         { Int_Lit($1) }
| BINARY          { Bin_Lit($1) }
| HEX             { Hex_Lit($1) }

constant_bits:
| BIT_BINARY      { Bit_Binary_Lit($1) }
| BIT_HEX         { Bit_Hex_Lit($1) }  /* not yet handled */