open Ast

let program_func_list = [];;

let variable_scope = [];;

let variable_scope_ref = ref [("stuff","stuff")];;

let rec get_id_type v_id= function
    [] -> raise(Failure("Variable not declared: " ^ v_id))
    | (dt,id)::v_scope -> if ( id=v_id) then dt else get_id_type v_id v_scope

let offlimits = ["main"; "grid_size"; "set_actor"; "set_chronon"; "set_grid_pattern"; "set_grid_random"; "read_grid"; "cellat"; "assign_type"; "move"; "neighborhood"; "randomof"; "update"; "colorblock"; "set_cell_size"];;

let exists k l =
      List.fold_left ( fun b x -> b || x = k ) false l

let get_id_type_string = function
       _ -> "BVal"

let rec get_name_of_functions start_list= function
       [] -> start_list
       | ActorType(f)::f_list -> get_name_of_functions (f.aname::start_list) f_list
       | CFunc(f)::f_list -> get_name_of_functions (f.fname::start_list) f_list

let rec get_name_of_variables start_list = function
       [] -> start_list
       | VDecl(_,name,_)::v_list -> get_name_of_variables (name::start_list) v_list 

let rec check_variable_names = function
        []-> true
        | VDecl(_,name,_)::v_list -> if ( exists name (get_name_of_variables [] v_list) ) then raise(Failure("Duplicate variable names!")) else check_variable_names v_list

let rec check_expr_string = function
        Literal(i) -> "_Literal"
        | Boolean(b) -> "_Boolean"
        | String(s) -> "_String"
        | Id(i) -> get_id_type i !variable_scope_ref
        | Grid(sll) -> "_Grid"
        | Direction(d) -> "_Direction"
        | Bracket(e) -> check_expr_string e 
        | Binop(e,o,e2) -> if ( (check_expr_string e = "_Literal") && (check_expr_string e2 = "_Literal") ) then "_Literal" else raise(Failure("Binary operator must be between ints"))
        | Assign(s,e) -> "hack1" (* need to get a map of id->type *)
        | Call(s,el) -> "hack2"
        | Noexpr -> "_Nothing"
        | BVal(bv) -> "_BVal"
        | RExpr (e,r,e2) -> if ( (check_expr_string e = "_Literal") && (check_expr_string e2 = "_Literal") ) then "_BVal" else raise(Failure("Binary operator must be between ints"))
        | EExpr (ex,eo,ex2) -> if( (check_expr_string ex = "_Grid") || (check_expr_string ex = "_Direction") || ( (check_expr_string ex) != (check_expr_string ex2) ) ) then raise(Failure("== or != must be of same types of int, string, bool...")) else "_BVal"
        | BExpr (ex,bo,ex2) -> if ( (check_expr_string ex = "_BVal") && (check_expr_string ex2 = "_BVal") ) then "_BVal" else raise(Failure("&& and || are for binary operators"))

let rec check_stmt = function
        Expr(e) -> if ( (check_expr_string e) != "stuff" ) then true else false
        | Return(e) -> if ( (check_expr_string e) != "stuff" ) then true else false
        | Block([]) -> true
        | Block(s::s_list) -> ignore(check_stmt s); ignore(check_stmt (Block(s_list))) ; true
        | If(e1,s1,s2) -> if ( ((check_expr_string e1)="_BVal") && (check_stmt s1) && (check_stmt s2) ) then true else raise(Failure("If statement must have boolean value"))
        | While(e,s) -> if ((check_expr_string e) = "_BVal") then true else raise(Failure("While statement must have boolean value"))

let rec check_stmt_list = function
        [] -> true
        | stmt::s_list -> if ( check_stmt stmt ) then check_stmt_list s_list else raise(Failure("stmt error!"))   

let rec enter_formals_scope = function
         [] -> true
         | FParam(BoolType,id)::f_para -> ignore(variable_scope_ref := ("_BVal" , id)::!variable_scope_ref) ; enter_formals_scope f_para
         | FParam(CharType,id)::f_para -> ignore(variable_scope_ref := ("_String" , id)::!variable_scope_ref)  ; enter_formals_scope f_para
         | FParam(IntType, id)::f_para -> ignore(variable_scope_ref := ("_Literal" , id)::!variable_scope_ref) ; enter_formals_scope f_para
         | FParam(VoidType,id)::f_para -> ignore(variable_scope_ref := ("_Void" , id)::!variable_scope_ref) ; enter_formals_scope f_para
         | FParam(GridType,id)::f_para -> ignore(variable_scope_ref := ("_Grid" , id)::!variable_scope_ref) ; enter_formals_scope f_para
         | FParam(DirectionType,id)::f_para -> ignore(variable_scope_ref := ("_Direction" , id)::!variable_scope_ref) ; enter_formals_scope f_para
         | FParam(Actor_TypeType,id)::f_para -> ignore(variable_scope_ref := ("_Actor" , id)::!variable_scope_ref) ; enter_formals_scope f_para

let rec enter_vdecl_scope = function
         [] -> true
         | VDecl(BoolType,id,_)::f_para -> ignore(variable_scope_ref := ("_BVal" , id)::!variable_scope_ref) ; enter_vdecl_scope f_para
         | VDecl(CharType,id,_)::f_para -> ignore(variable_scope_ref := ("_String" , id)::!variable_scope_ref) ; enter_vdecl_scope f_para
         | VDecl(IntType, id,_)::f_para -> ignore(variable_scope_ref := ("_Literal" , id)::!variable_scope_ref) ; enter_vdecl_scope f_para
         | VDecl(VoidType,id,_)::f_para -> ignore(variable_scope_ref := ("_Void" , id)::!variable_scope_ref) ; enter_vdecl_scope f_para
         | VDecl(GridType,id,_)::f_para -> ignore(variable_scope_ref := ("_Grid" , id)::!variable_scope_ref) ; enter_vdecl_scope f_para
         | VDecl(DirectionType,id,_)::f_para -> ignore(variable_scope_ref := ("_Direction" , id)::!variable_scope_ref) ; enter_vdecl_scope f_para
         | VDecl(Actor_TypeType,id,_)::f_para -> ignore(variable_scope_ref := ("_Actor" , id)::!variable_scope_ref) ; enter_vdecl_scope f_para


(* check functions -------------------------------------------------- *)

let rec check_types = function
       [] -> true
       | ActorType(f)::f_list -> check_types f_list
       | CFunc(f)::f_list -> ignore( variable_scope_ref := [] ); ignore(enter_formals_scope f.formals); ignore(enter_vdecl_scope f.locals); if (check_stmt_list f.body) then check_types f_list else raise(Failure("Statement error"))

let rec check_for_duplicate_vars = function
       [] -> true
       | ActorType(f)::f_list -> if (check_variable_names f.alocals) then (check_for_duplicate_vars f_list) else  raise(Failure("Duplicate variable names!"))
       | CFunc(f)::f_list -> if (check_variable_names f.locals) then (check_for_duplicate_vars f_list) else  raise(Failure("Duplicate variable names!"))


let rec check_for_duplicates = function
       [] -> true
       | ActorType(f)::f_list -> if (exists f.aname (get_name_of_functions [] f_list  ) )then raise(Failure("Duplicate function names!")) else check_for_duplicates f_list
       | CFunc(f)::f_list -> if (exists f.fname (get_name_of_functions [] f_list  ) )then raise(Failure("Duplicate function names!")) else check_for_duplicates f_list     


let rec check_for_setup = function
      [] -> raise(Failure("no void setup function with 0 arguments"))
      | ActorType(f)::f_list -> if ( f.aname="setup" ) then raise(Failure("actor_type cannot be named setup")) else check_for_setup f_list
      | CFunc(f)::f_list -> if ( (f.fname="setup") && (f.dtype=VoidType) && ((List.length f.formals)=0)) then true else check_for_setup f_list

let rec check_for_restricted_functions  = function  
       [] -> true
       | ActorType(f)::f_list ->  if (exists f.aname offlimits) then raise(Failure("this function name is offlimits: " ^ f.aname)) else check_for_restricted_functions f_list
       | CFunc(f)::f_list -> if ( exists f.fname offlimits) then raise(Failure("this function name is offlimits: " ^ f.fname)) else check_for_restricted_functions f_list


let check_empty_program = function
     [] -> raise(Failure("No functions or actor types in program"))
     | _ -> true
    
(* entry point - checks the semantics of the program *)
let check_program flist = 
       let _empty = check_empty_program flist  in  (*check to see if program has anything in it *) 
       let _checksetup = check_for_setup flist in
       let _checkrestricted = check_for_restricted_functions flist in (* check to see if there is a function called main *)
       let _check_duplicate_functions = check_for_duplicates flist in (*check to see if there are duplicate function names *)
       let _check_duplicate_variable_decs = check_for_duplicate_vars flist in (*check to see if there are duplicate variable names in each function *)
       let _check_types = check_types flist in (* check to see if variables used are declared *)
       let _ = print_endline "Semantic success. " in
               true





(* Fake Program for testing------------------------------------------- *)

(*
(*int x = 1 *)
let dt1 = IntType;;
let s1 = "x" ;;
let s2 = "1" ;;

let v1 = VDecl(dt1,s1,s2);;

let dt2 = IntType;;
let s3 = "y" ;;
let s4 = "2" ;;

let v2 = VDecl(dt2,s3,s4);;

let v_ls = [v1;v2];;

let cf_body = [If( BVal(True), If( BVal(True), Expr(BVal(True)), Expr(Literal(2))), Expr(Binop(Literal(2),Add,Literal(2))))];;

let at1 = ActorType( {aname = "test"; alocals=v_ls; arules=[]; adefault=[];} );;
let at2 = ActorType ( {aname = "test2"; alocals=v_ls; arules=[]; adefault=[];} );;
let cf1 = CFunc ( {dtype = VoidType; fname = "setup"; formals = [] ; locals = v_ls ; body = cf_body } );;

let flist = [at1;at2;cf1];;

check_program flist;;
*)