(* Scanner for CAL *)

{ open Parser }

rule token = parse
  [' ' '\t' '\r' '\n'] { token lexbuf }      (* Whitespace *)
| "//"          { comment lexbuf }           (* Comments *)

(* Punctuations *)
| '('           { LPAREN }
| ')'           { RPAREN }
| '{'           { LBRACE }
| '}'           { RBRACE }
| '['           { LBRACK }
| ']'           { RBRACK }
| '|'           { BAR }
| ';'           { SEMI }
| ':'           { COLON }
| ','           { COMMA }

(* Arithmetic Operators *)
| '+'           { PLUS }
| '-'           { MINUS }
| '*'           { TIMES }
| '/'           { DIVIDE }
| '%'           { MOD }
| '='           { ASSIGN }

(* Relational Operators *)
| "=="          { EQ }
| "!="          { NEQ }
| "&&"          { AND }
| "||"          { OR }
| '<'           { LT }
| "<="          { LEQ }
| ">"           { GT }
| ">="          { GEQ }
| "=>"          { ARROW }
| '!'	        { NOT }

(* Key Words *)
| "if"          { IF }
| "else"        { ELSE }
| "true"        { TRUE }
| "false"       { FALSE }
| "while"       { WHILE }
| "bool"        { BOOL }
| "char"        { CHAR }
| "int"         { INT }
| "void"		{ VOID }
| "grid"        { GRID }
| "direction"   { DIRECTION }
| "north"       { NORTH }
| "south"       { SOUTH }
| "east"        { EAST }
| "west"        { WEST }
| "northwest"   { NORTHWEST }
| "southwest"   { SOUTHWEST }
| "northeast"   { NORTHEAST }
| "southeast"   { SOUTHEAST }
| "center"        { CENTER }
| "rules"       { RULES }
| "return"      { RETURN }
| "def"         { DEF }
| "init"        { INIT }
| "default" 	{ DEFAULT }
| "actor_type"  { ACTOR_TYPE }

(* Literals *)
| ['0'-'9']+ as lxm { LITERAL(int_of_string lxm) }
| '\"'[^'\"']*'\"' as lxm { STR_LITERAL(lxm) }
| ['a'-'z' 'A'-'Z' '_']['a'-'z' 'A'-'Z' '0'-'9' '_']* as lxm { ID(lxm) }
| eof { EOF }

(* Special Character Process*)
| _ as char { raise (Failure("illegal character " ^ Char.escaped char)) }

and comment = parse
  '\n'        { token lexbuf }
| _           { comment lexbuf }
