%{ 
open Ast
let parse_error s = 
  print_endline s;
  flush stdout

%}

%token LPAREN RPAREN LBRACE RBRACE LBRACK RBRACK BAR SEMI COLON COMMA
%token PLUS MINUS TIMES DIVIDE MOD ASSIGN
%token EQ NEQ AND OR LT LEQ GT GEQ ARROW NOT 
%token IF ELSE TRUE FALSE WHILE 
%token BOOL CHAR INT GRID VOID
%token DIRECTION NORTH SOUTH EAST WEST NORTHWEST SOUTHWEST NORTHEAST SOUTHEAST CENTER
%token RULES RETURN DEF INIT DEFAULT ACTOR_TYPE

%token <int> LITERAL
%token <string> STR_LITERAL
%token <string> ID

%token EOF

%nonassoc NOELSE
%nonassoc ELSE
%right ASSIGN
%left OR 
%left AND
%left EQ NEQ
%left LT GT LEQ GEQ
%left PLUS MINUS
%left TIMES DIVIDE
%left MOD


%start program
%type <Ast.program> program


%%


program: 
	 /* nothing */ { [] }
   | program fdecl { ($2 :: $1) }
   | program actor_type { ($2 :: $1) }

fdecl:
          DEF datatype id LPAREN formals_opt RPAREN LBRACE vdecl_list stmt_list RBRACE
                        {
                                CFunc({ 
					dtype = $2; 
					fname = $3; 
					formals = $5; 
					locals = List.rev $8; 
					body= List.rev $9
				})
          
                        }


formals_opt:
          /* nothing */ { [] }
        | formal_list   { List.rev $1 }

formal_list:
          datatype id                   { [FParam($1, $2)] }
        | formal_list COMMA datatype id { FParam($3, $4) :: $1 }


id:
	 ID     { $1 }

grid:
        LBRACK grid_matrix RBRACK { $2 }

grid_row:
        ID                      { [$1] }
        | grid_row COMMA ID { $3 :: $1 }

grid_matrix:
        grid_row SEMI           { [$1] }
        | grid_matrix grid_row SEMI { $2 :: $1 }


datatype:
          BOOL          { BoolType }
          | CHAR        { CharType }
          | INT         { IntType }
          | VOID        { VoidType }
          | GRID        { GridType }
          | DIRECTION   { DirectionType }
          | ACTOR_TYPE  { Actor_TypeType }


rule:
	expr ARROW LBRACE stmt_list RBRACE { Rule($1, $4) }

rule_list:
	 /* nothin */ { [] }
	 | rule_list rule { $2 :: $1 }

default_rule:     
         DEFAULT ARROW LBRACE stmt_list RBRACE  { $4 }

vdecl_list:
          /* nothing */    { [] }
        | vdecl_list vdecl { $2 :: $1 }

vdecl:
        datatype id ASSIGN expr SEMI { VDecl($1,$2,string_of_expr $4) }


actor_type:
        ACTOR_TYPE id ASSIGN BAR INIT COLON vdecl_list RULES COLON rule_list default_rule BAR
                        {
                                ActorType({
				  aname = $2;
				   alocals = $7;
				   arules = $10;
				   adefault = $11;})

                        }



stmt_list:
          /* No empty block allowed */ { [] }
        | stmt_list stmt               { $2 :: $1 }



stmt:
      expr SEMI                                         { Expr($1) }
      | RETURN expr SEMI                                { Return($2) }
      | LBRACE stmt_list RBRACE                         {Block(List.rev $2) }
      | IF LPAREN expr RPAREN stmt %prec NOELSE         { If($3, $5, Block([])) }
      | IF LPAREN expr RPAREN stmt ELSE stmt            { If($3, $5, $7) }
      | WHILE LPAREN expr RPAREN stmt                   { While($3, $5 ) }

expr:
	LITERAL                         { Literal($1) }
        | STR_LITERAL                   { String($1) }
	| id 				{ Id($1) }
	| grid                          { Grid($1) }
	| expr PLUS expr 		{ Binop($1, Add, $3 ) }
	| expr MINUS  expr             	{ Binop($1, Sub,   $3) }
        | expr TIMES  expr            	{ Binop($1, Mult,  $3) }
        | expr MOD expr 	        { Binop($1, Div,   $3) }
	/*Directions*/
	| NORTH                         { Direction(North)     }
        | SOUTH                         { Direction(South)     }
        | WEST                          { Direction(West)      }
        | EAST                          { Direction(East)      }
        | NORTHEAST                     { Direction(NorthEast) }
        | NORTHWEST                     { Direction(NorthWest) }
        | SOUTHEAST                     { Direction(SouthEast) }
        | SOUTHWEST                     { Direction(SouthWest) }
        | CENTER                        { Direction(Center) }
	/* bools */
        | TRUE                          { BVal(True) }
        | FALSE                         { BVal(False) }
	| expr EQ expr                  { EExpr($1, BEqual, $3) }
        | expr NEQ expr                 { EExpr($1, BNeq, $3) }
        | expr GT expr                  { RExpr($1, BGreater, $3) }
        | expr GEQ expr                 { RExpr($1, BGeq, $3) }
        | expr LT expr                  { RExpr($1, BLess, $3) }
        | expr LEQ expr                 { RExpr($1, BLeq, $3) }
        | expr AND expr                 { BExpr($1, And, $3) }
        | expr OR expr                  { BExpr($1, Or, $3) }   	
	| id ASSIGN expr                { Assign($1, $3) }
        | id LPAREN actuals_opt RPAREN  { Call($1, $3) }	
        | LPAREN expr RPAREN            { Bracket($2) }


actuals_opt:
	/* nothin */ { [] }
	| actual_list {List.rev $1}

actual_list:
	expr	{ [$1] } 
	| actual_list COMMA expr { $3 :: $1 }
