(* AST *)

type op = Add | Sub | Mult | Div | Equal | Neq | Less | Leq | Greater | Geq | Mod
type direc = North | South | East | West | NorthEast | NorthWest | SouthEast | SouthWest | Center

type bv = True | False 

type bop = And | Or
type eop = BEqual | BNeq
type rop = BLess | BLeq | BGreater | BGeq

type mop = MTimes | MDivide | MMod (*multiplicative expr ops*)
type aop = AAdd | ASub (*additve expr ops*)

type vop = VAdd | VSub | VMult | VDiv

type dt = BoolType | CharType | IntType | VoidType | GridType | DirectionType | Actor_TypeType 

type fparam = FParam of dt * string 

type vdecl = VDecl of dt * string * string  

type expr =
          Literal of int
        | Boolean of bool
        | String of string
        | Id of string
        | Grid of string list list
        | Direction of direc
        | Bracket of expr
        | Binop of expr * op * expr
        | Assign of string * expr
        | Call of string * expr list
        | Noexpr
        | BVal of bv 
        | RExpr of expr * rop * expr 
        | EExpr of expr * eop * expr 
        | BExpr of expr * bop * expr 

type stmt =
         Expr of expr
        | Return of expr
        | Block of stmt list
        | If of expr * stmt * stmt
        | While of expr * stmt


type rule = Rule of expr * stmt list

type arithexpr =
        | ALiteral of int
        | AId of string

type varexpr = 
        | VLiteral of int
        | VId of string
        | VStringLit of string
        | VBoolLit of bool
        | VBinop of varexpr *  vop * varexpr

type func_decl = {
        dtype : dt;
        fname : string;
        formals : fparam list;
        locals : vdecl list;
        body : stmt list;
}

type actor_type = {
          aname : string;
          alocals : vdecl list;
          arules : rule list;
          adefault: stmt list;

}

type func =
        | CFunc of func_decl 
        | ActorType of actor_type


type program = func list


let string_of_var_dec (a,b,c) = a ^ b ^ c

let string_of_vop  = function
        | VAdd -> "+"
        | VSub -> "-"
        | VMult-> "*"
        | VDiv -> "/"

let string_of_arithexpr  = function
        | ALiteral(i) -> string_of_int i
        | AId(s) -> s


let rec string_of_varexpr = function
        | VLiteral(i) -> string_of_int i
        | VId(s) -> s
        | VStringLit(s) -> s
        | VBoolLit(b) -> string_of_bool b
        | VBinop(v1,op,v2) -> string_of_varexpr v1 ^ " " ^ string_of_vop op ^ " " ^ string_of_varexpr v2


let string_of_dt = function
        BoolType -> "bool"
        | CharType -> "char"
        | IntType -> "int"
        | VoidType -> "void"
        | GridType -> "grid"
        | DirectionType -> "direction"
        | Actor_TypeType -> "actor_type"


let string_of_bop = function
        | And -> "&&"
        | Or -> "||"

let string_of_rop = function
        | BLess -> "<"
        | BLeq -> "<="
        | BGreater -> ">"
        | BGeq -> ">="

let string_of_eop = function
        | BEqual -> "=="
        | BNeq -> "!="

let string_of_bv = function
        | True -> "true"
        | False -> "false"

let string_of_op = function
          Add -> "+"
        | Sub -> "-"
        | Mult -> "*"
        | Div -> "/"
        | Equal -> "=="
        | Neq -> "!="
        | Less -> "<"
        | Leq -> "<="
        | Greater -> ">"
        | Geq -> ">="
        | Mod -> "%"


let rec string_of_listlist = function
        [] -> ""
        | a::b -> (String.concat "," a) ^ ";" ^ ( string_of_listlist b)


let string_of_grid = function
        g -> "[" ^ (string_of_listlist g) ^ "]"

let rec string_of_expr = function
          Literal(l) -> string_of_int l
        | Boolean(b) -> string_of_bool b
        | String(s) -> s
        | Id(s) -> s
        | Grid(g) -> string_of_grid g
        | Direction(d) -> 
                      begin
                               match d with 
                                    North -> "NORTH" | South -> "SOUTH" | East -> "EAST" | West -> "WEST"  | Center -> "CENTER"
                                  | NorthEast -> "NORTHEAST" | SouthEast -> "SOUTHEAST" | NorthWest -> "NORTHWEST" | SouthWest -> "SOUTHWEST"
		       end
	| Binop(e1, o, e2) ->
                      begin
                               match o with
                                  | _ ->
                                  string_of_expr e1 ^ " " ^ (match o with
                                                  Add -> "+" | Sub -> "-" | Mult -> "*" | Div -> "/"
                                                | Equal -> "==" | Neq -> "!="
                                                | Less -> "<" | Leq -> "<=" | Greater -> ">" | Geq -> ">=" | Mod -> "%")
                                        ^ " " ^ string_of_expr e2
                      end
        | Assign(v, e) -> v ^ " = " ^ string_of_expr e
        | Call(f, el) -> (match Str.string_match (Str.regexp "move") f  0 with
            true -> f ^ "(" ^ String.concat ", " (List.map string_of_expr el) ^ ", read_grid, write_grid, i, j)"
            |_ -> match Str.string_match (Str.regexp "assign_type") f  0 with
                true -> f ^ "(" ^ String.concat ", " (List.map string_of_expr el) ^ ", write_grid, i, j)"
                |_ -> match Str.string_match (Str.regexp "neighborhood") f  0 with
                    true -> f ^ "(" ^ String.concat ", " (List.map string_of_expr el) ^ ", read_grid, i, j)"
                    |_ -> match Str.string_match (Str.regexp "randomof") f  0 with
                        true -> f ^ "(" ^ String.concat ", " (List.map string_of_expr el) ^ ", read_grid , i, j)"
                        |_ -> match Str.string_match (Str.regexp "cellat") f  0 with
                            true -> f ^ "(" ^ String.concat ", " (List.map string_of_expr el) ^ ", read_grid , i, j)"
                            |_ -> f ^ "(" ^ String.concat ", " (List.map string_of_expr el) ^ ")")
        | Noexpr -> ""
        | BVal(v) -> string_of_bv v
        | RExpr(e1,o,e2) -> string_of_expr e1 ^ " " ^ string_of_rop o ^ " " ^ string_of_expr e2
        | EExpr(e1,o,e2) -> string_of_expr e1 ^ " " ^ string_of_eop o ^ " " ^ string_of_expr e2
        | BExpr(e1,o,e2) -> string_of_expr e1 ^ " " ^ string_of_bop o ^ " " ^ string_of_expr e2
        | Bracket(e1) -> " ( " ^ string_of_expr e1 ^ " ) "

let rec string_of_stmt = function
          Block(stmts) -> "{\n" ^ String.concat "" (List.map string_of_stmt stmts) ^ "}\n"
        | Expr(expr) -> string_of_expr expr ^ ";\n";
        | Return(expr) -> "return " ^ string_of_expr expr ^ ";\n";
        | If(e, s, Block([])) -> "if (" ^ string_of_expr e ^ ")\n" ^ string_of_stmt s
        | If(e, s1, s2) ->  "if (" ^ string_of_expr e ^ ")\n" ^ string_of_stmt s1 ^ "else\n" ^ string_of_stmt s2
        | While(e, s) -> "while (" ^ string_of_expr e ^ ") " ^ string_of_stmt s

let rec string_of_expr_at = function
          Literal(l) -> string_of_int l
        | Boolean(b) -> string_of_bool b
        | String(s) -> s
        | Id(s) -> s
        | Grid(g) -> string_of_grid g
        | Direction(d) -> 
                     begin
                               match d with 
                                    North -> "NORTH" | South -> "SOUTH" | East -> "EAST" | West -> "WEST" | Center -> "CENTER"
                                  | NorthEast -> "NORTHEAST" | SouthEast -> "SOUTHEAST" | NorthWest -> "NORTHWEST" | SouthWest -> "SOUTHWEST"
                     end
        | Binop(e1, o, e2) ->
                     begin
                               match o with
                                  | _ ->
                                                  string_of_expr_at e1 ^ " " ^ (match o with
                                                                  Add -> "+" | Sub -> "-" | Mult -> "*" | Div -> "/"
                                                                | Equal -> "==" | Neq -> "!="
                                                                | Less -> "<" | Leq -> "<=" | Greater -> ">" | Geq -> ">=" | Mod -> "%")
                                                        ^ " " ^ string_of_expr_at e2
                        end
        | Assign(v, e) -> "!addwritegrid!" ^ v ^ " = " ^ string_of_expr_at e
        | Call(f, el) -> (match Str.string_match (Str.regexp "move") f  0 with
            true -> f ^ "(" ^ String.concat ", " (List.map string_of_expr_at el) ^ ", read_grid, write_grid, i, j, width, height)"
            |_ -> match Str.string_match (Str.regexp "assign_type") f  0 with
                true -> f ^ "(" ^ String.concat ", " (List.map string_of_expr_at el) ^ ", write_grid, i, j, width, height)"
                |_ -> match Str.string_match (Str.regexp "neighborhood") f  0 with
                    true -> f ^ "(" ^ String.concat ", " (List.map string_of_expr_at el) ^ ", read_grid, i, j, width, height)"
                    |_ -> match Str.string_match (Str.regexp "randomof") f  0 with
                        true -> f ^ "(" ^ String.concat ", " (List.map string_of_expr_at el) ^ ", read_grid , i, j, width, height)"
                        |_ -> match Str.string_match (Str.regexp "cellat") f  0 with
                            true -> f ^ "(" ^ String.concat ", " (List.map string_of_expr_at el) ^ ", read_grid , i, j, width, height)"
                            |_ -> f ^ "(" ^ String.concat ", " (List.map string_of_expr_at el) ^ ")")
        | Noexpr -> ""
        | BVal(v) -> string_of_bv v
        | RExpr(e1,o,e2) -> string_of_expr_at e1 ^ " " ^ string_of_rop o ^ " " ^ string_of_expr_at e2
        | EExpr(e1,o,e2) -> string_of_expr_at e1 ^ " " ^ string_of_eop o ^ " " ^ string_of_expr_at e2
        | BExpr(e1,o,e2) -> string_of_expr_at e1 ^ " " ^ string_of_bop o ^ " " ^ string_of_expr_at e2
        | Bracket(e1) -> " ( " ^ string_of_expr_at e1 ^ " ) "



let rec string_of_stmt_at = function
          Block(stmts) -> "{\n" ^ String.concat "" (List.map string_of_stmt_at stmts) ^ "}\n"
        | Expr(expr) -> string_of_expr_at expr ^ ";\n";
        | Return(expr) -> "return " ^ string_of_expr_at expr ^ ";\n";
        | If(e, s, Block([])) -> "if (" ^ string_of_expr_at e ^ ")\n" ^ string_of_stmt_at s
        | If(e, s1, s2) ->  "if (" ^ string_of_expr_at e ^ ")\n" ^ string_of_stmt_at s1 ^ "else\n" ^ string_of_stmt_at s2
        | While(e, s) -> "while (" ^ string_of_expr_at e ^ ") " ^ string_of_stmt_at s

let rec string_of_expr_setup = function
          Literal(l) -> string_of_int l
        | Boolean(b) -> string_of_bool b
        | String(s) -> s
        | Id(s) -> s
        | Grid(g) -> string_of_grid g
        | Direction(d) -> 
                       begin
                               match d with 
                                    North -> "NORTH" | South -> "SOUTH" | East -> "EAST" | West -> "WEST" | Center -> "CENTER"
                                  | NorthEast -> "NORTHEAST" | SouthEast -> "SOUTHEAST" | NorthWest -> "NORTHWEST" | SouthWest -> "SOUTHWEST"
                       end
        | Binop(e1, o, e2) ->
                        begin
                               match o with
                                  | _ ->
                                                        string_of_expr_setup e1 ^ " " ^ (match o with
                                                                  Add -> "+" | Sub -> "-" | Mult -> "*" | Div -> "/"
                                                                | Equal -> "==" | Neq -> "!="
                                                                | Less -> "<" | Leq -> "<=" | Greater -> ">" | Geq -> ">=" | Mod -> "%")
                                                        ^ " " ^ string_of_expr_setup e2
                        end
        | Assign(v, e) -> "!addwritegrid!" ^ v ^ " = " ^ string_of_expr_setup e
        | Call(f, el) -> (match Str.string_match (Str.regexp "move") f  0 with
            true -> f ^ "(" ^ String.concat ", " (List.map string_of_expr_setup el) ^ ", read_grid, write_grid, i, j, width, height)"
            |_ -> match Str.string_match (Str.regexp "assign_type") f  0 with
                true -> f ^ "(" ^ String.concat ", " (List.map string_of_expr_setup el) ^ ", write_grid, i, j, width, height)"
                |_ -> match Str.string_match (Str.regexp "neighborhood") f  0 with
                    true -> f ^ "(" ^ String.concat ", " (List.map string_of_expr_setup el) ^ ", read_grid, i, j, width, height)"
                    |_ -> match Str.string_match (Str.regexp "randomof") f  0 with
                        true -> f ^ "(" ^ String.concat ", " (List.map string_of_expr_setup el) ^ ", read_grid , i, j, width, height)"
                        |_ -> match Str.string_match (Str.regexp "cellat") f  0 with
                            true -> f ^ "(" ^ String.concat ", " (List.map string_of_expr_setup el) ^ ", read_grid , i, j, width, height)"
                            |_ -> match Str.string_match (Str.regexp "grid_size") f  0 with
                                true -> f ^ "(" ^ String.concat ", " (List.map string_of_expr_setup el) ^ ", &grid1, &grid2, &MAX_X, &MAX_Y)"
                                |_ -> match Str.string_match (Str.regexp "set_actor") f  0 with
                                    true -> f ^ "(" ^ String.concat ", " (List.map string_of_expr_setup el) ^ ", grid1 , grid2)"
                                    |_ -> match Str.string_match (Str.regexp "cellat") f  0 with
                                        true -> f ^ "(" ^ String.concat ", " (List.map string_of_expr_setup el) ^ ", read_grid , i, j)"
                                        |_ -> match Str.string_match (Str.regexp "set_chronon") f  0 with
                                            true -> f ^ "(" ^ String.concat ", " (List.map string_of_expr_setup el) ^ ", &STEP_TIME)"
                                            |_ -> match Str.string_match (Str.regexp "set_grid_pattern") f  0 with
                                                true -> f ^ "(" ^ String.concat ", " (List.map string_of_expr_setup el) ^ ", grid1, grid2)"
                                                |_ -> match Str.string_match (Str.regexp "set_cell_size") f  0 with
                                                    true -> f ^ "(" ^ String.concat ", " (List.map string_of_expr_setup el) ^ ", &CELL_SIZE)"
                                                    |_ -> match Str.string_match (Str.regexp "set_grid_random") f  0 with
                                                        true -> f ^ "(" ^ String.concat ", " (List.map string_of_expr_setup el) ^ "grid1, grid2, MAX_X, MAX_Y)"
                                                        |_ -> match Str.string_match (Str.regexp "read_grid") f  0 with
                                                            true -> f ^ "(" ^ String.concat ", " (List.map string_of_expr_setup el) ^ ", grid1, grid2, MAX_X, MAX_Y)"
                                                            |_ -> f ^ "(" ^ String.concat ", " (List.map string_of_expr_setup el) ^ ")")
        | Noexpr -> ""
        | BVal(v) -> string_of_bv v
        | RExpr(e1,o,e2) -> string_of_expr_at e1 ^ " " ^ string_of_rop o ^ " " ^ string_of_expr_at e2
        | EExpr(e1,o,e2) -> string_of_expr_at e1 ^ " " ^ string_of_eop o ^ " " ^ string_of_expr_at e2
        | BExpr(e1,o,e2) -> string_of_expr_at e1 ^ " " ^ string_of_bop o ^ " " ^ string_of_expr_at e2
        | Bracket(e1) -> " ( " ^ string_of_expr_at e1 ^ " ) "



let rec string_of_stmt_setup = function
          Block(stmts) -> "{\n" ^ String.concat "" (List.map string_of_stmt_setup stmts) ^ "}\n"
        | Expr(expr) -> string_of_expr_setup expr ^ ";\n";
        | Return(expr) -> "return " ^ string_of_expr_setup expr ^ ";\n";
        | If(e, s, Block([])) -> "if (" ^ string_of_expr_setup e ^ ")\n" ^ string_of_stmt_setup s
        | If(e, s1, s2) ->  "if (" ^ string_of_expr_setup e ^ ")\n" ^ string_of_stmt_setup s1 ^ "else\n" ^ string_of_stmt_setup s2
        | While(e, s) -> "while (" ^ string_of_expr_setup e ^ ") " ^ string_of_stmt_setup s


let string_of_vdecl = function
        VDecl(dtt, nm, v) ->  string_of_dt dtt ^ " " ^ nm ^ " = " ^ v ^ ";\n"


let string_of_fparam = function
        FParam(dt,s) -> string_of_dt dt ^ " " ^ s


let string_of_fdecl  = function
        | CFunc(fdecl) -> if (String.compare fdecl.fname "setup") !=0 then
                        "\n" ^ string_of_dt fdecl.dtype ^ " " ^ fdecl.fname ^ "(" ^ String.concat ", " (List.map string_of_fparam fdecl.formals) ^ ") {\n" ^
                        String.concat "" (List.map string_of_vdecl fdecl.locals) ^
                        String.concat "" (List.map string_of_stmt fdecl.body) ^
                        "}\n"
                    else ""
        |_ -> ""


let string_of_program (funcs) = String.concat "\n" (List.map string_of_fdecl funcs)




