{ open Parser }

rule token = parse
  | [' ' '\t'] { token lexbuf } (* Whitespace *)
  | '\r'?'\n' { NEWLINE } (* Note - newlines are NOT just whitespace in the boredgame language *)
  | "/*"     { comment lexbuf }            (* Comments *)
  | "//"     { comment_singleline lexbuf } (* Comments *)
  | '('      { LPAREN }
  | ')'      { RPAREN }
  | '{'      { LBRACE }
  | '}'      { RBRACE }
  | '.'      { DOT }
  | '['      { LBRACKET }
  | ']'      { RBRACKET }
  | ';'      { SEMI }
  | ','      { COMMA }
  | '+'      { PLUS }
  | '-'      { MINUS }
  | '*'      { TIMES }
  | '/'      { DIVIDE }
  | '@'      { AT  }
  | '='      { ASSIGN }
  | "=="     { EQ }
  | "!="     { NEQ }
  | '<'      { LT }
  | "<="     { LEQ }
  | ">"      { GT }
  | ">="     { GEQ }
  | "and"     { AND }
  | "or"     { OR }
  | "->"     { ARROW }
  | "~="     { MATCH }
  | "if"     { IF }
  | "then"     { THEN }
  | "else"   { ELSE }
  | "elseif" { ELSEIF }
  | "for"    { FOR }
  | "do"  { DO }
  | "while"  { WHILE }
  | "return" { RETURN }
  | "end"    { END }
  | "function"    { FUNCTION }
  | "enum"    { ENUM }
  | "true"     { BOOLEANLITERAL(true) }
  | "false"     { BOOLEANLITERAL(false) }
  | ['0'-'9']+ as lxm { LITERAL(int_of_string lxm) }
  | '\"'[^'\"']*'\"' as lxm { STRINGLITERAL(String.sub lxm 1 ((String.length lxm) - 2)) }
  | ['a'-'z' 'A'-'Z']['a'-'z' 'A'-'Z' '0'-'9' '_']* as lxm { ID(lxm) }
  | eof { EOF }
  | _ as char { raise (Failure(Printf.sprintf "illegal character %s at %d near %s"
                         (Char.escaped char)
                         lexbuf.Lexing.lex_start_pos
                         (String.sub lexbuf.Lexing.lex_buffer (lexbuf.Lexing.lex_start_pos - 5) 10)
                              )) }
and comment = parse
  | "*/" { token lexbuf }
  | _    { comment lexbuf }
and comment_singleline = parse
  | '\r'?'\n' { token lexbuf }
  | _    { comment_singleline lexbuf }
      
        
    
    