open Ast;;
open Printf;;
open Env;;
open Common;;



let op_expected_type = function
  | Add -> int_type
  | Sub -> int_type
  | Mult -> int_type
  | Div -> int_type
  | Equal -> boolean_type
  | Neq -> boolean_type
  | Less -> boolean_type
  | Leq -> boolean_type
  | Greater -> boolean_type
  | Geq -> boolean_type
  | Or -> boolean_type
  | And -> boolean_type
  | Concat -> string_type
;;


(*
   Convert the incoming expression to a semantically-analyzed expression.
   Tack on the autogenerated symbol reference if needed
 *)
let rec to_sast_expr env expr = match expr with
  | IntLiteral(l) ->
    (* Note - this can be represented as a direct literal in C; hence, no need for the get_next_tmp_id *)
    {e = SastIntLiteral(l); etype = int_type; eref = string_of_int l};
  | BooleanLiteral(l) ->
    (* Note - this can be represented as a direct literal in C; hence, no need for the get_next_tmp_id *)
    {e = SastBooleanLiteral(l); etype = boolean_type; eref = string_of_bool l};
  | StringLiteral(l) ->
    (* Note - this can be represented as a direct literal in C; hence, no need for the get_next_tmp_id *)
    {e = SastStringLiteral(l); etype = string_type; eref = sprintf "\"%s\"" l};
  | Id(id) ->
    let idref = (Env.lookup_symbol env id) in
    {e = SastId(id); etype = idref.ref_type; eref = idref.code_name};
  | BoardLiteral(array) ->
    let numrows = List.length array in
    if numrows = 0 then raise(CompileException("Board must have >0 rows defined"));
    let numcols = List.length (List.hd array) in
    if numcols = 0 then raise(CompileException("Board must have >0 columns defined"));
    let idref = Env.lookup_symbol env (List.hd (List.hd array)) in
    let boardtype = idref.ref_type in
    if List.for_all (fun l -> List.length l = numcols) array <> true then raise(CompileException("All rows of a board must have the same number of columns"));

    let board_elems =
      (List.map (fun elems -> List.map
        (fun id ->
          let valexpr = to_sast_expr env (Id(id)) in
          if not(type_is_of_type "enum" valexpr.etype) then raise(CompileException("Only enum types are allowed in a board"));
          valexpr
        ) elems) array) in
    (* The board literal is too complex to represent as a standalone literal in C, as we did for int, boolean, string
       Hence, we will use get_next_tmp_id here for this *)
    {e = SastBoardLiteral(numrows, numcols, board_elems); etype = GenericTypeId("board", boardtype); eref = get_next_tmp_id () };
  | Assign(expr, e) ->
    let exprtype, exprref =
      (match expr with
        | Id(id) -> let idref = Env.lookup_symbol env id in (idref.ref_type, idref.code_name)
        | MatrixOp(boardexp, rowexp, colexp) -> let sast_expr = to_sast_expr env expr in (sast_expr.etype, sast_expr.eref)
        | _ -> raise(CompileException("not yet ipmlemented for others"))
      )
    in
    let rhs = to_sast_expr env e in
    if rhs.etype = exprtype
    then {e = SastAssign(rhs); etype = rhs.etype; eref = exprref}
    else raise(CompileException(sprintf "args do not have matching types: LHS[%s] vs. RHS[%s]" (type_id_to_string exprtype) (type_id_to_string rhs.etype)));
  | MatrixOp(boardexp, rowexp, colexp) ->
    let board = to_sast_expr env boardexp in
    let row = to_sast_expr env rowexp in
    let col = to_sast_expr env colexp in
    if not (type_is_of_type "board" board.etype) then raise(CompileException("lhs in matrix operation must be of type board; instead it was " ^ (type_id_to_string board.etype)));
    if not (type_is_of_type "int" row.etype) then raise(CompileException("index operands in matrix operation must be of type int; instead it was " ^ (type_id_to_string row.etype)));
    if not (type_is_of_type "int" col.etype) then raise(CompileException("index operands in matrix operation must be of type int; instead it was " ^ (type_id_to_string col.etype)));
    let arrayref = calc_matrix_to_array_index_for_c_code board row col in
    {e = SastMatrixOp(board, row, col); etype = type_to_generic_subtype board.etype; eref = arrayref}
  | ClassOp(expr, memberName) ->
    let sexpr = to_sast_expr env expr in
    if not(type_is_of_type "board" sexpr.etype) then raise(CompileException("Dot operator can only be used on classes (e.g. board)"));
    let return_type = match memberName with
      | "rowlength" -> int_type
      | "collength" -> int_type
      | _ -> raise(CompileException("No other args than rowlength and collength are supported"))
    in
    {e = SastClassOp(sexpr, memberName); etype = return_type; eref = get_next_tmp_id ()}
  | RegexpMatcher(inputstr, regexpstr, vars) ->
    let input = to_sast_expr env inputstr in
    let regexp = to_sast_expr env regexpstr in
    if input.etype <> string_type then raise(CompileException("Input to regexp matching operation must be a string"));
    if regexp.etype <> string_type then raise(CompileException("Regexp in regexp matching operation must be a string"));
    (* the actual symbols (i.e. the 4th arg to SastRegexpMatcher) will be fixed later in the If logic inside to_sast_stmt *)
    {e = SastRegexpMatcher(input, regexp, vars, []); etype = boolean_type; eref = get_next_tmp_id ()}
  | Binop(e1, op, e2) ->
    let sexp1 = to_sast_expr env e1 in
    let tmpid = get_next_tmp_id () in
    let sexp2 = to_sast_expr env e2 in
    (match op with
      | Concat ->
        if sexp1.etype <> string_type && sexp2.etype <> string_type
        then raise(CompileException("One of the types in a concatenation must be a string"));
      | _ ->
        if sexp1.etype <> sexp2.etype
        then raise(CompileException("Both sides of Binop need to evaluate to the same type"));
    );
    {e = SastBinop(sexp1, op, sexp2); etype = op_expected_type op; eref = tmpid}
  | Call (id, exprs) ->
    let idref = Env.lookup_symbol env id in
    if idref.ref_type <> function_type
    then raise(CompileException("Attempting to make a function call on a non-function " ^ id));
    let sexprs = List.map (fun expr -> to_sast_expr env expr) exprs in
    let func = Env.lookup_function env id in
    let numfuncargs = List.length func.params in
    let numcallargs = List.length sexprs in
    if numcallargs <> numfuncargs
    then raise(CompileException(sprintf "Function mismatch in terms of # params passed in. Func [%s] expects %d args, only received %d" func.fname numfuncargs numcallargs ));
    List.iter2
      (fun fparam cexpr ->
        let ftype = type_id_to_string fparam.vtype in
        let ctype = type_id_to_string cexpr.etype in
        if ftype <> ctype then raise(CompileException(sprintf "Mismatch in function types to call of %s for var %s [func-%s vs. call-%s]" func.fname fparam.vname ftype ctype));
      )
      func.params sexprs;
    {e = SastCall(id, sexprs); etype = func.rettype; eref = get_next_tmp_id ()}
;;


let rec to_sast_stmt env = function
  | NoOpStmt -> SastNoOpStmt, env
  | Block(stmts) ->
    (* Note - for the statements in a block, we do want the environment changes from one statement to carry
       over to the next. Hence, we use fold_left here *)
    let sast_stmts, env = List.fold_left
      (fun (sast_stmts, env) stmt ->
        let new_sast_stmt, new_env = to_sast_stmt env stmt in
        sast_stmts @ [new_sast_stmt], new_env
      ) ([], env) stmts in
    SastBlock(sast_stmts), env;
  | VarDecl(v) ->
    let env = Env.add_var_decl env v in
    let idref = Env.lookup_symbol env v.vname in
    SastVarDecl(v.vname, idref), env;
  | VarDeclAndAssign(v, expr) ->
    let env = Env.add_var_decl env v in
    let idref = Env.lookup_symbol env v.vname in
    let sexpr = to_sast_expr env expr in
    SastVarDeclAndAssign(v.vname, idref, sexpr), env;
  | Expr(e) ->
    (
    let sexpr = to_sast_expr env e in
    match e with
      | Assign(id, e) -> SastExpr(sexpr), env;
      | Call (id, exprs) -> SastExpr(sexpr), env;
      | _ -> raise(CompileException("No other expression type is allowed as a statement"));
    )
  | Return(e) ->
    let sexpr = to_sast_expr env e in
    let func = Env.lookup_function env env.cur_func_name in
    if sexpr.etype = func.rettype
    then SastReturn(sexpr), env
    else raise(CompileException(sprintf "Return statement value [%s] in function [%s] does not match the return type of the function [%s]" (type_id_to_string sexpr.etype) func.fname (type_id_to_string func.rettype)))
  | If(e, s1, elseifs, s2) ->
    let conds = (e, s1) :: elseifs in
    let sast_conds = List.fold_left
      (fun list (expr, stmt) ->
        let sexpr = to_sast_expr env expr in
        if sexpr.etype <> boolean_type then raise (CompileException ("Clause of the if-statement must have a boolean value"));
        let sexpr, newenv = match sexpr.e with
          | SastRegexpMatcher(input, regexp, vars, _) ->
            let newenv = List.fold_left (fun env var -> Env.add_var_decl env var) env vars in
        	let newrefs = List.map (fun var -> Env.lookup_symbol newenv var.vname) vars in
            let newsexpr = { e = SastRegexpMatcher(input, regexp, vars, newrefs); etype = sexpr.etype; eref = sexpr.eref } in
            newsexpr, newenv;
          | _ -> sexpr, env
            (* no need to carry the environment over, as it will be in a block and any newly-created variables would lose scope *)
        in
        let sast_stmt, _ = to_sast_stmt newenv stmt in
        list @ [(sexpr, sast_stmt)];
      ) [] conds in
    let sast_stmt2, newenv = to_sast_stmt env s2 in
    (SastIf(sast_conds, sast_stmt2), env)
  | While(expr, stmt) ->
    let sexpr = to_sast_expr env expr in
    (* no need to carry the environment over, as it will be in a block and any newly-created variables would lose scope *)
    let sast_stmt, _ = to_sast_stmt env stmt in
    SastWhile(sexpr, sast_stmt), env
  | For(e1, e2, e3, s) ->
    let sexpr1 = to_sast_expr env e1 in
    let sexpr2 = to_sast_expr env e2 in
    let sexpr3 = to_sast_expr env e3 in
    let sast_stmt, _ = to_sast_stmt env s in
    if sexpr2.etype <> boolean_type then raise (CompileException ("2nd clause of the for-loop declaration must return a boolean"));
    SastFor(sexpr1, sexpr2, sexpr3, sast_stmt), env;
;;
  