%{ open Ast %}

%token SEMI LPAREN RPAREN LBRACE RBRACE COMMA NEWLINE
%token DOT
%token ARROW MATCH
%token LBRACKET RBRACKET
%token PLUS MINUS TIMES DIVIDE ASSIGN
%token EQ NEQ LT LEQ GT GEQ
%token AT
%token AND OR
%token RETURN IF THEN ELSE ELSEIF FOR WHILE END DO
%token FUNCTION ENUM
%token <int> LITERAL
%token <string> STRINGLITERAL
%token <string> ID
%token <bool> BOOLEANLITERAL
%token EOF

%nonassoc NOELSE
%nonassoc ELSE
%nonassoc ARROW MATCH
%nonassoc COMMA
%right ASSIGN
%left OR AND
%left EQ NEQ
%left LT GT LEQ GEQ
%left AT
%left LBRACKET
%left DOT
%left PLUS MINUS
%left TIMES DIVIDE

%start program
%type <Ast.program> program
%type <Ast.enum_decl> enum_decl
    
%%

/* A general note to the instructor on the coding of the rules below, and specifically on concatenating lists:
  I explicitly am choosing to concatenate lists using the list concatenation @, instead of a new head to another list
  and then reversing the list when setting it to its final home
  
  Though the concatenation is not tail-recursive and so less-efficient processing-wise, it makes the code
  easier to understand and debug, as I don't have to debug potential issues about the order of the list that
  results.
  
  We can change this if performance is an issue (but per the notes in class, we don't have to worry about performance 
  too much here
  */


program:
  /* nothing */ { [], [] }
  | program enum_decl { (fst $1 @ [$2] ), snd $1 }
  | program fdecl { fst $1, snd $1 @ [$2] }
  | program NEWLINE { $1 }
;

/*Note - this causes a shift-reduce conflict. We cannot avoid it w/ yacc as per various commentary
  on the web, such as this: http://stackoverflow.com/questions/17114634/yacc-shift-reduce-conflict-when-parse-c-template-arguments
  */
type_id:
  | ID               { SimpleTypeId($1) }
  | ID LT type_id GT { GenericTypeId($1, $3) }
;
  
fdecl:
  | FUNCTION type_id ID LPAREN vdecl_list RPAREN NEWLINE stmt_list END FUNCTION NEWLINE
    { { rettype = $2; fname = $3; params = $5; body = [Block($8)]} }
;

enum_decl:
  | ENUM ID NEWLINE formal_list NEWLINE END ENUM NEWLINE
    { { ename = $2; ids = $4 } }
;

vdecl:
  | type_id ID { { vtype = $1; vname = $2; } }
;

formal_list:
  | ID                   { [$1] }
  | formal_list COMMA ID { $1 @ [$3] }
;

vdecl_list:
  /* nothing */    { [] }
  | vdecl	{ [$1] }
  | vdecl_list COMMA vdecl { $1 @ [$3] }
;

stmt_list:
  /* nothing */  { [] }
  | stmt_list stmt { $1 @ [$2] }
;

stmt:
  | expr NEWLINE { Expr($1) }
  | RETURN expr NEWLINE { Return($2) }
  | IF expr THEN NEWLINE stmt_list elseifs %prec NOELSE END IF NEWLINE { If($2, Block($5), $6, Block([])) }
  | IF expr THEN NEWLINE stmt_list elseifs ELSE NEWLINE stmt_list END IF { If($2, Block($5), $6, Block($9)) }
  | FOR expr SEMI expr SEMI expr DO NEWLINE stmt_list END FOR NEWLINE
    { For($2, $4, $6, Block($9)) }
  | WHILE expr DO NEWLINE stmt_list END WHILE NEWLINE { While($2, Block($5)) }
  | vdecl NEWLINE { VarDecl($1) }
  | vdecl ASSIGN expr NEWLINE { VarDeclAndAssign($1, $3) }
  | NEWLINE { NoOpStmt }
;
            
elseifs:
  /* nothing */ { [] } 
  | elseifs ELSEIF expr THEN NEWLINE stmt_list { $1 @ [($3, Block($6))] }
;

id_list:
  | ID         { [$1] }
  | id_list ID { $1 @ [$2]  }
;

id_array:
  | id_list                  { [$1] }
  | id_array NEWLINE id_list { $1 @ [$3] }
;
  
expr:
  | LITERAL          { IntLiteral($1) }
  | BOOLEANLITERAL    { BooleanLiteral($1) }
  | STRINGLITERAL    { StringLiteral($1) }
  | LBRACE NEWLINE id_array NEWLINE RBRACE    { BoardLiteral($3) }
  | ID	   			 { Id($1) }
  | expr LBRACKET expr COMMA expr RBRACKET { MatrixOp($1, $3, $5) }
  | expr PLUS   expr { Binop($1, Add,   $3) }
  | expr MINUS  expr { Binop($1, Sub,   $3) }
  | expr TIMES  expr { Binop($1, Mult,  $3) }
  | expr DIVIDE expr { Binop($1, Div,   $3) }
  | expr AND expr { Binop($1, And,   $3) }
  | expr OR expr { Binop($1, Or,   $3) }
  | expr EQ     expr { Binop($1, Equal, $3) }
  | expr NEQ    expr { Binop($1, Neq,   $3) }
  | expr LT     expr { Binop($1, Less,  $3) }
  | expr LEQ    expr { Binop($1, Leq,   $3) }
  | expr GT     expr { Binop($1, Greater,  $3) }
  | expr GEQ    expr { Binop($1, Geq,   $3) }
  | expr AT    expr { Binop($1, Concat,   $3) }
  | expr DOT    ID { ClassOp($1, $3) }
  | expr ASSIGN expr   { Assign($1, $3) }
  | ID LPAREN actuals_opt RPAREN { Call($1, $3) }
  | LPAREN expr RPAREN { $2 }
  | expr MATCH expr ARROW vdecl_list { RegexpMatcher($1, $3, $5) }
;

actuals_opt:
    /* nothing */ { [] }
  | actuals_list  { $1 }
;

actuals_list:
    expr                    { [$1] }
  | actuals_list COMMA expr { $1 @ [$3] }
;
