open Ast;;
open Printf;;
open Common;;

module StringMap = Map.Make(String);;

(****************
 * Root type definitions for the environment
 ***************)

type symbol_table = { symbol_map : reference StringMap.t }
;;

type env =
  {symbols : symbol_table;  (* store all symbols in this map for quick access, incl. symbols for function names *)
   func_map : func_decl StringMap.t;  (* map for the function metadata *)
   type_map : type_id StringMap.t;  (* map for the type metadata *)
   cur_func_name : string;  (* The current function that is being processed for this environment. Used during the code generation *)
  }
;;

let create_new () =
  { func_map = StringMap.empty; type_map = StringMap.empty; symbols = { symbol_map = StringMap.empty }; cur_func_name = ""; }
;;


(****************
 * Lookup methods to search the environment
 ***************)

let lookup_function env id =
  StringMap.find id env.func_map
;;

let lookup_type env id =
  StringMap.find id env.type_map
;;

let lookup_symbol env id =
  let rec lookup_symbol_table tab id =
    if StringMap.mem id tab.symbol_map
    then StringMap.find id tab.symbol_map
    else raise(CompileException(sprintf "Could not find id reference [%s] inside code for function [%s]" id env.cur_func_name))
  in lookup_symbol_table env.symbols id
;;


(****************
 * Methods to facilitate type lookups to resolve type names to a unified
 * form (e.g. converting enums classes like say MyEnum to enum<MyEnum>
 ***************)

let rec resolve_type_id env typeid = match typeid with
  | SimpleTypeId(t) -> lookup_type env t
  | GenericTypeId(gentype, paramtype) ->
    (* This is to handle the case where the type is already property resolved, e.g.
       board<enum<myenum2>> should not resolve to board<enum<enum<myenum2>>>
       This won't scale well for "pure" generics, but for our limited case, this should work
     *)
    let resolved_type = (resolve_type_id env paramtype) in
    if (type_id_to_string typeid) = (type_id_to_string resolved_type)
    then typeid
    else GenericTypeId(gentype, resolved_type);
;;

let resolve_var_decl env v =
  { vtype = resolve_type_id env v.vtype; vname = v.vname }
;;
  

(****************
 * Processing-specific methods to modify some state of the environment
 * while functions are being processed
 ***************)

let assign_cur_func env func_name=
  { func_map = env.func_map; type_map = env.type_map; symbols = env.symbols; cur_func_name = func_name }
;;


(****************
 * Methods to add symbols and other information to the environment
 ***************)

(* The core method to add symbols to an environment *)
let add_symbol env symbol p_ref_type =
  let add_symbol_to_table table symbol p_ref_type =
    if StringMap.mem symbol table.symbol_map
    then raise (CompileException ("Symbol already exists: " ^ symbol))
    else {symbol_map = StringMap.add symbol { ref_type=p_ref_type; code_name="var" ^ get_next_tmp_id () } table.symbol_map;}
  in
  {func_map = env.func_map;
   type_map = env.type_map;
   symbols = add_symbol_to_table env.symbols symbol p_ref_type;
   cur_func_name = env.cur_func_name;
  }
;;

let add_var_decl env v =
  let v = resolve_var_decl env v in
  add_symbol env v.vname v.vtype
;;

let add_type env typename typeval =
  let env = add_symbol env typename type_type in
  {func_map = env.func_map;
   type_map = StringMap.add typename typeval env.type_map;
   symbols = env.symbols;
   cur_func_name = env.cur_func_name;
  }
;;


(*
   For enums, we register the enum itself as a type class, and all the enum
   elements inside of it as symbols in the global environment
 *)
let add_enum env enum =
  let enum_type = GenericTypeId("enum", (SimpleTypeId(enum.ename))) in
  let env = add_type env enum.ename enum_type in
  let env = List.fold_left (fun env enumid -> add_symbol env enumid enum_type) env enum.ids in
  env
;;

(*
   We always resolve the type information prior to adding something to the
   environment, as this makes the code generation logic easier to handle
 *)
let add_func env func =
  let func =
    {rettype = resolve_type_id env func.rettype;
     fname = func.fname;
     params = List.map (fun p -> resolve_var_decl env p) func.params;
     body = func.body} in
  let env = add_symbol env func.fname function_type in
  {func_map = StringMap.add func.fname func env.func_map;
   type_map = env.type_map;
   symbols = env.symbols;
   cur_func_name = env.cur_func_name;
  }
;;
