open Ast;;
open Printf;;
open Env;;
open Common;;
open Saster;;
open Codegenerator;;

module StringMap = Map.Make(String);;


let initialize_env_with_globals enums funcs =
  let env = Env.create_new () in
  (* register some default types *)
  let env = Env.add_type env "int" int_type in
  let env = Env.add_type env "string" string_type in
  let env = Env.add_type env "boolean" boolean_type in
  (* register some default functions *)
  let env = Env.add_func env {fname = "print"; rettype = int_type; params = [{vtype=string_type; vname="str"}]; body = []} in
  let env = Env.add_func env {fname = "printLine"; rettype = int_type; params = [{vtype=string_type; vname="str"}]; body = []} in
  let env = Env.add_func env {fname = "fail"; rettype = int_type; params = [{vtype=string_type; vname="str"}]; body = []} in
  let env = Env.add_func env {fname = "readLine"; rettype = string_type; params = []; body = []} in
  (* register the functions from our own code *)
  let env = (List.fold_left Env.add_enum) env enums in
  let env = (List.fold_left Env.add_func) env funcs in
  env
;;


let process_enum out env enum =
  List.iter
    (fun id ->
      let idref = (Env.lookup_symbol env id) in
      fprintf out "const char *%s = \"%s\";\n" idref.code_name id
    ) enum.ids
;;


let func_signature_for_prototype func =
  let var_strs = List.map (fun var -> sprintf "%s %s" (type_to_codetype var.vtype) var.vname ) func.params in
  sprintf "%s %s(%s)" (type_to_codetype func.rettype) func.fname (String.concat ", " var_strs);
;;


let func_signature env func =
  let var_strs = List.map
    (fun var ->
      let idref = Env.lookup_symbol env var.vname in
      sprintf "%s %s" (type_to_codetype var.vtype) idref.code_name
    ) func.params in
  sprintf "%s %s(%s)" (type_to_codetype func.rettype) func.fname (String.concat ", " var_strs);
;;

let process_function out env func =
  let env = Env.assign_cur_func env func.fname in
  let env = List.fold_left
    (fun env param -> Env.add_var_decl env param)
      env func.params in
  fprintf out "\n";
  fprintf out "%s\n" (func_signature env func);
  
  (* Now we emit the code for the functions. We will have special handling here for those functions that
     have c-code already defined
   *)
  match func.fname with
    | "print" -> fprintf out "{ printf (\"%s\", %s); return 0; } // special function\n" "%s" (Env.lookup_symbol env "str").code_name
    | "printLine" -> fprintf out "{ printf (\"%s\\n\", %s); return 0; } // special function\n" "%s" (Env.lookup_symbol env "str").code_name
    | "fail" -> fprintf out "{ printf (\"%s\\n\", %s); printf (\"Exiting due to failure\"); exit(-1);  return -1;} // special function\n" "%s" (Env.lookup_symbol env "str").code_name
    | "readLine" -> fprintf out "{
        char *myline;
        myline = (char *) malloc(MAX_STRING_SIZE + 1);
        size_t nbytes = MAX_STRING_SIZE;
        getline(&myline, &nbytes, stdin);
        return myline;
        // TODO handle memory cleanup
    } // special function\n"
    | fname ->
      match func.body with
        | [block] ->
          (match block with
            | Block(stmts) ->
              let sast_block, env = to_sast_stmt env block in
              fprintf out "{\n";
              print_sast_stmt out sast_block;
              fprintf out "printf(\"Runtime exception - no return defined in this path of the code\"); exit(-2);}\n\n";
            | _ -> raise (CompileException ("Function must only start with a block"))
          )
        | _ -> raise (CompileException ("Function must only start with a block"))
;;


let process_program program_name env enums funcs =
  let funcs = (List.map snd (StringMap.bindings env.func_map)) in
  let c_file_prefix = "zzzz_"^program_name in
  let c_file = c_file_prefix ^ ".c" in
  let out_file = program_name ^ ".out" in
  let out = open_out c_file in
  
  (* First, write out the contents of the code to a temporary file *)
  fprintf out "#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <regex.h>
#include <string.h>
#define MAX_STRING_SIZE 1024
#define MAX_MATCHING_GROUPS 5

struct board {
	const char** boardarray;
    int rowlength;
    int collength;
};

// credits on this code go to: http://stackoverflow.com/a/11864144
int compare_regexp (const char* source, const char* regexString, char groups[][MAX_STRING_SIZE]) {
	regex_t regexCompiled;
	regmatch_t groupArray[MAX_MATCHING_GROUPS];
	int foundGroups = 0;

	if (regcomp(&regexCompiled, regexString, REG_EXTENDED)) {
		// Could not compile regular expression
		return -1;
	};

	if (regexec(&regexCompiled, source, MAX_MATCHING_GROUPS, groupArray, 0) == 0) {
		unsigned int g = 0;
		for (g = 0; g < MAX_MATCHING_GROUPS; g++) {
			if (groupArray[g].rm_so == (size_t)-1)
			break;  // No more groups

			foundGroups++;
			
			char sourceCopy[strlen(source) + 1];
			strcpy(sourceCopy, source);
			sourceCopy[groupArray[g].rm_eo] = 0;
			strncpy(groups[g], sourceCopy + groupArray[g].rm_so, 1024);
		}
	}
	regfree(&regexCompiled);
	
	if ( foundGroups > 0) {
		foundGroups--;  // this is because one of the groups in the method is the full matched string itself
		// if no groups were specified in the regexp, then this still returns 1. Hence, we force it to 0,
		// as no actual groups were found
	}
	return foundGroups;
}
";
  
  List.iter (fun func -> fprintf out "%s;\n" (func_signature_for_prototype func)) funcs;
  fprintf out "\n";
  List.iter (fun enum -> process_enum out env enum) enums;
  fprintf out "\n";
  List.iter (fun func -> process_function out env func) funcs;
  close_out out;
  
  (* Now compile that C code to its final binary output *)
  (* Note that we throw a CriticalCompileException here if we fail. We segregate this from the regular
     CompileException, as a compilation failure in C code implies that we missed out on some checks
     here in the Ocaml code, and ideally our Ocaml code should always generate valid C code (and if it
     can't, it should throw the error before even trying to compile) 
   *)
  let result = Unix.system (sprintf "/usr/bin/gcc %s -o %s" c_file out_file) in
  match result with
    | Unix.WEXITED(exit_code) ->
      (match exit_code with
        | 0 -> printf "Compile executed successfully for program %s\n" program_name;
        | _ -> raise(CriticalCompileException(sprintf "Compile of c-code failed and returned with exit code %d" exit_code));
      )
    | _ -> ignore(raise(CriticalCompileException("Unexpected process status")));
  ();
;;


(* Here, we generate some functions specific for each enum 
   So far, we have:
   1) parseEnum_<enumName> to return an enum value given a string
   2) printing out a board of enums
   3) printing out a row of enums on a board *)
let get_enum_functions enum =
  let signature = sprintf "function %s parseEnum_%s(string str)\n" enum.ename enum.ename in
  let body = List.fold_left
    (fun l id ->
      let ifbody = sprintf "if str == \"%s\" then\nreturn %s\nend if\n" id id in
      ifbody :: l
    ) [] enum.ids in
  let body_string = String.concat "\n" body in
  let end_sig = "end function\n" in
  let printBoardFunc = sprintf "function int printBoard_%s(board<%s> myboard)
  	int row
  	for row = 1; row <= myboard.rowlength; row = row + 1 do
  		printBoardRow_%s(myboard, row)
  	end for
  	return 0
  end function
  
function int printBoardRow_%s(board<%s> myboard, int row)
  	int col
  	for col = 1; col <= myboard.collength; col = col + 1 do
  		print(myboard[row, col] @ \" \")
  	end for
  	printLine(\"\")
	string dummy = \"\" /* This is only here to work around Eclipse syntax highlighting issue.... */
  	return 0
  end function
  " enum.ename enum.ename enum.ename enum.ename enum.ename in 
  
  signature ^ body_string ^ end_sig ^ printBoardFunc
;;

let parse_program_from_lexbuf lexbuf =
  let enums, funcs =
    try Parser.program Scanner.token lexbuf
    with Parsing.Parse_error ->
      let region_text =
        let start_index = max (lexbuf.Lexing.lex_start_pos - 20) 0 in
        let length =
          if start_index + 40 > String.length lexbuf.Lexing.lex_buffer
          then (String.length lexbuf.Lexing.lex_buffer) - start_index
          else 40
         in
        String.sub lexbuf.Lexing.lex_buffer start_index length
      in 
      (* need a separate printf from the raise-ParseException call, as Ocaml seems to cut off the error message when shown via the raise call *)
      printf "Found parsing error at character #%d on char %s in region\n[%s]\n"
        lexbuf.Lexing.lex_start_pos
        (Char.escaped (String.get lexbuf.Lexing.lex_buffer lexbuf.Lexing.lex_start_pos))
        region_text
      ;
      raise(ParseException("Found parsing error - see previous log message"))
  in
  enums, funcs
;;
  
let parse_program_from_string program_text =
  parse_program_from_lexbuf (Lexing.from_string program_text)
;;
  
let compile_program_from_lexbuf program_name lexbuf =
  let enums, funcs = parse_program_from_lexbuf lexbuf in
  let enumfuncs = List.fold_left
    (fun l enum ->
      let _ , funcs = parse_program_from_string (get_enum_functions enum) in
      funcs @ l
    ) [] enums in
  let funcs = funcs @ enumfuncs in
  let env = initialize_env_with_globals enums funcs in
  let main_method =
    try Env.lookup_function env "main"
    with Not_found -> raise(CompileException("Program must define a \"main\" method"))
  in
  if List.length main_method.params <> 0
  then raise(CompileException("\"main\" methods must not have any args defined"));
  process_program program_name env enums funcs;
;;

let compile_program_from_string program_name program_text =
  compile_program_from_lexbuf program_name (Lexing.from_string program_text);
;;
