open Ast;;
open Printf;;


(* Stored as an array so that it can be mutable *)
let sast_counter = [| 0 |];;

(* This is to get a new symbol for use in the 3-address code generation,
   essentially the equivalent of getting new registers to store addresses in
   
   We use an array to store this so that we have a globally-accessed variable for
   this. It proved to be inconvenient at first to put this as a field on environment
   and so the global variable proved to be flexible. We can change this one day if
   we choose
 *)
let get_next_tmp_id () =
  sast_counter.(0) <- (sast_counter.(0) + 1);
  "tmp" ^ (string_of_int sast_counter.(0));
;;


(************
 * Type-helper methods
 *)
let type_type = SimpleTypeId("type");;
let function_type = SimpleTypeId("function");;
let string_type = SimpleTypeId("string");;
let int_type = SimpleTypeId("int");;
let boolean_type = SimpleTypeId("boolean");;

let rec type_id_to_string = function
  | SimpleTypeId(id) -> id;
  | GenericTypeId(gentype,paramtype) -> sprintf "%s<%s>" gentype (type_id_to_string paramtype);
;;

let type_to_base_type t = match t with
  | SimpleTypeId(id) -> id;
  | GenericTypeId(gentype, _) -> gentype
;;

let type_is_of_type t2name t1 = (type_to_base_type t1) = t2name;
;;

let rec type_to_generic_subtype = function
  | SimpleTypeId(id) -> raise(CompileException("Cannot access generic subtype of SimpleType"));
  | GenericTypeId(gentype,paramtype) -> paramtype;
;;




(****
 * The two methods for calculationg the array index.
 * One is for Ocaml to the code, and one is for the c code itself
 * We keep them both here so that folks can easily see the logic in both places
 * 
 * Note - we store as a regulary array in C, and not a 2D array, as it proved to be much harder
 * to get 2d arrays to work on the stack in C.
 * *****)
let calc_matrix_to_array_index row col numcols =
  row * numcols + col
;;

(* We use the array reference directly for thee reference, as opposed to generating a new address via get_next_tmp_id, 
   as I could not come up w/ a 3-address code representation for array references that worked for
   both retrieval and storage. This way works out. In any case, the indexes themselves are used in a 3-address-like manner
 *)
let calc_matrix_to_array_index_for_c_code board row col =
  (* Use %s - 1 here as we index matrices in the code by 1-index, whereas C does it 0-indexed *)
  sprintf "%s.boardarray[(%s - 1) * %s.collength + (%s - 1)]" board.eref row.eref board.eref col.eref
;;
