open Ast;;
open Printf;;
open Common;;


let type_to_codetype = function
  | SimpleTypeId(simp_type) ->
    (match simp_type with
      | "boolean" -> "bool"
      | "string" -> "const char*"
      | any -> any
    )
  | GenericTypeId(gentype, paramtype) ->
    (match gentype with
      | "board" -> "struct board"
      | "enum" -> "const char*"
      | _ -> raise(CompileException("No other generic type allowed other than board or enum: " ^ gentype))
    )
;;


let op_to_string = function
  | Add -> "+"
  | Sub -> "-"
  | Mult -> "*"
  | Div -> "/"
  | Equal -> "=="
  | Neq -> "!="
  | Less -> "<"
  | Leq -> "<="
  | Greater -> ">"
  | Geq -> ">="
  | Or -> "||"
  | And -> "&&"
  | Concat -> raise(CompileException("concat is not implemented via this methodology"))
;;

  

(* emits the format value to use for the printf function on the C side *)
let type_to_print_format_for_concat sexpr =
  if type_is_of_type "int" sexpr.etype then "%d", sexpr.eref
  else if type_is_of_type "string" sexpr.etype then "%s", sexpr.eref
  else if type_is_of_type "enum" sexpr.etype then "%s", sexpr.eref
  else if type_is_of_type "boolean" sexpr.etype then "%s", (sprintf "(%s ? \"true\" : \"false\")" sexpr.eref)
  else raise(CompileException("No other type than string or int supported for concatenation:" ^ (type_id_to_string sexpr.etype)));
;;



(*
   print out the 3-address code to generate the values for a particular expression
   You will note that some of the blocks won't do anything. This is because the eref value itself already
   has the full informatio to represent the value (e.g. the literal in code, or the variable name)
   
   For the others, it is the responsibility of this method to emit the code to set the value for the
   sexpr.eref value that is passed into here
 *)
let rec print_sast_expr out sexpr =
  match sexpr.e with
    | SastIntLiteral(l) -> ();
    | SastStringLiteral(l) -> ();
    | SastBooleanLiteral(l) -> ();
    | SastBoardLiteral(numrows, numcols, board_elems) ->
      fprintf out "const char* %sArray[%d];\n" sexpr.eref (numrows * numcols);
      for row = 0 to numrows - 1 do
        for col = 0 to numcols - 1 do
          let elem = List.nth (List.nth board_elems row) col in
          fprintf out "%sArray[%d] = %s;\n" sexpr.eref (calc_matrix_to_array_index row col numcols) elem.eref;
        done
      done;
      fprintf out "struct board %s;\n" sexpr.eref;
      fprintf out "%s.boardarray = %sArray;\n" sexpr.eref sexpr.eref;
      fprintf out "%s.rowlength = %d;\n" sexpr.eref numrows;
      fprintf out "%s.collength = %d;\n" sexpr.eref numcols;
    | SastId(id) -> ();
    | SastMatrixOp(board, row, col) -> ()  (* the code earlier already did the 3-address code encompassing *)
    | SastClassOp (classexpr, member_name) ->
      if not (type_is_of_type "board" classexpr.etype)
      then raise(CompileException("Dot operator can only be used on classes (e.g. board)"));
      (match member_name with
        | "rowlength" -> fprintf out "%s %s = %s.rowlength;\n" (type_to_codetype sexpr.etype) sexpr.eref classexpr.eref;
        | "collength" -> fprintf out "%s %s = %s.collength;\n" (type_to_codetype sexpr.etype) sexpr.eref classexpr.eref;
        | _ -> raise(CompileException("No other field names supported for the board class"))
      )
    | SastRegexpMatcher(input, regexp, vars, var_refs) ->
      print_sast_expr out input;
      print_sast_expr out regexp;
      fprintf out "bool %s;\n" sexpr.eref;
      fprintf out "char %s_groups[MAX_MATCHING_GROUPS][MAX_STRING_SIZE];\n" sexpr.eref;
      List.iter
        (fun var_ref -> match (type_to_base_type var_ref.ref_type) with
          | "string" -> fprintf out "char* %s; // local regexp var\n" var_ref.code_name;
          | "int" -> fprintf out "int %s; // local regexp var\n" var_ref.code_name;
          | "enum" -> fprintf out "%s %s; // local regexp var\n" (type_to_codetype var_ref.ref_type) var_ref.code_name;
          | _ -> raise(CompileException("Do not support binding regexp params to any other type than string, int, enums"));
        ) var_refs;
      fprintf out "int %s_val = compare_regexp(%s, %s, %s_groups);\n" sexpr.eref input.eref regexp.eref sexpr.eref;
      fprintf out "if (%s_val > 0) {\n" sexpr.eref;
      fprintf out "%s = true;\n" sexpr.eref;
      for refnum = 0 to (List.length var_refs) - 1 do
        let var_ref = List.nth var_refs refnum in
        match (type_to_base_type var_ref.ref_type) with
          | "string" -> fprintf out "%s = %s_groups[%d + 1];\n" var_ref.code_name sexpr.eref refnum;
          | "int" -> fprintf out "%s = atoi(%s_groups[%d + 1]);\n" var_ref.code_name sexpr.eref refnum;
          | "enum" -> fprintf out "%s = parseEnum_%s(%s_groups[%d + 1]);\n" var_ref.code_name (type_id_to_string (type_to_generic_subtype var_ref.ref_type)) sexpr.eref refnum;
          | _ -> raise(CompileException("Do not support binding regexp params to any other type than string, int, enums"));
      done;
      fprintf out "} else if (%s_val == 0) {\n" sexpr.eref;
      fprintf out "%s = false;\n" sexpr.eref;
      fprintf out "} else {\n";
      fprintf out "printf(\"Regexp did not compile. This is a runtime exception\"); exit(-1);\n";
      fprintf out "}\n";
    | SastBinop(e1, op, e2) ->
      print_sast_expr out e1;
      (match op with
        | Concat ->
          (print_sast_expr out e2;
           fprintf out "char %s[1001];\n" sexpr.eref;
           let e1printformat, e1val = type_to_print_format_for_concat e1 in
           let e2printformat, e2val = type_to_print_format_for_concat e2 in
           fprintf out "snprintf(%s, 1000, \"%s%s\", %s, %s);\n" sexpr.eref e1printformat e2printformat e1val e2val
          );
        | Equal ->
          print_sast_expr out e2;
          if type_is_of_type "string" e1.etype
          then
            (fprintf out "int %sCmpVal = strcmp(%s, %s);\n" sexpr.eref e1.eref e2.eref;
             fprintf out "bool %s = %sCmpVal == 0;\n" sexpr.eref sexpr.eref;
            )
          else fprintf out "%s %s = %s %s %s;\n" (type_to_codetype sexpr.etype) sexpr.eref e1.eref (op_to_string op) e2.eref;
        | Neq ->
          print_sast_expr out e2;
          if type_is_of_type "string" e1.etype
          then
            (fprintf out "int %sCmpVal = strcmp(%s, %s);\n" sexpr.eref e1.eref e2.eref;
             fprintf out "bool %s = %sCmpVal != 0;\n" sexpr.eref sexpr.eref;
            )
          else fprintf out "%s %s = %s %s %s;\n" (type_to_codetype sexpr.etype) sexpr.eref e1.eref (op_to_string op) e2.eref;
        | And ->
          (* For And and Or, we take care to do lazy-evaluation, i.e. to ensure that we only evaluate the code
             generated for the second expression if it is warranted. For And, we break off early if
             the first arg is false, and for Or, we break off early if it is true *)
          fprintf out "%s %s;\n" (type_to_codetype sexpr.etype) sexpr.eref;
          fprintf out "if (%s == false) { %s = %s; } " e1.eref sexpr.eref e1.eref;
          fprintf out "else {\n";
          print_sast_expr out e2;
          fprintf out "%s = %s %s %s;\n" sexpr.eref e1.eref (op_to_string op) e2.eref;
          fprintf out "}\n";
        | Or ->
          fprintf out "%s %s;\n" (type_to_codetype sexpr.etype) sexpr.eref;
          fprintf out "if (%s == true) { %s = %s; } " e1.eref sexpr.eref e1.eref;
          fprintf out "else {\n";
          print_sast_expr out e2;
          fprintf out "%s = %s %s %s;\n" sexpr.eref e1.eref (op_to_string op) e2.eref;
          fprintf out "}\n";
        | _ ->
          print_sast_expr out e2;
          fprintf out "%s %s = %s %s %s;\n" (type_to_codetype sexpr.etype) sexpr.eref e1.eref (op_to_string op) e2.eref;
      )
    | SastAssign(e) ->
      print_sast_expr out e;
      fprintf out "%s = %s;\n" sexpr.eref e.eref;
    | SastCall (id, exprs) ->
      let args = String.concat ", " (List.map (fun e -> e.eref) exprs) in
      List.iter (print_sast_expr out) exprs;
      fprintf out "%s %s = %s(%s);\n" (type_to_codetype sexpr.etype) sexpr.eref id args;
;;


let rec print_sast_stmt out input_stmt = match input_stmt with
  | SastNoOpStmt -> ()
  | SastBlock(stmts) ->
    fprintf out "{\n";
    List.iter (print_sast_stmt out) stmts;
    fprintf out "}\n";
  | SastVarDecl(vname, idref) ->
    fprintf out "%s %s;  // for var %s\n" (type_to_codetype idref.ref_type) idref.code_name vname;
  | SastVarDeclAndAssign(vname, idref, expr) ->
    print_sast_expr out expr;
    fprintf out "%s %s = %s;  // for var %s\n" (type_to_codetype idref.ref_type) idref.code_name expr.eref vname;
  | SastExpr(e) ->
    print_sast_expr out e;
  | SastReturn(e) ->
    print_sast_expr out e;
    fprintf out "return %s;\n" e.eref;
  | SastIf(conds, s2) ->
    let iflabel = get_next_tmp_id () in
    fprintf out "// Start of if-else block for %s\n" iflabel;
    List.iter
      (fun (expr, stmt) ->
        print_sast_expr out expr;
        fprintf out "if (%s) {" expr.eref;
        print_sast_stmt out stmt;
    	fprintf out "goto %s;\n" iflabel;
        fprintf out "}\n";
      ) conds;
    print_sast_stmt out s2;
    fprintf out "%s:;\n" iflabel;
    fprintf out "// End of if-else block for %s\n" iflabel;
  | SastWhile(sexpr, sast_stmt) ->
    (* The order of operations of the while loop is modeled similarly to what was shown in class,
       where the condition statement was placed "after" the body of the while in the generated code,
       so that we have one less jump/goto method to execute *)
    let label = get_next_tmp_id () in
    fprintf out "goto %sCond;\n" label;
    fprintf out "%sStart:;\n" label;
    print_sast_stmt out sast_stmt;
    fprintf out "%sCond:;\n" label;
    print_sast_expr out sexpr;
    fprintf out "if (%s) {" sexpr.eref;
    fprintf out "goto %sStart;\n" label;
    fprintf out "}\n";
  | SastFor(e1, e2, e3, s) ->
    (* See the comment in the SastWhile block around how the condition/body of the for is organized.
       We do it in a similar way here *)
    let label = get_next_tmp_id () in
    print_sast_expr out e1;
    fprintf out "goto %sCond;\n" label;
    fprintf out "%sStart:;\n" label;
    print_sast_stmt out s;
    print_sast_expr out e3;
    fprintf out "%sCond:;\n" label;
    print_sast_expr out e2;
    fprintf out "if (%s) {" e2.eref;
    fprintf out "goto %sStart;\n" label;
    fprintf out "}\n";
;;

