open Printf

exception CompileException of string;;
exception ParseException of string;;
exception CriticalCompileException of string;;


  
type op = Add | Sub | Mult | Div | Equal | Neq | Less | Leq | Greater | Geq | Concat | And | Or
;;

type type_id =
  | SimpleTypeId of string
  | GenericTypeId of string * type_id
;;

type var_decl = { vtype : type_id; vname : string; }
;;

type expr =
  IntLiteral of int
  | StringLiteral of string
  | BooleanLiteral of bool
  | BoardLiteral of string list list
  | Id of string
  | Binop of expr * op * expr
  | MatrixOp of expr * expr * expr
  | ClassOp of expr * string
  | Assign of expr * expr
  | Call of string * expr list
  | RegexpMatcher of expr * expr * var_decl list  (* both expressions should evaluate to strings *)
;;

type stmt =
  Block of stmt list
  | Expr of expr
  | Return of expr
  | If of expr * stmt * (expr * stmt) list * stmt
  | For of expr * expr * expr * stmt
  | While of expr * stmt
  | VarDecl of var_decl
  | VarDeclAndAssign of var_decl * expr
  | NoOpStmt
;;

type func_decl =
  {rettype : type_id;
   fname : string;
   params : var_decl list;
   body : stmt list; (* TODO make the body a stmt so that we can limit it to be a block early on *)
  }
;;

type enum_decl = {
    ename : string;
    ids : string list;
  }
;;


type program = enum_decl list * func_decl list
;;



(* Corresponds to a reference to be used in the 3-address code generation
   The code_name would represent some snippet of C (whether a variable name 
   or a literal) that can directly be plugged in anywhere
 *)
type reference =
  {ref_type : type_id;
   code_name : string;
  }
;;


(***********
 * Semantically-analyzed syntax tree (i.e. SAST data structures)
 *)
type sast_expr = { e : sast_expr_detail; etype : type_id; eref : string }
and sast_expr_detail =
  SastIntLiteral of int
  | SastStringLiteral of string
  | SastBooleanLiteral of bool
  | SastBoardLiteral of int * int * sast_expr list list
  | SastId of string
  | SastMatrixOp of sast_expr * sast_expr * sast_expr
  | SastBinop of sast_expr * op * sast_expr
  | SastClassOp of sast_expr * string
  | SastAssign of sast_expr
  | SastCall of string * sast_expr list
  | SastRegexpMatcher of sast_expr * sast_expr * var_decl list * reference list  (* both expressions should evaluate to strings *)
;;
  

type sast_stmt =
  SastBlock of sast_stmt list
  | SastExpr of sast_expr
  | SastReturn of sast_expr
  | SastIf of (sast_expr * sast_stmt) list * sast_stmt
  | SastFor of sast_expr * sast_expr * sast_expr * sast_stmt
  | SastWhile of sast_expr * sast_stmt
  | SastVarDecl of string * reference
  | SastVarDeclAndAssign of string * reference * sast_expr
  | SastNoOpStmt
;;


