--Editor: Zeshi Wang
--Data: 2013

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity de2_wm8731_audio is
port (
    clk : in std_logic;       --  Audio CODEC Chip Clock AUD_XCK (18.43 MHz)
    reset_n : in std_logic;
    data : in std_logic_vector(7 downto 0) := "00000000";
	 write      : in std_logic;
	 chipselect : in std_logic;

    -- Audio interface signals
    AUD_ADCLRCK  : out  std_logic;   --    Audio CODEC ADC LR Clock
    AUD_ADCDAT   : in   std_logic;   --    Audio CODEC ADC Data
    AUD_DACLRCK  : out  std_logic;   --    Audio CODEC DAC LR Clock
    AUD_DACDAT   : out  std_logic;   --    Audio CODEC DAC Data
    AUD_BCLK     : inout std_logic  --    Audio CODEC Bit-Stream Clock
	
  );
end  de2_wm8731_audio;

architecture rtl of de2_wm8731_audio is     

    signal lrck : std_logic;
    signal bclk : std_logic;
    signal xck  : std_logic;
    
    signal lrck_divider : unsigned(15 downto 0); 
    signal bclk_divider : unsigned(11 downto 0);
    
    signal set_bclk : std_logic;
    signal set_lrck : std_logic;
    signal clr_bclk : std_logic;
    signal lrck_lat : std_logic;
    
    signal shift_out : unsigned(15 downto 0);
     
	 signal start_audio : std_logic_vector(3 downto 0);
	 type count_type is array(0 to 3) of unsigned(11 downto 0);
	 signal count           : count_type;
	 --signal ram_address : count_type;
	 signal data_in : std_logic_vector(15 downto 0);
	 signal ram_address : std_logic_vector(13 downto 0);


component  sound_rom IS
	PORT
	(
		address		: IN STD_LOGIC_VECTOR (13 DOWNTO 0);
		clock		: IN STD_LOGIC  := '1';
		q		: OUT STD_LOGIC_VECTOR (15 DOWNTO 0)
	);
END component;


begin
	
	
	ROM  : sound_rom port map(
		address		=>  ram_address,
		clock		=>  clk,
		q		   =>  data_in
	);
  
    -- LRCK divider 
    -- Audio chip main clock is 18.432MHz / Sample rate 48KHz
    -- Divider is 18.432 MHz / 48KHz = 192 (X"C0")
    -- Left justify mode set by I2C controller
	 
 

  
  --Set start_audio signal
  process (clk)
  begin
    if rising_edge(clk) then
	     if reset_n = '0' then 
           start_audio <= (others => '0');
	     else 
				for I in 0 to 3 loop
					if (write = '1' and chipselect = '1' and data(I) = '1') then
						start_audio(I) <= '1';
					elsif count(I) = x"08ae" then
						start_audio(I) <= '0';
					end if;
				end loop;
		 end if;
	 end if;
  end process;
  
  
  process (clk)
  begin
    if rising_edge(clk) then
      if reset_n = '0' then 
        lrck_divider <= (others => '0');
      elsif lrck_divider = X"08FF"  then        -- "C0" minus 1
        lrck_divider <= X"0000";
      else 
        lrck_divider <= lrck_divider + 1;
      end if;
    end if;   
  end process;

  process (clk)
  begin
    if rising_edge(clk) then      
      if reset_n = '0' then 
        bclk_divider <= (others => '0');
      elsif bclk_divider = X"08F" or set_lrck = '1'  then  
        bclk_divider <= X"000";
      else 
        bclk_divider <= bclk_divider + 1;
      end if;
    end if;
  end process;

  set_lrck <= '1' when lrck_divider = X"08FF" else '0';
    
  process (clk)
  begin
    if rising_edge(clk) then
      if reset_n = '0' then
        lrck <= '0';
      elsif set_lrck = '1' then 
        lrck <= not lrck;
      end if;
    end if;
  end process;
  
process(clk)
    begin
      if rising_edge(clk) then
        lrck_lat <= lrck;
      end if;
  end process;
  
    
  -- BCLK divider
  set_bclk <= '1' when bclk_divider(11 downto 0) = X"047" else '0';
  clr_bclk <= '1' when bclk_divider(11 downto 0) = X"08F" else '0';
  
  process (clk)
  begin
    if rising_edge(clk) then
      if reset_n = '0' then
        bclk <= '0';
      elsif set_lrck = '1' or clr_bclk = '1' then
        bclk <= '0';
      elsif set_bclk = '1' then 
        bclk <= '1';
      end if;
    end if;
  end process;

  -- Audio data shift output
 
 process (clk)
  begin
    if rising_edge(clk) then
      if reset_n = '0' then
        shift_out <= (others => '0');
		elsif (start_audio(0) or start_audio(1) or start_audio(2) or start_audio(3)) = '1' then
        if set_lrck = '1' then 
            shift_out <= unsigned(data_in);		 
		  elsif clr_bclk = '1' then 
				shift_out <= shift_out (14 downto 0) & '0';
		  end if;
		else 
		   shift_out <= (others => '0');
		end if;
			
    end if;   
  end process;
  

    -- Audio outputs
    
    AUD_ADCLRCK  <= lrck;          
    AUD_DACLRCK  <= lrck;          
    AUD_DACDAT   <= shift_out(15); 
    AUD_BCLK     <= bclk;          

    -- ram address counter
	 -- counter for ball_hit audio
-- Update count
	process(clk)      
    begin
      if rising_edge(clk) then
        if reset_n = '0' then 
            for I in 0 to 3 loop
					count(I) <= (others => '0');
				end loop;
        elsif lrck_lat = '1' and lrck = '0'  then  
				for I in 0 to 3 loop
					if count(I) = x"8ae" then  
						count(I) <= (others => '0');
					elsif (start_audio(I) = '1') then
						count(I) <= count(I) + 1;
					end if;
					
				end loop;
			 
        end if;
      end if;
    end process;

	--Update Address
	process(clk)
	begin
		if rising_edge(clk) then
			if reset_n = '0' then
				ram_address <= (others => '0');
			elsif start_audio(0) = '1' then
				ram_address(13 downto 2) <= std_logic_vector(count(0));
				ram_address(1 downto 0) <= "00";
			elsif start_audio(1) = '1' then
				ram_address(13 downto 2) <= std_logic_vector(count(1));
				ram_address(1 downto 0) <= "01";
			elsif start_audio(2) = '1' then
				ram_address(13 downto 2) <= std_logic_vector(count(2));
				ram_address(1 downto 0) <= "10";
			elsif start_audio(3) = '1' then
				ram_address(13 downto 2) <= std_logic_vector(count(3));
				ram_address(1 downto 0) <= "11";
			else
				ram_address <= (others => '0');
			end if;
		end if;
	end process;

end architecture;


