-------------------------------------------------------------------------------
--
-- VGA controller for American Pool Game
--
-- Eidtor: Jiawan Zhang
-- Data: 2013
--
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity de2_vga_raster is
  
  port (
  	 --**************************************************
    reset : in std_logic;
    clk50   : in std_logic;                    -- Should be 25.125 MHz

	 read       : in  std_logic;
    write      : in  std_logic;
    chipselect : in  std_logic;
    address    : in  std_logic_vector(6 downto 0); --7 bits 128 addresses
    readdata   : out std_logic_vector(15 downto 0);
    writedata  : in  std_logic_vector(15 downto 0);
	 irq : out std_logic;
	 --**************************************************
    VGA_CLK,                         -- Clock
    VGA_HS,                          -- H_SYNC
    VGA_VS,                          -- V_SYNC
    VGA_BLANK,                       -- BLANK
    VGA_SYNC : out std_logic;        -- SYNC
    VGA_R,                           -- Red[9:0]
    VGA_G,                           -- Green[9:0]
    VGA_B : out std_logic_vector(9 downto 0); -- Blue[9:0]
	 CUERAM_clk : out std_logic;
	 CUERAM_addr : out std_logic_vector(8 downto 0);
	 CUERAM_q : in std_logic_vector(31 downto 0);
	 LEDR : out std_logic_vector (17 downto 0);
	 LEDG : out std_logic_vector (8 downto 0)
    );

end de2_vga_raster;

architecture rtl of de2_vga_raster is
  
  component Ball_Map_rom port (
	address : in std_logic_vector(13 downto 0);
	clock : in std_logic;
	q : out std_logic_vector(1 downto 0)
   );
	end component;
	
	component PoolCue1_rom port (
	address : in std_logic_vector(15 downto 0);
	clock : in std_logic;
	q : out std_logic_vector(3 downto 0)
   );
	end component;
	
	component Level1_rom port (
	address : in std_logic_vector(13 downto 0);
	clock : in std_logic;
	q : out std_logic_vector(4 downto 0)
   );
	end component;
	
	component Words_rom port
	(
		address		: IN STD_LOGIC_VECTOR (12 DOWNTO 0);
		clock		: IN STD_LOGIC  := '1';
		q		: OUT STD_LOGIC_VECTOR (0 DOWNTO 0)
	);
	end component;
	
  --clock
  signal clk : std_logic := '0';
  signal clk_count : unsigned(1 downto 0) := "00";
  
  --Interrupt
  signal Interrupt : std_logic;
  
  -- Video parameters
  --Keep as oringinal
  constant HTOTAL       : integer := 800;
  constant HSYNC        : integer := 96;
  constant HBACK_PORCH  : integer := 48;
  constant HACTIVE      : integer := 640;
  constant HFRONT_PORCH : integer := 16;
  
  constant VTOTAL       : integer := 525;
  constant VSYNC        : integer := 2;
  constant VBACK_PORCH  : integer := 33;
  constant VACTIVE      : integer := 480;
  constant VFRONT_PORCH : integer := 10;
  --End: Keep as oringinal
  
  -- Signals for the video controller
  --Keep as oringinal
  signal Hcount : unsigned(9 downto 0);  -- Horizontal position (0-800)
  signal Vcount : unsigned(9 downto 0);  -- Vertical position (0-524)
  signal EndOfLine, EndOfField : std_logic;

  signal vga_hblank, vga_hsync, vga_vblank, vga_vsync : std_logic;  -- Sync. signals
  --End: Keep as oringinal
 
  --Color table
  type colortable_type is array(1 to 31) of std_logic_vector(7 downto 0);
  constant COLOR_TABLE_R : colortable_type :=(
  "00110010", -- 1. table bed color : limegreen
  "11001101", -- 2. pool cue color !!!!
  "11111111", -- 3. yellow 1
  "11111111", -- 4. yellow 2
  "11111010", -- 5. yellow 3
  "00000000", -- 6. blue 1
  "00000000", -- 7. blue 2
  "10100101", -- 8. blue 3
  "11111111", -- 9. red 1
  "11111111", -- 10. red 2
  "11111111", -- 11. red 3
  "11011100", -- 12. purple 1
  "11101011", -- 13. purple 2
  "11101110", -- 14. purple 3
  "11111111", -- 15. orange 1
  "11111111", -- 16. orange 2
  "11111111", -- 17. orange 3
  "00000000", -- 18. green 1
  "00000000", -- 19. green 2
  "11011010", -- 20. green 3
  "10100000", -- 21. brown 1
  "10110100", -- 22. brown 2
  "11110100", -- 23. brown 3
  "00000000", -- 24. black 1
  "00101000", -- 25. black 2
  "01101001", -- 26. black 3
  "11100000", -- 27. while 1
  "11110101", -- 28. while 2
  "11111111", -- 29. while 3
  "10000000", -- 30. table edge 1: marron
  "01000000" -- 31. table edge 2: oliverdrab
  );
  
  constant COLOR_TABLE_G : colortable_type :=(
  "11001101", -- 1. table bed color : limegreen
  "10000101", -- 2. pool cue color
  "11110000", -- 3. yellow 1
  "11111111", -- 4. yellow 2
  "11111010", -- 5. yellow 3
  "00000000", -- 6. blue 1
  "00000000", -- 7. blue 2
  "11010000", -- 8. blue 3
  "00000000", -- 9. red 1
  "01000101", -- 10. red 2
  "10100000", -- 11. red 3
  "00010101", -- 12. purple 1
  "00000000", -- 13. purple 2
  "10000010", -- 14. purple 3
  "10001100", -- 15. orange 1
  "10100101", -- 16. orange 2
  "11100100", -- 17. orange 3
  "01100100", -- 18. green 1
  "01110110", -- 19. green 2
  "11001101", -- 20. green 3
  "01010010", -- 21. brown 1
  "01101001", -- 22. brown 2
  "10100100", -- 23. brown 3
  "00000000", -- 24. black 1
  "00101000", -- 25. black 2
  "01101001", -- 26. black 3
  "11100000", -- 27. while 1
  "11110101", -- 28. while 2
  "11111111", -- 29. while 3
  "00000000", -- 30. table edge 1: marron
  "10001110" -- 31. table edge 2: oliverdrab
  );

  constant COLOR_TABLE_B : colortable_type :=(
  "00110010", -- 1. table bed color : limegreen
  "00111111", -- 2. pool cue color
  "00000000", -- 3. yellow 1
  "00000000", -- 4. yellow 2
  "10110100", -- 5. yellow 3
  "11001101", -- 6. blue 1
  "11111111", -- 7. blue 2
  "11000110", -- 8. blue 3
  "00000000", -- 9. red 1
  "00000000", -- 10. red 2
  "01111010", -- 11. red 3
  "11001000", -- 12. purple 1
  "11101011", -- 13. purple 2
  "11101110", -- 14. purple 3
  "00000000", -- 15. orange 1
  "00000000", -- 16. orange 2
  "10110101", -- 17. orange 3
  "00000000", -- 18. green 1
  "00000000", -- 19. green 2
  "00110010", -- 20. green 3
  "00101101", -- 21. brown 1
  "00011110", -- 22. brown 2
  "01100000", -- 23. brown 3
  "00000000", -- 24. black 1
  "00101000", -- 25. black 2
  "01101001", -- 26. black 3
  "11100000", -- 27. while 1
  "11110101", -- 28. while 2
  "11111111", -- 29. while 3
  "00000000", -- 30. table edge 1: marron
  "00100011" -- 31. table edge 2: oliverdrab
  );
  
  --VGA Ram
  type ram_type is array(127 downto 0) of
            std_logic_vector(15 downto 0);
  signal VGA_RAM : ram_type;
  signal vga_ram_address : unsigned(6 downto 0);
  
  signal x_coord, y_coord : unsigned(9 downto 0);  --scan positions in screen
  signal level0_active, level1_active, level2_active, level3_active : std_logic;  -- = 1 means certain level has something to print
  
  --Level 0 
  --Pool Table parameters
  constant TABLE1_HSTART : integer := 0;
  constant TABLE1_HEND   : integer := 639;
  constant TABLE1_VSTART : integer := 140;
  constant TABLE1_VEND   : integer := 479;
  
  constant TABLE2_HSTART : integer := 20;
  constant TABLE2_HEND   : integer := 619;
  constant TABLE2_VSTART : integer := 155;
  constant TABLE2_VEND   : integer := 464;
  
  constant TABLE3_HSTART : integer := 32;
  constant TABLE3_HEND   : integer := 607;
  constant TABLE3_VSTART : integer := 166;
  constant TABLE3_VEND   : integer := 453;
  
  signal table1 : std_logic;  -- table areas
  signal table2 : std_logic;  -- table areas
  signal table3 : std_logic;  -- table areas
  -- strength bar
  constant SBAR_HSTART : integer := 197;
  constant SBAR_HEND   : integer := 444;
  constant SBAR_VSTART : integer := 124;
  constant SBAR_VEND   : integer := 138;
  signal SBAR_Strength : integer;

  signal Strength : unsigned(7 downto 0) := "00011100";  --!!from ram  32 levels use 5 bits
  
  signal sbar_h, sbar_v, sbar : std_logic;  -- strength bar areas
  signal sstrengh_h, sstrengh_v, sstrengh : std_logic;  -- strength bar areas
  
  -- Serve line
  constant SERVELINE_H : integer := 172;
  signal serveline_en : std_logic;
  signal serveline : std_logic;
  
  signal colorcode_level0 : unsigned(4 downto 0);
  
  --Level 1
  signal level1_address : unsigned(13 downto 0);
  signal level1_data, level1_data_vga : std_logic_vector(4 downto 0);
  signal level1_vga : std_logic;
  signal pocket_flag1, pocket_flag2 : std_logic_vector(5 downto 0);
  type pocket_eachaddrtype is array(0 to 5) of unsigned(10 downto 0);
  signal pocket_eachaddr : pocket_eachaddrtype;
  
  --Level 2

  type ball_type is array(0 to 195) of unsigned(1 downto 0);
  constant BallMask : ball_type :=(
	"00", "00","00","00","00","01","01","01","01","00", "00","00","00","00",  --1
	"00", "00","00","01","01","01","01","01","01","01", "01","00","00","00",  --2
	"00", "00","01","01","01","10","10","10","10","10", "01","01","00","00",  --3
	"00", "01","01","01","10","10","10","10","10","10", "10","01","01","00",  --4
	"00", "01","01","10","10","10","10","10","11","11", "10","10","01","00",  --5
	"01", "01","01","10","10","10","10","11","11","11", "11","10","01","01",  --6
	"01", "01","01","10","10","10","10","11","11","11", "11","10","01","01",  --7
	"01", "01","01","10","10","10","10","10","11","11", "10","10","01","01",  --8
	"01", "01","01","01","10","10","10","10","10","10", "10","01","01","01",  --9
	"00", "01","01","01","01","10","10","10","10","10", "01","01","01","00",  --10
	"00", "01","01","01","01","01","01","01","01","01", "01","01","01","00",  --11
	"00", "00","01","01","01","01","01","01","01","01", "01","01","00","00",  --12
	"00", "00","00","01","01","01","01","01","01","01", "01","00","00","00",  --13
	"00", "00","00","00","00","01","01","01","01","00", "00","00","00","00"  --14
	);
	
  type ball_mask_addrtype is array(0 to 15) of integer;
  signal ball_mask_addr : ball_mask_addrtype;
  signal ball_mask_addr2 : ball_mask_addrtype;
  type ball_map_addrtype is array(0 to 15) of unsigned(9 downto 0);
  signal ball_map_addr : ball_map_addrtype;
  
  signal ball_map_address : unsigned(13 downto 0);
  --signal ball1_mask_addr : integer;
  signal ball_data : std_logic_vector(1 downto 0);
  signal ball_data_vga : integer;
  --signal ball1_map_addr : unsigned(9 downto 0);

  signal ball_mask_flag1 : std_logic_vector(15 downto 0);
  signal ball_mask_flag2 : std_logic_vector(15 downto 0);
  signal level2_vga : std_logic;
  signal ball_mask_color : integer;
  type ball_pos is array(0 to 15) of unsigned(15 downto 0);
  signal BALL_X : ball_pos;
  signal BALL_Y : ball_pos;
  type ball_bias is array(0 to 15) of unsigned(3 downto 0);
  signal BALL_BIAS_X : ball_bias;
  signal BALL_BIAS_Y : ball_bias;
  
  --Level 3 Pool cue    
  signal cue_line_begin : unsigned(15 downto 0) := "0000000000001000";
  signal cue_line_end : unsigned(15 downto 0) := "0000000000011000";
  
  signal level3_vga : std_logic;
  
  --Level 4 Words
  type word_type is array(0 to 7) of unsigned(15 downto 0);
  signal Word_start_x : word_type;
  signal Word_start_y : word_type;
  type word_eachaddrtype is array(0 to 7) of unsigned(9 downto 0);
  signal Word_eachaddr : word_eachaddrtype;
  signal Word_address : unsigned(12 downto 0);
  signal Word_en : std_logic_vector(7 downto 0);
  signal Word_hl_en : std_logic_vector(7 downto 0);   --hight light
  signal Word_hl : std_logic;
  signal Word_flag1, Word_flag2 : std_logic_vector(7 downto 0);
  signal Word_data : std_logic_vector(0 downto 0);
  signal Word_color : integer;
  signal level4_vga : std_logic;
  signal level4_active : std_logic;

begin
  x_coord <= Hcount - (HSYNC + HBACK_PORCH) + 1;
  y_coord <= Vcount - (VSYNC + VBACK_PORCH);
  
  CUERAM_clk <= clk;
  
  irq <= Interrupt;
  
  --25MHz clock generator
  CLOCK25: process(clk50)
  begin
	if rising_edge(clk50) then
		clk <= not clk;

	end if;
  end process;
  
  --VGA Ram
  vga_ram_address <= unsigned(address(6 downto 0));
  VGARAM : process(clk50)
  begin
	if rising_edge(clk50) then
		if reset = '1' then
			readdata <= (others => '0');
		else
			if chipselect = '1' then
				if read = '1' then
--              readdata <= VGA_RAM(to_integer(vga_ram_address));
				  
            elsif write = '1' then
              VGA_RAM(to_integer(vga_ram_address)) <= writedata;					
            end if;
			end if;
		end if;
	
	end if;
  end process VGARAM;
  
  Updata : process(clk)
  begin
    if rising_edge(clk) then
	    if reset = '1' then
		 	Strength <= "00011100";
			for I in 0 to 15 loop
				BALL_X(I) <= "0000000000000000";
				BALL_Y(I) <= "0000000000000000";
				BALL_BIAS_X(I) <= "0000";
				BALL_BIAS_Y(I) <= "0000";
			end loop;
			
			Word_en <= "00000000";
			Word_hl_en <= "00000000";
			for I in 0 to 7 loop
				Word_start_x(I) <= "0000000000000000";
				Word_start_y(I) <= "0000000000000000";
			end loop;
			
			Interrupt <= '0';
			
			elsif (Hcount = 0 and Vcount = 0) then		
				for I in 0 to 15 loop
					BALL_X(I) <= unsigned(VGA_RAM(4*I + 0));
					BALL_Y(I) <= unsigned(VGA_RAM(4*I + 1));
					if unsigned(VGA_RAM(4*I + 2)) <= 13 then
						BALL_BIAS_X(I) <= unsigned(VGA_RAM(4*I + 2)(3 downto 0));
					else
						BALL_BIAS_X(I) <= "0000";
					end if;
					
					if unsigned(VGA_RAM(4*I + 3)) <= 13 then
						BALL_BIAS_Y(I) <= unsigned(VGA_RAM(4*I + 3)(3 downto 0));
					else
						BALL_BIAS_Y(I) <= "0000";
					end if;
				end loop;
				
				Strength <= unsigned(VGA_RAM(64)(7 downto 0));  --Strength: Addr 1 (7 to 0)
				serveline_en <= VGA_RAM(65)(0);  --Enable show serveline
				
				Word_en <= VGA_RAM(66)(7 downto 0);
				Word_hl_en <= VGA_RAM(67)(7 downto 0);
				
				for I in 0 to 7 loop
					Word_start_x(I) <= unsigned(VGA_RAM(68 + 2*I));
					Word_start_y(I) <= unsigned(VGA_RAM(68 + 2*I + 1));
				end loop;
				
				Interrupt <= '1';
			else
				Interrupt <= '0';
			
		end if;
	 end if;
  end process Updata;
  
  LEDR(17) <= Interrupt;
  
  -- Horizontal and vertical counters
  HCounter : process (clk)
  begin
    if rising_edge(clk) then      
      if reset = '1' then
        Hcount <= (others => '0');
      elsif EndOfLine = '1' then
        Hcount <= (others => '0');
      else
        Hcount <= Hcount + 1;
      end if;      
    end if;
  end process HCounter;

  EndOfLine <= '1' when Hcount = HTOTAL - 1 else '0';
  
  VCounter: process (clk)
  begin
    if rising_edge(clk) then      
      if reset = '1' then
        Vcount <= (others => '0');
      elsif EndOfLine = '1' then
        if EndOfField = '1' then
          Vcount <= (others => '0');
        else
          Vcount <= Vcount + 1;
        end if;
      end if;
    end if;
  end process VCounter;

  EndOfField <= '1' when Vcount = VTOTAL - 1 else '0';

  -- State machines to generate HSYNC, VSYNC, HBLANK, and VBLANK

  HSyncGen : process (clk)
  begin
    if rising_edge(clk) then     
      if reset = '1' or EndOfLine = '1' then
        vga_hsync <= '1';
      elsif Hcount = HSYNC - 1 then
        vga_hsync <= '0';
      end if;
    end if;
  end process HSyncGen;
  
  HBlankGen : process (clk)
  begin
    if rising_edge(clk) then
      if reset = '1' then
        vga_hblank <= '1';
      elsif Hcount = HSYNC + HBACK_PORCH then
        vga_hblank <= '0';
      elsif Hcount = HSYNC + HBACK_PORCH + HACTIVE then
        vga_hblank <= '1';
      end if;      
    end if;
  end process HBlankGen;

  VSyncGen : process (clk)
  begin
    if rising_edge(clk) then
      if reset = '1' then
        vga_vsync <= '1';
      elsif EndOfLine ='1' then
        if EndOfField = '1' then
          vga_vsync <= '1';
        elsif Vcount = VSYNC - 1 then
          vga_vsync <= '0';
        end if;
      end if;      
    end if;
  end process VSyncGen;

  VBlankGen : process (clk)
  begin
    if rising_edge(clk) then    
      if reset = '1' then
        vga_vblank <= '1';
      elsif EndOfLine = '1' then
        if Vcount = VSYNC + VBACK_PORCH - 1 then
          vga_vblank <= '0';
        elsif Vcount = VSYNC + VBACK_PORCH + VACTIVE - 1 then
          vga_vblank <= '1';
        end if;
      end if;
    end if;
  end process VBlankGen;

  -- Rectangle generator

  -- Generate Table flags
  Level0_Gen : process (clk)
  variable tempStrength : unsigned(7 downto 0);
  begin
	 
    if rising_edge(clk) then  
		--serveline
		if reset = '1' then
			serveline <= '0';
		elsif serveline_en = '1' and x_coord = SERVELINE_H and y_coord >= TABLE3_VSTART and y_coord <= TABLE3_VEND then
			serveline <= '1';
		else
			serveline <= '0';
		end if;
		
		--table1
		if reset = '1' then
			table1 <= '0';
		elsif x_coord >= TABLE1_HSTART and x_coord <= TABLE1_HEND and y_coord >= TABLE1_VSTART and y_coord <= TABLE1_VEND then
			table1 <= '1';
		else
			table1 <= '0';
		end if;
		
		--table2		
		if reset = '1' then
			table2 <= '0';
		elsif x_coord >= TABLE2_HSTART and x_coord <= TABLE2_HEND and y_coord >= TABLE2_VSTART and y_coord <= TABLE2_VEND then
			table2 <= '1';
		else
			table2 <= '0';
		end if;
		
		--table3
		if reset = '1' then
			table3 <= '0';
		elsif x_coord >= TABLE3_HSTART and x_coord <= TABLE3_HEND and y_coord >= TABLE3_VSTART and y_coord <= TABLE3_VEND then
			table3 <= '1';
		else
			table3 <= '0';
		end if;
		
		
		-- Strength Bar
		tempStrength(2 downto 0) := "000";
		tempStrength(7 downto 3) := Strength(4 downto 0);
		SBAR_Strength <= SBAR_HSTART + to_integer(tempStrength);
		
		if reset = '1' or y_coord = SBAR_VEND then
        sbar_v <= '0';
		  sstrengh_v <= '0';
      elsif y_coord = SBAR_VSTART then
        sbar_v <= '1';
		  sstrengh_v <= '1';
      end if;
		
		if reset = '1' then
			sbar <= '0';
		elsif x_coord >= SBAR_HSTART and x_coord <= SBAR_HEND and y_coord >= SBAR_VSTART and y_coord <= SBAR_VEND then
			sbar <= '1';
		else
			sbar <= '0';
		end if;
				
		if reset = '1' then
			sstrengh <= '0';
		elsif x_coord >= SBAR_HSTART and x_coord <= SBAR_Strength and y_coord >= SBAR_VSTART and y_coord <= SBAR_VEND then
			sstrengh <= '1';
		else
			sstrengh <= '0';
		end if;
				
		level0_active <= table1 or table2 or table3 or sbar or sstrengh;
		
		if serveline = '1' then
			colorcode_level0 <= "11101";
		elsif table3 = '1' then
			colorcode_level0 <= "00001";
		elsif table2 = '1' then
			colorcode_level0 <= "11111";
		elsif table1 = '1' then
			colorcode_level0 <= "11110";
		elsif sstrengh = '1' then
			colorcode_level0 <= "01001";
		elsif sbar = '1' then
			colorcode_level0 <= "10101";
		end if;		
    end if;
  end process Level0_Gen;
  ------------------------------------------------------------------------------------------- 
  ------------------------------------------------------------------------------------------- 
  --Level 1
  Level1_rom_inst : Level1_rom port map(
	address => std_logic_vector(level1_address),
	clock => clk,
	q => level1_data
	);
	
  Level1_FLAG_Gen : process(clk)
   begin
	if rising_edge(clk) then
		if reset = '1' then
			for I in 0 to 5 loop
				pocket_flag1(I) <= '0';
			end loop;
		else
			if(x_coord + 2 >= TABLE3_HSTART - 19 and x_coord + 2 < TABLE3_HSTART + 16 and y_coord >= TABLE3_VSTART - 19 and y_coord < TABLE3_VSTART + 16) then
				pocket_flag1(0) <= '1';  --pocket 1
			else
				pocket_flag1(0) <= '0'; 
			end if;
			if(x_coord + 2 >= TABLE3_HSTART + 272 and x_coord + 2 < TABLE3_HSTART + 307 and y_coord >= TABLE3_VSTART - 23 and y_coord < TABLE3_VSTART + 12) then
				pocket_flag1(1) <= '1';  --pocket 2
			else
				pocket_flag1(1) <= '0'; 
			end if;
			if(x_coord + 2 >= TABLE3_HEND - 16 and x_coord + 2 < TABLE3_HEND + 19 and y_coord >= TABLE3_VSTART - 19 and y_coord < TABLE3_VSTART + 16) then
				pocket_flag1(2) <= '1';  --pocket 3
			else
				pocket_flag1(2) <= '0'; 
			end if;
			if(x_coord + 2 >= TABLE3_HSTART - 19 and x_coord + 2 < TABLE3_HSTART + 16 and y_coord >= TABLE3_VEND - 19 and y_coord < TABLE3_VEND + 16) then
				pocket_flag1(3) <= '1';  --pocket 4
			else
				pocket_flag1(3) <= '0'; 
			end if;
			if(x_coord + 2 >= TABLE3_HSTART + 272 and x_coord + 2 < TABLE3_HSTART + 307 and y_coord >= TABLE3_VEND and y_coord < TABLE3_VEND + 35) then
				pocket_flag1(4) <= '1';  --pocket 5	
			else
				pocket_flag1(4) <= '0'; 
			end if;
			if(x_coord + 2 >= TABLE3_HEND - 16 and x_coord + 2 < TABLE3_HEND + 19 and y_coord >= TABLE3_VEND - 16 and y_coord < TABLE3_VEND + 19) then
				pocket_flag1(5) <= '1';  --pocket 6
			else
				pocket_flag1(5) <= '0'; 
			end if;
			
		end if;
		
	end if;
  end process Level1_FLAG_Gen;
  
  Level1_AddrGen1 : process(clk)
  begin
	if rising_edge(clk) then
		if reset = '1' then
			for I in 0 to 5 loop
				pocket_eachaddr(I) <= "00000000000";  --11 bits
			end loop;
		else
			-- pocket 1
			if(x_coord = TABLE3_HSTART + 16 and y_coord = TABLE3_VSTART + 16) then
				pocket_eachaddr(0) <= "00000000000";  --11 bits
			elsif(x_coord + 1 >= TABLE3_HSTART - 19 and x_coord + 1 < TABLE3_HSTART + 16 and y_coord >= TABLE3_VSTART - 19 and y_coord < TABLE3_VSTART + 16) then
				pocket_eachaddr(0) <= pocket_eachaddr(0) + 1; 
			end if;
			-- pocket 2
			if(x_coord = TABLE3_HSTART + 307 and y_coord = TABLE3_VSTART + 12) then
				pocket_eachaddr(1) <= "00000000000";  --11 bits
			elsif(x_coord + 1 >= TABLE3_HSTART + 272 and x_coord + 1 < TABLE3_HSTART + 307 and y_coord >= TABLE3_VSTART - 23 and y_coord < TABLE3_VSTART + 12) then
				pocket_eachaddr(1) <= pocket_eachaddr(1) + 1; 
			end if;
			-- pocket 3
			if(x_coord = TABLE3_HEND + 19 and y_coord = TABLE3_VSTART + 16) then
				pocket_eachaddr(2) <= "00000000000";  --11 bits
			elsif(x_coord + 1 >= TABLE3_HEND - 16 and x_coord + 1 < TABLE3_HEND + 19 and y_coord >= TABLE3_VSTART - 19 and y_coord < TABLE3_VSTART + 16) then
				pocket_eachaddr(2) <= pocket_eachaddr(2) + 1; 
			end if;
			-- pocket 4
			if(x_coord = TABLE3_HSTART + 16 and y_coord = TABLE3_VSTART + 16) then
				pocket_eachaddr(3) <= "00000000000";  --11 bits
			elsif(x_coord + 1 >= TABLE3_HSTART - 19 and x_coord + 1 < TABLE3_HSTART + 16 and y_coord >= TABLE3_VEND - 19 and y_coord < TABLE3_VEND + 16) then
				pocket_eachaddr(3) <= pocket_eachaddr(3) + 1; 
			end if;
			-- pocket 5
			if(x_coord = TABLE3_HSTART + 307 and y_coord = TABLE3_VEND + 35) then
				pocket_eachaddr(4) <= "00000000000";  --11 bits
			elsif(x_coord + 1 >= TABLE3_HSTART + 272 and x_coord + 1 < TABLE3_HSTART + 307 and y_coord >= TABLE3_VEND and y_coord < TABLE3_VEND + 35) then
				pocket_eachaddr(4) <= pocket_eachaddr(4) + 1; 
			end if;
			-- pocket 6
			if(x_coord = TABLE3_HEND + 19 and y_coord = TABLE3_VEND + 19) then
				pocket_eachaddr(5) <= "00000000000";  --11 bits
			elsif(x_coord + 1 >= TABLE3_HEND - 16 and x_coord + 1 < TABLE3_HEND + 19 and y_coord >= TABLE3_VEND - 16 and y_coord < TABLE3_VEND + 19) then
				pocket_eachaddr(5) <= pocket_eachaddr(5) + 1; 
			end if;		
		end if;
	end if;
  end process Level1_AddrGen1;
  
  Level1_AddrGen2 : process(clk)
  begin
	if rising_edge(clk) then	
		--	 Generate ball address
		if reset = '1' then
			level1_address <= "00000000000000";  -- 14 bits
		elsif pocket_flag1(0) = '1' then
			level1_address(13 downto 3) <= pocket_eachaddr(0);
			level1_address(2 downto 0) <= "000";
		elsif pocket_flag1(1) = '1' then
			level1_address(13 downto 3) <= pocket_eachaddr(1);
			level1_address(2 downto 0) <= "001";
		elsif pocket_flag1(2) = '1' then
			level1_address(13 downto 3) <= pocket_eachaddr(2);
			level1_address(2 downto 0) <= "010";
		elsif pocket_flag1(3) = '1' then
			level1_address(13 downto 3) <= pocket_eachaddr(3);
			level1_address(2 downto 0) <= "011";
		elsif pocket_flag1(4) = '1' then
			level1_address(13 downto 3) <= pocket_eachaddr(4);
			level1_address(2 downto 0) <= "100";
		elsif pocket_flag1(5) = '1' then
			level1_address(13 downto 3) <= pocket_eachaddr(5);
			level1_address(2 downto 0) <= "101";
		else
			level1_address <= "00000000000000";  -- 14 bits
		end if;
				
		pocket_flag2 <= pocket_flag1;
	end if;
  end process Level1_AddrGen2;
  
  Level1_ActiveFGen : process(clk)
  begin
	if rising_edge(clk) then
		if reset = '1' then
			level1_active <= '0';
		else
		   level1_active <= pocket_flag2(0) or pocket_flag2(1) or pocket_flag2(2) or 
									pocket_flag2(3) or pocket_flag2(4) or pocket_flag2(5);

		end if;
	end if;
  end process Level1_ActiveFGen;

  Level1_Gen : process(clk)
	begin
	if rising_edge(clk) then
		if reset = '1' then
			level1_vga <= '0';
		elsif (level1_active = '1' and (not (level1_data = "00000"))) then
		   level1_vga <= '1';
		else
			level1_vga <= '0';
		end if;
		level1_data_vga <= level1_data;
	end if;
  end process Level1_Gen;  


 ------------------------------------------------------------------------------------------- 
  ------------------------------------------------------------------------------------------- 
  --Level 2
	
  BALLMaskFLAG_Gen : process(clk)
   begin
	if rising_edge(clk) then
		if reset = '1' then
			ball_mask_flag1 <= "0000000000000000";
		else
			for I in 0 to 15 loop
				if (x_coord + 2>= BALL_X(I) - 7 and x_coord + 2< BALL_X(I) + 7 and y_coord>= BALL_Y(I) - 7 and y_coord< BALL_Y(I) + 7) then
					ball_mask_flag1(I) <= '1';
				else
					ball_mask_flag1(I) <= '0';
				end if;
			end loop;
			
		end if;
	end if;
  end process BALLMaskFLAG_Gen;
  
	Ball_mask_AddrGen : process(clk)
   begin
	if rising_edge(clk) then	
		if reset = '1' then
			for I in 0 to 15 loop
				ball_mask_addr(I) <= -1; 
				ball_mask_addr2(I) <= -1;
			end loop;
		else
			for I in 0 to 15 loop
				if (x_coord = BALL_X(I) + 7 and y_coord = BALL_Y(I) + 7) then
					ball_mask_addr(I) <= -1;  
				elsif (x_coord + 1+ 7 >= BALL_X(I)  and x_coord + 1 < BALL_X(I) + 7 and y_coord + 7 >= BALL_Y(I) and y_coord < BALL_Y(I) + 7) then
					ball_mask_addr(I) <= ball_mask_addr(I) + 1;
				end if;

				if (x_coord = BALL_X(I) + 7 and y_coord = BALL_Y(I) + 7) then
					ball_mask_addr2(I) <= -1;  
				elsif (x_coord + 2+ 7 >= BALL_X(I)  and x_coord + 2 < BALL_X(I) + 7 and y_coord + 7 >= BALL_Y(I) and y_coord < BALL_Y(I) + 7) then
					ball_mask_addr2(I) <= ball_mask_addr2(I) + 1;
				end if;		
				
			end loop;
		end if;
	end if;
  end process Ball_mask_AddrGen;
    
   Ball_map_AddrGen1 : process(clk)
   begin
	if rising_edge(clk) then
		if reset = '1' then
			for I in 0 to 15 loop
				ball_map_addr(I) <= "0000000000";   --10 bits
			end loop;		
		else
			for I in 0 to 15 loop
				if (x_coord + BALL_BIAS_X(I) = BALL_X(I) + 20  and y_coord + BALL_BIAS_Y(I) = BALL_Y(I) + 20) then
					ball_map_addr(I) <= "0000000000";
				elsif (x_coord + 1 + BALL_BIAS_X(I) + 7>= BALL_X(I)  and x_coord + 1 + BALL_BIAS_X(I) < BALL_X(I) + 20 and y_coord + BALL_BIAS_Y(I) + 7>= BALL_Y(I) and y_coord + BALL_BIAS_Y(I)< BALL_Y(I) + 20) then
					ball_map_addr(I) <= ball_map_addr(I) + 1;
				end if;
			end loop;
			
		end if;
	end if;
  end process Ball_map_AddrGen1;
    
  Ball_map_AddrGen2 : process(clk)
  begin
	if rising_edge(clk) then	
		--	 Generate ball address
		if reset = '1' then
			ball_map_address <= "00000000000000";  --14 bits
		--elsif ball_mask_flag1(15) = '1' and (not(BallMask(ball_mask_addr(15) + 1) = "00")) then    --ball number 16
		elsif ball_mask_flag1(15) = '1' and (not(BallMask(ball_mask_addr2(15)) = "00")) then    --ball number 16
			ball_map_address(13 downto 4) <= ball_map_addr(15);
			ball_map_address(3 downto 0) <= "1111";
		--elsif ball_mask_flag1(0) = '1' and (not(BallMask(ball_mask_addr(0) + 1) = "00")) then    --ball number 1
		elsif ball_mask_flag1(0) = '1' and (not(BallMask(ball_mask_addr2(0)) = "00")) then    --ball number 1
			ball_map_address(13 downto 4) <= ball_map_addr(0);
			ball_map_address(3 downto 0) <= "0000";
		--elsif ball_mask_flag1(1) = '1' and (not(BallMask(ball_mask_addr(1) + 1) = "00")) then    --ball number 2
		elsif ball_mask_flag1(1) = '1' and (not(BallMask(ball_mask_addr2(1)) = "00")) then    --ball number 2
			ball_map_address(13 downto 4) <= ball_map_addr(1);
			ball_map_address(3 downto 0) <= "0001";
		--elsif ball_mask_flag1(2) = '1' and (not(BallMask(ball_mask_addr(2) + 1) = "00")) then    --ball number 3
		elsif ball_mask_flag1(2) = '1' and (not(BallMask(ball_mask_addr2(2)) = "00")) then    --ball number 3
			ball_map_address(13 downto 4) <= ball_map_addr(2);
			ball_map_address(3 downto 0) <= "0010";
		--elsif ball_mask_flag1(3) = '1' and (not(BallMask(ball_mask_addr(3) + 1) = "00")) then    --ball number 4
		elsif ball_mask_flag1(3) = '1' and (not(BallMask(ball_mask_addr2(3)) = "00")) then    --ball number 4
			ball_map_address(13 downto 4) <= ball_map_addr(3);
			ball_map_address(3 downto 0) <= "0011";
		--elsif ball_mask_flag1(4) = '1' and (not(BallMask(ball_mask_addr(4) + 1) = "00")) then    --ball number 5
		elsif ball_mask_flag1(4) = '1' and (not(BallMask(ball_mask_addr2(4)) = "00")) then    --ball number 5
			ball_map_address(13 downto 4) <= ball_map_addr(4);
			ball_map_address(3 downto 0) <= "0100";
		--elsif ball_mask_flag1(5) = '1' and (not(BallMask(ball_mask_addr(5) + 1) = "00")) then    --ball number 6
		elsif ball_mask_flag1(5) = '1' and (not(BallMask(ball_mask_addr2(5)) = "00")) then    --ball number 6
			ball_map_address(13 downto 4) <= ball_map_addr(5);
			ball_map_address(3 downto 0) <= "0101";
		--elsif ball_mask_flag1(6) = '1' and (not(BallMask(ball_mask_addr(6) + 1) = "00")) then    --ball number 7
		elsif ball_mask_flag1(6) = '1' and (not(BallMask(ball_mask_addr2(6)) = "00")) then    --ball number 7
			ball_map_address(13 downto 4) <= ball_map_addr(6);
			ball_map_address(3 downto 0) <= "0110";
		--elsif ball_mask_flag1(7) = '1' and (not(BallMask(ball_mask_addr(7) + 1) = "00")) then    --ball number 8
		elsif ball_mask_flag1(7) = '1' and (not(BallMask(ball_mask_addr2(7)) = "00")) then    --ball number 8
			ball_map_address(13 downto 4) <= ball_map_addr(7);
			ball_map_address(3 downto 0) <= "0111";
		--elsif ball_mask_flag1(8) = '1' and (not(BallMask(ball_mask_addr(8) + 1) = "00")) then    --ball number 9
		elsif ball_mask_flag1(8) = '1' and (not(BallMask(ball_mask_addr2(8)) = "00")) then    --ball number 9
			ball_map_address(13 downto 4) <= ball_map_addr(8);
			ball_map_address(3 downto 0) <= "1000";
		--elsif ball_mask_flag1(9) = '1' and (not(BallMask(ball_mask_addr(9) + 1) = "00")) then    --ball number 10
		elsif ball_mask_flag1(9) = '1' and (not(BallMask(ball_mask_addr2(9)) = "00")) then    --ball number 10
			ball_map_address(13 downto 4) <= ball_map_addr(9);
			ball_map_address(3 downto 0) <= "1001";
		--elsif ball_mask_flag1(10) = '1' and (not(BallMask(ball_mask_addr(10) + 1) = "00")) then    --ball number 11
		elsif ball_mask_flag1(10) = '1' and (not(BallMask(ball_mask_addr2(10)) = "00")) then    --ball number 11
			ball_map_address(13 downto 4) <= ball_map_addr(10);
			ball_map_address(3 downto 0) <= "1010";
		--elsif ball_mask_flag1(11) = '1' and (not(BallMask(ball_mask_addr(11) + 1) = "00")) then    --ball number 12
		elsif ball_mask_flag1(11) = '1' and (not(BallMask(ball_mask_addr2(11)) = "00")) then    --ball number 12
			ball_map_address(13 downto 4) <= ball_map_addr(11);
			ball_map_address(3 downto 0) <= "1011";
		--elsif ball_mask_flag1(12) = '1' and (not(BallMask(ball_mask_addr(12) + 1) = "00")) then    --ball number 13
		elsif ball_mask_flag1(12) = '1' and (not(BallMask(ball_mask_addr2(12)) = "00")) then    --ball number 13
			ball_map_address(13 downto 4) <= ball_map_addr(12);
			ball_map_address(3 downto 0) <= "1100";
		--elsif ball_mask_flag1(13) = '1' and (not(BallMask(ball_mask_addr(13) + 1) = "00")) then    --ball number 14
		elsif ball_mask_flag1(13) = '1' and (not(BallMask(ball_mask_addr2(13)) = "00")) then    --ball number 14
			ball_map_address(13 downto 4) <= ball_map_addr(13);
			ball_map_address(3 downto 0) <= "1101";
		--elsif ball_mask_flag1(14) = '1' and (not(BallMask(ball_mask_addr(14) + 1) = "00")) then    --ball number 15
		elsif ball_mask_flag1(14) = '1' and (not(BallMask(ball_mask_addr2(14)) = "00")) then    --ball number 15
			ball_map_address(13 downto 4) <= ball_map_addr(14);
			ball_map_address(3 downto 0) <= "1110";
		else
			ball_map_address <= "00000000000000";
		end if;
		
		ball_mask_flag2 <= ball_mask_flag1;
	end if;
  end process Ball_map_AddrGen2;
  
  Ball_Map_inst : Ball_Map_rom port map(
	address => std_logic_vector(ball_map_address),
	clock => clk,
	q => ball_data
	);
	

  
  Ball_mask_colorGen : process(clk)
	begin
		if rising_edge(clk) then	
			if reset = '1' then
				ball_mask_color <= 0;
			elsif (ball_mask_flag2(15) = '1' and (not(BallMask(ball_mask_addr(15)) = "00"))) then  -- ball num 16
				ball_mask_color <= to_integer(BallMask(ball_mask_addr(15))) + 26;
			elsif (ball_mask_flag2(0) = '1' and (not(BallMask(ball_mask_addr(0)) = "00"))) then  -- ball num 1
				ball_mask_color <= to_integer(BallMask(ball_mask_addr(0))) + 2;
			elsif (ball_mask_flag2(1) = '1' and (not(BallMask(ball_mask_addr(1)) = "00"))) then  -- ball num 2
				ball_mask_color <= to_integer(BallMask(ball_mask_addr(1))) + 5;
			elsif (ball_mask_flag2(2) = '1' and (not(BallMask(ball_mask_addr(2)) = "00"))) then  -- ball num 3
				ball_mask_color <= to_integer(BallMask(ball_mask_addr(2))) + 8;
			elsif (ball_mask_flag2(3) = '1' and (not(BallMask(ball_mask_addr(3)) = "00"))) then  -- ball num 4
				ball_mask_color <= to_integer(BallMask(ball_mask_addr(3))) + 11;
			elsif (ball_mask_flag2(4) = '1' and (not(BallMask(ball_mask_addr(4)) = "00"))) then  -- ball num 5
				ball_mask_color <= to_integer(BallMask(ball_mask_addr(4))) + 14;
			elsif (ball_mask_flag2(5) = '1' and (not(BallMask(ball_mask_addr(5)) = "00"))) then  -- ball num 6
				ball_mask_color <= to_integer(BallMask(ball_mask_addr(5))) + 17;
			elsif (ball_mask_flag2(6) = '1' and (not(BallMask(ball_mask_addr(6)) = "00"))) then  -- ball num 7
				ball_mask_color <= to_integer(BallMask(ball_mask_addr(6))) + 20;
			elsif (ball_mask_flag2(7) = '1' and (not(BallMask(ball_mask_addr(7)) = "00"))) then  -- ball num 8
				ball_mask_color <= to_integer(BallMask(ball_mask_addr(7))) + 23;
			elsif (ball_mask_flag2(8) = '1' and (not(BallMask(ball_mask_addr(8)) = "00"))) then  -- ball num 9
				ball_mask_color <= to_integer(BallMask(ball_mask_addr(8))) + 2;
			elsif (ball_mask_flag2(9) = '1' and (not(BallMask(ball_mask_addr(9)) = "00"))) then  -- ball num 10
				ball_mask_color <= to_integer(BallMask(ball_mask_addr(9))) + 5;
			elsif (ball_mask_flag2(10) = '1' and (not(BallMask(ball_mask_addr(10)) = "00"))) then  -- ball num 11
				ball_mask_color <= to_integer(BallMask(ball_mask_addr(10))) + 8;
			elsif (ball_mask_flag2(11) = '1' and (not(BallMask(ball_mask_addr(11)) = "00"))) then  -- ball num 12
				ball_mask_color <= to_integer(BallMask(ball_mask_addr(11))) + 11;
			elsif (ball_mask_flag2(12) = '1' and (not(BallMask(ball_mask_addr(12)) = "00"))) then  -- ball num 13
				ball_mask_color <= to_integer(BallMask(ball_mask_addr(12))) + 14;
			elsif (ball_mask_flag2(13) = '1' and (not(BallMask(ball_mask_addr(13)) = "00"))) then  -- ball num 14
				ball_mask_color <= to_integer(BallMask(ball_mask_addr(13))) + 17;
			elsif (ball_mask_flag2(14) = '1' and (not(BallMask(ball_mask_addr(14)) = "00"))) then  -- ball num 15
				ball_mask_color <= to_integer(BallMask(ball_mask_addr(14))) + 20;	
			else	
				ball_mask_color <= 0;
			end if;
		end if;
	end process Ball_mask_colorGen;
  
  Level2_ActiveFGen : process(clk)
	begin
	if rising_edge(clk) then
		if reset = '1' then
			level2_active <= '0';
		else
		   level2_active <= ball_mask_flag2(0) or ball_mask_flag2(1) or ball_mask_flag2(2) or ball_mask_flag2(3) or 
									ball_mask_flag2(4) or ball_mask_flag2(5) or ball_mask_flag2(6) or ball_mask_flag2(7) or 
									ball_mask_flag2(8) or ball_mask_flag2(9) or ball_mask_flag2(10) or ball_mask_flag2(11) or 
									ball_mask_flag2(12) or ball_mask_flag2(13) or ball_mask_flag2(14) or ball_mask_flag2(15);
		end if;
	end if;
  end process Level2_ActiveFGen;
  
  Level2_Gen : process(clk)
	begin
	if rising_edge(clk) then
		if reset = '1' then
			level2_vga <= '0';
		elsif ((level2_active = '1') and (not (ball_mask_color = 0))) then
--		elsif (level2_active = '1') then
		   level2_vga <= '1';
		else
			level2_vga <= '0';
		end if;
		
		if reset = '1' then
			ball_data_vga <= 0;
		elsif ball_data(1) = '1' then
			if ball_data(0) = '0' then
				ball_data_vga <= 24;  --black
			elsif ball_data(0) = '1' then
				ball_data_vga <= 29;  --while
			end if;
		else --ball_data(1) = '0'
			ball_data_vga <= ball_mask_color;
		end if;
	end if;
  end process Level2_Gen;
  
   ------------------------------------------------------------------------------------------- 
  ------------------------------------------------------------------------------------------- 
  --Level 3
  
  --level 3
	PoolCueRAM : process(clk)
	  begin
	  if rising_edge(clk) then
		if reset = '1' then
			CUERAM_addr <= "000000000"; -- 9 bits std_logic_vector
			cue_line_begin <= "0000000000000000";
			cue_line_end <= "0000000000000000";
		else
			CUERAM_addr <= std_logic_vector(y_coord(8 downto 0));
			cue_line_begin <= unsigned(CUERAM_q(31 downto 16));
			cue_line_end <= unsigned(CUERAM_q(15 downto 0));
		end if;
	  end if;
	end process PoolCueRAM;
	
	Level3_Gen : process(clk)
	  begin
	  if rising_edge(clk) then
		if reset = '1' then
			level3_vga <= '0';
			--cue_line_begin <= "0000" 
		else
			if (x_coord >= cue_line_begin and x_coord < cue_line_end) then
				level3_vga <= '1';
			else
				level3_vga <= '0';
			end if;
			
		end if;
	  end if;
	end process Level3_Gen;
	
	 ------------------------------------------------------------------------------------------- 
  ------------------------------------------------------------------------------------------- 
  --Level 4
  
   Words_rom_inst : Words_rom port map(
	address => std_logic_vector(Word_address),
	clock => clk,
	q => Word_data
	);
	
	Word_FLAG_Gen : process(clk)
   begin
	if rising_edge(clk) then
		if reset = '1' then
			Word_flag1 <= (others => '0');
		else
			for I in 0 to 7 loop
				if(Word_en(I) = '1' and x_coord + 2 >= Word_start_x(I) and x_coord + 2 < Word_start_x(I) + 40 and y_coord >= Word_start_y(I) and y_coord < Word_start_y(I) + 16) then
					Word_flag1(I) <= '1';  
				else
					Word_flag1(I) <= '0'; 
				end if;
			end loop;
		end if;
		
	end if;
  end process Word_FLAG_Gen;
  
  Word_AddrGen1 : process(clk)
  begin
	if rising_edge(clk) then
		if reset = '1' then
			for I in 0 to 7 loop
				Word_eachaddr(I) <= (others => '0');
			end loop;
		else
		
			for I in 0 to 7 loop
			   if(x_coord = Word_start_x(I) + 40 and y_coord = Word_start_y(I) + 16) then
					Word_eachaddr(I) <= (others => '0');
				elsif(x_coord + 1 >= Word_start_x(I) and x_coord + 1 < Word_start_x(I) + 40 and y_coord >= Word_start_y(I) and y_coord < Word_start_y(I) + 16) then
					Word_eachaddr(I) <= Word_eachaddr(I) + 1;  
				end if;
			end loop;
	
		end if;
	end if;
  end process Word_AddrGen1;
  
  Word_AddrGen2 : process(clk)
  begin
	if rising_edge(clk) then	
		--	 Generate ball address
		if reset = '1' then
			Word_address <= (others => '0');  
		elsif word_flag1(0) = '1' then
			Word_address(12 downto 3) <= Word_eachaddr(0);
			Word_address(2 downto 0) <= "000";
			Word_hl <= Word_hl_en(0);
		elsif word_flag1(1) = '1' then
			Word_address(12 downto 3) <= Word_eachaddr(1);
			Word_address(2 downto 0) <= "001";
			Word_hl <= Word_hl_en(1);
		elsif word_flag1(2) = '1' then
			Word_address(12 downto 3) <= Word_eachaddr(2);
			Word_address(2 downto 0) <= "010";
			Word_hl <= Word_hl_en(2);
		elsif word_flag1(3) = '1' then
			Word_address(12 downto 3) <= Word_eachaddr(3);
			Word_address(2 downto 0) <= "011";
			Word_hl <= Word_hl_en(3);
		elsif word_flag1(4) = '1' then
			Word_address(12 downto 3) <= Word_eachaddr(4);
			Word_address(2 downto 0) <= "100";
			Word_hl <= Word_hl_en(4);
		elsif word_flag1(5) = '1' then
			Word_address(12 downto 3) <= Word_eachaddr(5);
			Word_address(2 downto 0) <= "101";
			Word_hl <= Word_hl_en(5);
		elsif word_flag1(6) = '1' then
			Word_address(12 downto 3) <= Word_eachaddr(6);
			Word_address(2 downto 0) <= "110";
			Word_hl <= Word_hl_en(6);
		elsif word_flag1(7) = '1' then
			Word_address(12 downto 3) <= Word_eachaddr(7);
			Word_address(2 downto 0) <= "111";
			Word_hl <= Word_hl_en(7);
		else
			Word_address <= (others => '0');  
			Word_hl <= '0';
		end if;
				
		Word_flag2 <= Word_flag1;
	end if;
  end process Word_AddrGen2;
  
  Level4_ActiveFGen : process(clk)
  begin
	if rising_edge(clk) then
		if reset = '1' then
			level4_active <= '0';
		else
		   level4_active <= word_flag2(0) or word_flag2(1) or word_flag2(2) or 
									word_flag2(3) or word_flag2(4) or word_flag2(5) or 
									word_flag2(6) or word_flag2(7);

		end if;
	end if;
  end process Level4_ActiveFGen;

  Level4_Gen : process(clk)
	begin
	if rising_edge(clk) then
		if reset = '1' then
			level4_vga <= '0';
			Word_color <= 0;
		elsif level4_active = '1' then
			if Word_data = "1" then
				level4_vga <= '1';
				Word_color <= 24;
			elsif Word_hl = '1' then
				level4_vga <= '1';
				Word_color <= 15;
			else
				level4_vga <= '0';
			end if;
		else
			level4_vga <= '0';
		end if;

	end if;
  end process Level4_Gen;  
	
	
  -- Registered video signals going to the video DAC
 

  VideoOut: process (clk, reset)
  begin
	 VGA_R(1 downto 0) <= "00";
	 VGA_G(1 downto 0) <= "00";
	 VGA_B(1 downto 0) <= "00";
    if reset = '1' then
      VGA_R(9 downto 2) <= "00000000";
      VGA_G(9 downto 2) <= "00000000";
      VGA_B(9 downto 2) <= "00000000";
    elsif clk'event and clk = '1' then
		if vga_hblank = '1' or vga_vblank ='1' then
		  VGA_R(9 downto 2) <= "00000000";
        VGA_G(9 downto 2) <= "00000000";
        VGA_B(9 downto 2) <= "00000000";
		elsif level4_vga = '1' then
		  VGA_R(9 downto 2) <= COLOR_TABLE_R(Word_color);
        VGA_G(9 downto 2) <= COLOR_TABLE_G(Word_color);
        VGA_B(9 downto 2) <= COLOR_TABLE_B(Word_color);
		elsif level3_vga = '1' then
		  VGA_R(9 downto 2) <= COLOR_TABLE_R(2);
        VGA_G(9 downto 2) <= COLOR_TABLE_G(2);
        VGA_B(9 downto 2) <= COLOR_TABLE_B(2);
		elsif level2_vga = '1' then
		  VGA_R(9 downto 2) <= COLOR_TABLE_R(ball_data_vga);
        VGA_G(9 downto 2) <= COLOR_TABLE_G(ball_data_vga);
        VGA_B(9 downto 2) <= COLOR_TABLE_B(ball_data_vga);
		elsif level1_vga = '1' then
		  VGA_R(9 downto 2) <= COLOR_TABLE_R(to_integer(unsigned(level1_data_vga)));
        VGA_G(9 downto 2) <= COLOR_TABLE_G(to_integer(unsigned(level1_data_vga)));
        VGA_B(9 downto 2) <= COLOR_TABLE_B(to_integer(unsigned(level1_data_vga)));
      elsif level0_active = '1' then
        VGA_R(9 downto 2) <= COLOR_TABLE_R(to_integer(colorcode_level0));
        VGA_G(9 downto 2) <= COLOR_TABLE_G(to_integer(colorcode_level0));
        VGA_B(9 downto 2) <= COLOR_TABLE_B(to_integer(colorcode_level0));

      else --background color
        VGA_R(9 downto 2) <= COLOR_TABLE_R(5);
        VGA_G(9 downto 2) <= COLOR_TABLE_G(5);
        VGA_B(9 downto 2) <= COLOR_TABLE_B(5);
    
      end if;
    end if;
  end process VideoOut;

  VGA_CLK <= clk;
  VGA_HS <= not vga_hsync;
  VGA_VS <= not vga_vsync;
  VGA_SYNC <= '0';
  VGA_BLANK <= not (vga_hsync or vga_vsync);

end rtl;
