-------------------------------------------------------------------------------
--
-- VGA raster display of abritrary memory
--
-- David Calhoun
-- dmc2202@columbia.edu
-- 
-- Adapted from code written by Stephen A. Edwards, sedwards@cs.columbia.edu
--
-------------------------------------------------------------------------------

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity de2_vga_raster is
  
	port (
		reset : in std_logic;
		clk   : in std_logic;                    -- Should be 25.125 MHz
	
		-- Read from memory to access position
		read			:	in std_logic;
		write		: 	in std_logic;
		chipselect	:	in std_logic;
		address		: 	in std_logic_vector(3 downto 0);
		readdata	:	out std_logic_vector(15 downto 0);
		writedata	:	in std_logic_vector(15 downto 0);
	
		-- address to write pixel obtained from external source
		read_address : out std_logic_vector(7 downto 0);
		-- pixel data from external source
		data_from : in std_logic_vector(7 downto 0);
		-- selects which image sample (2-5) should be read from
		rden_selects 	: out std_logic_vector(3 downto 0);
	
		-- VGA connectivity
		VGA_CLK,                         -- Clock
		VGA_HS,                          -- H_SYNC
		VGA_VS,                          -- V_SYNC
		VGA_BLANK,                       -- BLANK
		VGA_SYNC : out std_logic;        -- SYNC
		VGA_R,                           -- Red[9:0]
		VGA_G,                           -- Green[9:0]
		VGA_B : out std_logic_vector(9 downto 0) -- Blue[9:0]
	);

end de2_vga_raster;

architecture rtl of de2_vga_raster is
  
  	component RAM1 is
		port(
			clock		: in std_logic :='1';
			data		: in std_logic_vector (7 downto 0);
			rdaddress	: in std_logic_vector (13 downto 0);
			wraddress	: in std_logic_vector (13 downto 0);
			wren		: in std_logic :='0';
			q			: out std_logic_vector (7 downto 0)
		);
	end component;

	component map_memory is
	port(
		clock : in std_logic;
		addr_in : in unsigned(13 downto 0);
		wren : in std_logic;
		addr_aout : out unsigned(13 downto 0);
		addr_sout : out unsigned(7 downto 0)
		
	);
	end component;

	component pll is
	port(
		inclk0 : in std_logic;
		c0 : out std_logic
	);
	end component;
		
	-- Video parameters
    constant HTOTAL       : integer := 800;
	constant HSYNC        : integer := 96;
	constant HBACK_PORCH  : integer := 48;
	constant HACTIVE      : integer := 640;
	constant HFRONT_PORCH : integer := 16;
	constant VTOTAL       : integer := 525;
	constant VSYNC        : integer := 2;
	constant VBACK_PORCH  : integer := 33;
	constant VACTIVE      : integer := 480;
	constant VFRONT_PORCH : integer := 10;

	constant BOX_SET_XSTART : integer := 100;
	constant BOX_SET_XEND : integer := 356;
	constant BOX_SET_YSTART : integer := 100;
	constant BOX_SET_YEND : integer := 356;
	constant BOX_SET_XSTART2 : integer := 498;
	constant BOX_SET_XEND2 : integer := 530;	
	constant BOX_SET_YSTART2 : integer := 220;
	constant BOX_SET_YEND2 : integer := 252;

	signal ram_address : unsigned(13 downto 0);
	signal ram_address2 : unsigned(7 downto 0);
	signal display_address11 : unsigned(13 downto 0) := "00000000000000";
	signal display_address21 : unsigned(7 downto 0) := "00000000";

	-- Signals for the video controller
	signal Hcount : unsigned(9 downto 0);  -- Horizontal position (0-800)
	signal Vcount : unsigned(9 downto 0);  -- Vertical position (0-524)
	signal EndOfLine, EndOfField : std_logic;
	signal vga_hblank, vga_hsync,
    vga_vblank, vga_vsync : std_logic;  -- Sync. signals

	signal area : std_logic := '0';  -- flag for within writable area
	signal area_x : std_logic := '0';  -- flag for within writable area
	signal area_y : std_logic := '0';  -- flag for within writable area
	signal area2 : std_logic := '0';  -- flag for within writable area
	signal area_x2 : std_logic := '0';  -- flag for within writable area
	signal area_y2 : std_logic := '0';  -- flag for within writable area 
	signal both_areas : std_logic:= '0';

	signal display_pixel : std_logic_vector(7 downto 0) := "00000000";
	signal pixel : unsigned(7 downto 0);
	signal waitx : std_logic := '1';
	signal waity : std_logic := '1';
	signal waitx2 : std_logic := '1';
	signal waity2 : std_logic := '1';

	signal display_pixel2 : std_logic_vector(7 downto 0) := "00000000";
	signal pixel2 : unsigned(7 downto 0);
	signal rdaddress : std_logic_vector(13 downto 0);
	--signal data : std_logic_vector(7 downto 0);
	--signal wren : std_logic := '1'; 
	signal rdaddress2 : std_logic_vector(7 downto 0);
	--signal data2 : std_logic_vector(7 downto 0);
	signal wren2 : std_logic := '1'; 
	-- need to clock at about 25 MHz for NTSC VGA
	signal clk_25 : std_logic;
	
	signal start_ram : unsigned (13 downto 0) := "00011100000000";
	signal q1, q2, q3, q4, q5 : std_logic_vector(7 downto 0);
	--signal data_from : std_logic_vector(7 downto 0);
	signal addr_aout : unsigned(13 downto 0);
	signal addr_sout : unsigned(7 downto 0);
	--signal rden_selects : std_logic_vector(3 downto 0) := "0001";

	signal init1 : std_logic := '0';
	signal init2 : std_logic := '0';
	signal aggr_en : std_logic := '0';
	signal check_selects : std_logic_vector(3 downto 0) := "0001";
	signal async_reset : std_logic := '0';
	signal data_to_aggr : std_logic_vector(7 downto 0) := "00000000";
	signal write_to_address : unsigned(13 downto 0);
	signal clear_address : unsigned(13 downto 0) := "00000000000000";
	signal async_count : unsigned(3 downto 0) := x"0";
	signal box_status : std_logic_vector(5 downto 0) := "000000";

begin
	
	RAM1_inst : RAM1 PORT MAP (
		clock	 => clk,
		data	 => data_to_aggr,
		rdaddress	 => rdaddress,
		wraddress	 => std_logic_vector(write_to_address),
		--wren	 => (not area),
		wren	 => (not area) and aggr_en,
		q	 => display_pixel
	);
	
	-- Originally implemented PLL, but jitter issues were more prevalent using PLL vs. clock division
--	pll_inst : pll PORT MAP (
--		inclk0	 => clk,
--		c0	 => clk_25
--	);
	
	MAP_inst : map_memory PORT MAP (
		clock	 => clk_25,
		wren => both_areas,
		addr_in => start_ram,
		addr_aout => addr_aout,
		addr_sout => addr_sout
	);
	
	rden_selects <= check_selects;
	
	Mem_Wr : process (clk,async_reset)
	begin
		if rising_edge(clk) then
			if reset = '1' then
				write_to_address <= addr_aout;
				data_to_aggr <= data_from;

			elsif async_reset = '1' then
				
				write_to_address <= clear_address;
				data_to_aggr <= (others => '0');
				
			else
				
				write_to_address <= addr_aout;
				data_to_aggr <= data_from;
			end if;
		end if;	
	end process Mem_Wr;

	Clr_addr : process(clk)
	begin
		if rising_edge(clk) then
			clear_address <= clear_address+1;
		end if;
	
	end process Clr_addr;
	
--	-- set up 25 MHz clock
	process (clk)
	begin
		if rising_edge(clk) then
			clk_25 <= not clk_25;
		end if;
	end process;
		
	-- Write current location of writing area
	TL_Write : process (clk)
	begin
		
		if rising_edge(clk) then
			if reset = '1' then
				readdata <= (others => '0');
			elsif chipselect = '1' then
				if read = '1' then
					-- for purposes of checking blank
					if address= "0000" then
						readdata <=  "000000000000000" & (vga_vsync or vga_hsync);
					-- return bottom-right location of writing area
					elsif address= "0001" then
						readdata <=  "00" & std_logic_vector(start_RAM);
					-- check the image indices
					elsif address = "0010" then
						readdata <= "000000000000" & check_selects;
					else 
						readdata <= "0000000000000000";
					end if;
				end if;
				if write = '1' then
					--write new bottom-right starting address
					if address = "0011" then						
						start_ram <= unsigned(writedata(13 downto 0)); 
					--select memory buffer to read from
					elsif address = "0100" then	
						check_selects <= std_logic_vector(unsigned(writedata(3 downto 0)));
					--enable for writing to aggregate memory
					elsif address = "0101" then	
						aggr_en <= writedata(0);
					--box color
					elsif address = "0110" then
						box_status <= writedata(5 downto 0);
					--reset of aggregate image
					elsif address = "0111" then	
						async_reset <= writedata(0);
					
					else 
						start_RAM <= start_RAM;
						box_status <= box_status;
					end if;
				end if;
			end if;
		end if;
	end process TL_Write;

	  -- Horizontal and vertical counters

	HCounter : process (clk_25)
	begin
		if rising_edge(clk_25) then      
			if reset = '1' then
				Hcount <= (others => '0');
			elsif EndOfLine = '1' then
				Hcount <= (others => '0');
			else
				Hcount <= Hcount + 1;
		 
			end if;      
		end if;
	end process HCounter;

	EndOfLine <= '1' when Hcount = HTOTAL - 1 else '0';
	  
	VCounter: process (clk_25)
	begin
		if rising_edge(clk_25) then      
			if reset = '1' then
				Vcount <= (others => '0');
			elsif EndOfLine = '1' then
				if EndOfField = '1' then
					Vcount <= (others => '0');
				else
					Vcount <= Vcount + 1;
				end if;
			end if;
		end if;
	end process VCounter;

	EndOfField <= '1' when Vcount = VTOTAL - 1 else '0';

	-- State machines to generate HSYNC, VSYNC, HBLANK, and VBLANK

	HSyncGen : process (clk_25)
	begin
		if rising_edge(clk_25) then     
			if reset = '1' or EndOfLine = '1' then
				vga_hsync <= '1';
			elsif Hcount = HSYNC - 1 then
				vga_hsync <= '0';
			end if;
		end if;
	end process HSyncGen;

	HBlankGen : process (clk_25)
	begin
		if rising_edge(clk_25) then
			if reset = '1' then
				vga_hblank <= '1';
			elsif Hcount = HSYNC + HBACK_PORCH then
				vga_hblank <= '0';
			elsif Hcount = HSYNC + HBACK_PORCH + HACTIVE then
				vga_hblank <= '1';
			end if;      
		end if;
	end process HBlankGen;

	VSyncGen : process (clk_25)
	begin
		if rising_edge(clk_25) then
			if reset = '1' then
				vga_vsync <= '1';
			elsif EndOfLine ='1' then
				if EndOfField = '1' then
					vga_vsync <= '1';
				elsif Vcount = VSYNC - 1 then
					vga_vsync <= '0';
				end if;
			end if;      
		end if;
	end process VSyncGen;

	VBlankGen : process (clk_25)
	begin
		if rising_edge(clk_25) then    
			if reset = '1' then
				vga_vblank <= '1';
			elsif EndOfLine = '1' then
				if Vcount = VSYNC + VBACK_PORCH - 1 then
					vga_vblank <= '0';
				elsif Vcount = VSYNC + VBACK_PORCH + VACTIVE - 1 then
					vga_vblank <= '1';
				end if;
			end if;
		end if;
	end process VBlankGen;

	Area_Check_X : process(clk_25)
	begin
		if rising_edge(clk_25) then
			if reset = '1' or Hcount = HSYNC + HBACK_PORCH + BOX_SET_XSTART-1 then
				area_x <= '1';
			elsif Hcount = HSYNC + HBACK_PORCH + BOX_SET_XEND-1 then
				area_x <= '0';
			end if;
		
		end if;
	
	end process Area_Check_X;
	

	Area_Check_Y : process(clk_25)
	begin
		if rising_edge(clk_25) then
			if reset = '1' then
				area_y <= '0';
			elsif EndOfLine = '1' then
				if Vcount = VSYNC + VBACK_PORCH - 1 + BOX_SET_YSTART-1 then
					area_y <= '1';
				elsif Vcount = VSYNC + VBACK_PORCH - 1 + BOX_SET_YEND-1 then
					area_y <= '0';
				end if;
			end if;
		
		end if;
	
	end process Area_Check_Y;	
	
	Area_Check_X2 : process(clk_25)
	begin
		if rising_edge(clk_25) then
			if reset = '1' or Hcount = HSYNC + HBACK_PORCH + BOX_SET_XSTART2-1 then
				area_x2 <= '1';
			elsif Hcount = HSYNC + HBACK_PORCH + BOX_SET_XEND2-1 then
				area_x2 <= '0';
			end if;
		
		end if;
	
	end process Area_Check_X2;
	

	Area_Check_Y2 : process(clk_25)
	begin
		if rising_edge(clk_25) then
			if reset = '1' then
				area_y2 <= '0';
			elsif EndOfLine = '1' then
				if Vcount = VSYNC + VBACK_PORCH - 1 + BOX_SET_YSTART2-1 then
					area_y2 <= '1';
				elsif Vcount = VSYNC + VBACK_PORCH - 1 + BOX_SET_YEND2-1 then
					area_y2 <= '0';
				end if;
			end if;
		
		end if;
	
	end process Area_Check_Y2;	
	
	-- Performs counting and pixel doubling for first active region
	
	Display_from_memory : process(clk_25)
	begin
		if rising_edge(clk_25) then
			if reset = '1' then
				waitx <= '0';
				waity <= '0';
				display_address11 <= "11111111111111";
				init1 <= '0';
			elsif area = '1' then
					init1 <= '1';
					if waitx = '1' then
						if display_address11(13 downto 7) = "00000000" then		
							if waity = '1' then	
								if display_address11 = "00000000000000" then
									display_address11 <= display_address11 - 1;
								else
									display_address11 <= display_address11 - "00000010000001";
								end if;
							else
								display_address11 <= display_address11 - "00000010000000";
							end if;
							waity <= not waity;
						else
							if display_address11 = "00000000000000" then
								display_address11 <= "11111111111111";
							else
								display_address11 <= display_address11 - "00000010000000";
							end if;
						end if;
					end if;
					
					
					waitx <= not waitx;
			elsif area = '0' and init1 = '0' then
				waitx <= '0';
				waity <= '0';
				display_address11 <= "11111111111111";
			
			end if;
		end if;
	end process Display_from_memory;
	
	-- Performs counting and pixel doubling for second active region
	
	Display_from_memory2 : process(clk_25)
	begin
		if rising_edge(clk_25) then
			if reset = '1' then
				waitx2 <= '0';
				waity2 <= '0';
				display_address21 <= "11111111";
				init2 <= '0';
				
			elsif area2 = '1' then
		
				init2 <= '1';
				if waitx2 = '1' then
					if display_address21(7 downto 4) = "0000" then		
						if waity2 = '1' then	
							if display_address21 = "00000000" then
								display_address21 <= display_address21 - 1;
							else
								display_address21 <= display_address21 - "00010001";
							end if;
						else
							display_address21 <= display_address21 - "00010000";
						end if;
						waity2 <= not waity2;
					else
						if display_address21 = "00000000" then
							display_address21 <= "11111111";
						else
							display_address21 <= display_address21 - "00010000";
						end if;
					end if;
				end if;
					
					
				waitx2 <= not waitx2;
			elsif area2 = '0' and init2 = '0' then
				waitx2 <= '0';
				waity2 <= '0';
				display_address21 <= "11111111";
		
			end if;
		end if;
	end process Display_from_memory2;
				
	
	area <= area_x and area_y;
	area2 <= area_x2 and area_y2;
	both_areas <= area and area2;
	
	-- Maps full aggregate image to first active area
	
	Mem_map : process(clk)
	begin
		if area = '1' then	
			rdaddress <= std_logic_vector(display_address11);
		else
			rdaddress <= "00000000000000";
		end if;
	end process Mem_map;
	
	-- Maps sample image to second active area
	
	Mem_map2 : process(clk)
	begin
		if area2 = '1' then
			read_address <= std_logic_vector(display_address21);		
		elsif both_areas = '0' then
			read_address <= std_logic_vector(addr_sout);
		else
			read_address <= "00000000";
		end if;
	end process Mem_map2;
			

	display_pixel2 <= data_from;
	  
	-- Registered video signals going to the video DAC

	VideoOut : process (clk_25, reset)
	begin
		if reset = '1' then
			VGA_R <= "0000000000";
			VGA_G <= "0000000000";
			VGA_B <= "0000000000";
		elsif clk_25'event and clk_25 = '1' then
			if area = '1' then
				-- Checks if the display address is within the stiching-sensitive boundary
				if ((display_address11 >= start_RAM+1920) and (display_address11 <= start_RAM+1935)) or 
					((display_address11 >= start_RAM) and (display_address11 <= start_RAM+15)) or
					(display_address11 = start_RAM+128) or 
					(display_address11 = start_RAM+256) or 
					(display_address11 = start_RAM+384) or 
					(display_address11 = start_RAM+512) or 
					(display_address11 = start_RAM+640) or 
					(display_address11 = start_RAM+768) or 
					(display_address11 = start_RAM+896) or 
					(display_address11 = start_RAM+1024) or 
					(display_address11 = start_RAM+1152) or 
					(display_address11 = start_RAM+1280) or 
					(display_address11 = start_RAM+1408) or 
					(display_address11 = start_RAM+1536) or 
					(display_address11 = start_RAM+1664) or 
					(display_address11 = start_RAM+1792) or 
					(display_address11 = start_RAM+128+15) or 
					(display_address11 = start_RAM+256+15) or 
					(display_address11 = start_RAM+384+15) or 
					(display_address11 = start_RAM+512+15) or 
					(display_address11 = start_RAM+640+15) or 
					(display_address11 = start_RAM+768+15) or 
					(display_address11 = start_RAM+896+15) or 
					(display_address11 = start_RAM+1024+15) or 
					(display_address11 = start_RAM+1152+15) or 
					(display_address11 = start_RAM+1280+15) or 
					(display_address11 = start_RAM+1408+15) or 
					(display_address11 = start_RAM+1536+15) or 
					(display_address11 = start_RAM+1664+15) or 
					(display_address11 = start_RAM+1792+15) then
					case box_status is
						-- Yellow display box
						when "000000" =>
							VGA_R <= "1111111111";
							VGA_G <= "1111111111";
							VGA_B <= "0000000000";
						-- Green display box
						when "000001" =>
							VGA_R <= "0000000000";
							VGA_G <= "1111111111";
							VGA_B <= "0000000000";
						-- Red display box
						when others =>
							VGA_R <= "1111111111";
							VGA_G <= "0000000000";
							VGA_B <= "0000000000";
					end case;
				elsif async_reset = '1' then
					VGA_R <= "0000000000";
					VGA_G <= "0000000000";
					VGA_B <= "0000000000";
				else 
					VGA_R <= display_pixel(5 downto 0) & display_pixel(3 downto 0);
					VGA_G <= display_pixel(5 downto 0) & display_pixel(3 downto 0);
					VGA_B <= display_pixel(5 downto 0) & display_pixel(3 downto 0);

				end if;
			elsif area2 = '1' then
				VGA_R <= display_pixel2(5 downto 0) & display_pixel2(3 downto 0);
				VGA_G <= display_pixel2(5 downto 0) & display_pixel2(3 downto 0);
				VGA_B <= display_pixel2(5 downto 0) & display_pixel2(3 downto 0);
			elsif vga_hblank = '0' and vga_vblank = '0' then
				VGA_R <= "1111111111";
				VGA_G <= "1111111111";
				VGA_B <= "1111111111";
			else
				VGA_R <= "0000000000";
				VGA_G <= "0000000000";
				VGA_B <= "0000000000";    
			end if;
		end if;
	end process VideoOut;
		
		
		
	VGA_CLK <= clk_25;
	VGA_HS <= not vga_hsync;
	VGA_VS <= not vga_vsync;
	VGA_SYNC <= '0';
	VGA_BLANK <= not (vga_hsync or vga_vsync);

end rtl;
