open Type
open Ast
open Sast

module StringMap = Map.Make(String)
(*============ symbol table ============*)
type event_table = {
	type_name : string;
	member_list : (t * string) list
}

type symbol_table = {
	parent : symbol_table option;
	mutable vars : (t * string) list;
	mutable funcs : (t * string * (t list)) list;
	mutable events : event_table list;
	is_loop : bool (* true means this is a loop scope, false otherwise *)
}

(*============== Checking Functions ===============*)
(*string of, used for print out error*)
let string_of_type = function
	| Int -> "integer"
	| Float -> "float"
	| String -> "string"
	| Boolean -> "boolean"
	| Event_type tname -> ("Event(" ^ tname ^ ")")
	| Void -> "void"
	| Calendar -> "Calendar"

let string_of_op = function
	  Add -> "+"
	| Sub -> "-"
	| Mult -> "*" 
	| Div -> "/"
	| Equal -> "=="
	| Neq -> "!="
	| Less_than -> "<"
	| Leq -> "<="
	| Greater_than -> ">"
	| Geq -> ">="
	| Or -> "||"
	| And -> "&&"
	| Not -> "!"
	| Dot -> "."
	| Child -> "[]"
	
(* operand type error *)
let e_op_type op t =
	raise (Failure ("The operand of operator " ^
		(string_of_op op) ^ " can not be " ^ (string_of_type t)))	
(*TODO*)
let is_same_type t1 t2 =
	match (t1, t2) with
		| (Calendar, Event_type tt2)-> true(*TODO, which eventtype does calendar contains? how to do it???*)
		| (t1, t2) -> t1 = t2
		
(* split declaration list into single var declaration. e.g.  int a,b=0;=> int a; int b=0;*)
let partition decl = 
		let t = fst decl in
			if t = Void then raise (Failure ("variables cannot be declared with type void"))
			else
				List.rev (List.fold_left (fun l -> function
					Ast.WithInit (name, e) -> (t, name, Some(e))::l
					| Ast.WithoutInit name -> (t, name, None)::l) [] (snd decl))

(*find function by name, functions are all in global environment symbol table*)			
let rec find_function fname env =
	let funcs = env.funcs in
		try
			let _ = List.find (fun (_, n, _) -> n = fname) funcs in true
		with Not_found -> 
			match env.parent with (*find event in parent symbol table*)
				Some(parent) ->find_function fname parent
				| _ -> false

let rec get_function fname env =
	let funcs = env.funcs in
		try
			List.find (fun (_, n, _) -> n = fname) funcs
			
		with Not_found -> 
			match env.parent with (*find event in parent symbol table*)
				Some(parent) -> get_function fname parent
				| _ -> raise(Failure("The funtion "^fname ^" is not declared"))				
		
(*find variable by name*)	
let rec find_variable vname env =
	try
		let _ = List.find (fun (_, n) -> n = vname) env.vars in true
	with Not_found -> 
		match env.parent with (*find event in parent symbol table*)
				Some(parent) ->find_variable vname parent
				| _ -> false

let rec get_variable vname env =
	try
		List.find (fun (_, n) -> n = vname) env.vars
	with Not_found -> 
		match env.parent with (*find event in parent symbol table*)
				Some(parent) ->get_variable vname parent
				| _ -> raise(Failure("The variable "^ vname^" is not declared"))		
				
(*find event type by name*)	
let rec find_event ename env =
	try
		let _ = List.find (fun event -> event.type_name = ename) env.events in true
	with Not_found ->
		match env.parent with (*find event in parent symbol table*)
				Some(parent) ->find_event ename parent
				| _ -> false
				
let rec get_event ename env =
	try
		List.find (fun event -> event.type_name = ename) env.events
	with Not_found ->
		match env.parent with (*find event in parent symbol table*)
				Some(parent) ->get_event ename parent
				| _ -> raise (Failure ("Event model " ^ ename ^ " is not found"))
		
(*================= checking =================*)
let rec check_expr env t = function
	| Ast.Assign (e1, e2) ->
		let et1 = check_expr env t e1 and et2 = check_expr env t e2 in
			let (_, t1) = et1 and (_, t2) = et2 in			
				if (is_same_type t1 t2) then
			       Sast.Assign(et1, et2), t1
				else
			raise (Failure ("type mismatch in assignment"))
	| Ast.Uniop (un_op, e) ->
		let et = check_expr env t e in
			let t1 = snd et in
				let tt = match un_op with
					| Not -> if t1 = Boolean then Boolean else
						raise (Failure ("Only boolean is allowed for boolean operators"))
					| _ -> raise (Failure ("The operator " ^(string_of_op un_op) ^ " is not unary"))
				in
					Sast.Uniop(un_op, et), tt
	| Ast.ObjValue(elist) -> (match t with
		Event_type ename -> let e = get_event ename env in 
			let len1 = List.length e.member_list and len2 = List.length elist in
				if (len1 = len2) then Sast.Noexpr, t (*TODO: each attribute checking*)
				else raise(Failure("Event attributes inconsistent"))
		| Calendar ->  let e = check_expr env t (List.nth elist 0) in (*TODO: all events should be same type*)
			let tt = snd e in
			(match tt with
				Event_type ename -> Sast.Noexpr, tt
				|_ -> raise(Failure("Calendar cannot contain "^ string_of_type tt))
			)
		|_ -> raise(Failure("List is not valided for type "^ string_of_type t))	)
	| Ast.Call (func_name, params) -> 
		let fc = get_function func_name env in
			let(ftype,_,required_param_types) = fc in
				let typed_params = List.map (check_expr env t) params in
					let param_types = List.map (fun et -> snd et) typed_params in
						let _ = match func_name with
							| "print" -> (List.iter (function
								| Event_type _ -> 
									raise (Failure ("print function couldn't print event_type"))
								| Calendar ->
									raise (Failure ("print function couldn't print calendar"))
								| _ -> ()) param_types)
							| "printCalendar" -> List.iter (function
								| Calendar -> ()
								| _ ->
									raise (Failure ("printCalendar function can only print Calendar type"))) param_types
							| _ -> if param_types <> required_param_types then
								raise (Failure ("function parameter types don't match"))
						in
						Sast.Call(func_name,typed_params),ftype
	| Ast.Noexpr -> Sast.Noexpr, Type.Void 
	| Ast.Binop (e1, bin_op, e2) ->
		let et1 = check_expr env t e1 in
			if bin_op = Dot then
				match (snd et1) with
				| Event_type ename ->
					let event = get_event ename env in (*type=event_table *)
						let event_scope = { parent = None; vars = event.member_list;funcs=[];events=[];is_loop=false}
						in
						let et2 = check_expr event_scope (snd et1) e2 in
						Sast.Binop(et1, bin_op, et2), (snd et2)
				| _ -> raise (Failure ("left operand of . should be an event type"))
			else if bin_op = Child then
				(match (snd et1) with
				| Calendar -> let et2 =check_expr env (snd et1) e2 in
					(match (snd et2) with
						Int -> Sast.Binop(et1, bin_op, et2), snd et1(*TODO: calendar bug*)
						| _ -> raise(Failure ("expression inside [] should be int")))
				| _ -> raise (Failure ("left operand of [] should be a calendar")))
			else
				let et2 = check_expr env t e2 in
					let (_, t1) = et1 and (_, t2) = et2 in
						if not (is_same_type t1 t2) then
							raise (Failure ("Type mismatch for operator " ^(string_of_op bin_op) ^ ": left: " ^ (string_of_type t1) ^ ", right: " ^ (string_of_type t2)))
						else (match bin_op with (* check operand type for different operators *)
							| Add -> (match t1 with
									| Int | Float | String -> t1
									| _ -> e_op_type bin_op t1)
							| Sub | Mult | Div -> (match t1 with
									| Int | Float -> t1
									| _ -> e_op_type bin_op t1)
							| Equal | Neq -> (match t1 with
									| Int | Float | String | Boolean -> Boolean
									| _ -> e_op_type bin_op t1)
							| Less_than | Leq | Greater_than | Geq -> (match t1 with
									| Int | Float -> Boolean
									| _ -> e_op_type bin_op t1)
							| And | Or -> (if t1 = Boolean then Boolean else
							raise (Failure ("Only boolean is allowed for boolean operators")))
							| _ -> e_op_type bin_op t1);
						Sast.Binop(et1, bin_op, et2), t1
					
	| Ast.Literal lit ->
		let lit_type = match lit with
			| IntLit i -> Int
			| FloatLit f -> Float
			| BoolLit b -> Boolean
			| StringLit s -> String
		in
		Sast.Literal(lit), lit_type
	| Ast.Id id ->
		try 
			let (var_type, _) = get_variable id env in
				Sast.Id(id), var_type
		with Not_found ->
			raise (Failure ("undeclared identifier " ^ id))
	|_-> raise(Failure("Expr not catch"))	

and check_init env t = function
	Some e -> let e = (check_expr env t e) in
		let _,t1 = e in
		if t = Calendar then
			Some(e)(*TODO: not check calendar*)
		else if t<>t1 then
			raise(Failure ("Left is " ^(string_of_type t)^ ", while right is "^(string_of_type t1) ))
		else Some(e)
	| None -> None
	
and check_vdecl vars env = 
	let _ = match (fst vars) with 
		Event_type ename -> if(not (find_event ename env)) then raise(Failure("Event "^ename^" is not exist"))
		|_->()
	in
	let partv = partition vars in
	let vmem = List.map (fun (t, name, init) ->
		try 
			let _ = 
				List.find (fun (_, n) -> n = name) env.vars
			in raise (Failure ("Variable " ^ name ^ " is already exist in the current block"))
		with Not_found ->
			(t, name, check_init env t init)) partv
	in (*update symbol table*)
	env.vars <- (env.vars @ List.map (fun (t, name, _) -> (t, name)) vmem);
	Sast.Vardecl(vmem)
	
let rec check_stmt env = function
	Ast.Block(b) -> 
		let block_scope = { env with
			parent = Some(env); vars = []}
		in
		Sast.Block(check_stmtlist b block_scope)
	| Ast.Expr e -> Sast.Expr(check_expr env Type.Void e)
	| Ast.Return e -> Sast.Return(check_expr env Type.Void e)
	| Ast.ReturnVoid -> Sast.Return(Sast.Noexpr,Type.Void)
	| Ast.If(e, s1, s2) ->
		let e = check_expr env Type.Void e in
		if ((snd e) = Boolean) then
			let st1 = check_stmt env s1 in
				(* create a fake block for if statement to guarantee a new scope *)
				let new_st1 = match st1 with
					| Sast.Block(_) -> st1
					| _ -> check_stmt env (Ast.Block [s1])
				in
				let st2 = check_stmt env s2 in
				(* create a fake block for else clause to guarantee a new scope *)
				let new_st2 = match st2 with
					| Sast.Block(_) -> st2
					| _ -> check_stmt env (Ast.Block [s2])
				in 
				Sast.If(e, new_st1, new_st2)
			else
				raise (Failure ("the expression in if statement is not boolean"))

	| Ast.For(e1, e2, e3, s) ->
		let e2 = check_expr env Type.Void e2 in
		if ((snd e2) = Boolean) then
			let loop_scope = { env with is_loop = true } in
			
			let block_s = match s with
					| Ast.Block _ -> s
					| _ -> Ast.Block [s]
				in
				let st = check_stmt loop_scope block_s in
					Sast.For (check_expr env Type.Void e1, e2, check_expr env Type.Void e3, st)
		else
			raise (Failure ("the second expression in for statement is not boolean"))
			
	| Ast.While(e, s) -> 
		let e = check_expr env Type.Void e in
		if ((snd e) = Boolean) then
			let loop_scope = { env with is_loop = true } in
				
				let block_s = match s with
					| Ast.Block _ -> s
					| _ -> Ast.Block [s]
				in
				let st = check_stmt loop_scope block_s in
				Sast.While (e, st)
			else
				raise (Failure ("the expression in while statement is not boolean"))
			
	| Ast.Vardecl(v) ->(check_vdecl v env)		
	| Ast.Empty -> Sast.Empty
and check_stmtlist slist env = 
	match slist with
	[] -> []
	| head :: tail -> check_stmt env head :: check_stmtlist tail env
	


let check_event event env = 
	let ename = event.Ast.typename in
		try 
			let _ = List.find (fun e -> e.type_name = ename) env.events
				in raise (Failure ("Event model " ^ ename ^ " is already exist"))
		with Not_found -> 			
			let emem =
				List.concat (List.map partition event.Ast.members)
			in
				let et = {
					type_name = ename;
					member_list = List.map (fun (t, name, _) -> (t, name)) emem
				} in
				 env.events <- et::env.events;
				(*Sast.Event.members*)
					let new_tmem =
						List.map (fun (t,name,init)->
						let _ = match t with (* recursive tree type in members *)
							Event_type ename-> raise(Failure("Cannot declare nested event type"))
							|Calendar -> raise(Failure("Cannot declare Calendar type"))
							| _ -> ()
						in ( t, name, check_init env t init)) emem
					(*return Sast*)
					in {Sast.typename = ename; Sast.members= new_tmem}
				


let check_func func env =
	let ft = func.return_type
		and fn = func.fname
		and fp = func.params
	in
	if (find_function fn env) then
		raise (Failure ("function" ^ fn ^ "redeclared"))
	else (* function is effective *)
		(List.iter (fun (t, name) -> 
			if (t = Type.Void) then 
				raise (Failure ("parameter" ^ name ^ "cannot be void"))
			else
				let num = 
					List.fold_left (fun i (_, n) -> 
						if (n = name) then i + 1 else i) 0 fp
				in if (num > 1) then 
					raise (Failure ("duplicate parameter name" ^ name))	
		)fp);
		(*let required_param_types = (*get type info of params*)
			List.map (fun p -> fst p) fp
		in
		let new_scope = { env with funcs = (*add function to globe_scope*)
			(ft, fn, required_param_types)::env.funcs
		} in 
		let param_scope = { new_scope with (*add params to function scope*)
			parent = Some(new_scope); vars = fp}
		in
		let new_body =
			check_stmtlist func.body param_scope
		in*)
		let required_param_types = (* only type info of the params *)
							List.map (fun p -> fst p) fp
		in env.funcs <- (ft,fn,required_param_types)::env.funcs


let rec check_eventlist elist env =
	match elist with
	[] -> []
	| head :: tail -> check_event head env :: check_eventlist tail env
	
let rec check_vdecllist vlist env = 
	match vlist with
	[] -> []
	| head :: tail -> let _ = check_vdecl head env in check_vdecllist tail env



let rec check_funclist flist env = 
	match flist with
	[] -> []
	| head :: tail -> check_func head env :: check_funclist tail env

let check_main env =
	if (not (find_function "main" env)) then raise(Failure("function main() is not found"))

let check (p:program) =
	let global_env = {
		parent = None;
		vars = [];
		funcs = [(Void, "print", [String]);(Void,"printCalendar",[Calendar])]; (* built-in functions *)
		events = [];
		is_loop = false;
	} in
	(*use global_env*)
	let _ = check_eventlist p.eventdef global_env in
		let _ = check_vdecllist p.globalvar global_env in
			let _ = check_funclist p.funcdef global_env in
				let _ = check_main global_env 
					in true
				
					(*{Sast.eventdef = events; Sast.globalvar = variables; Sast.funcdef = functions; print_endline "\nSemantic analysis completed successfully.\n";}
				else *)