{ open Parser }

(* letter, digit *)

let esp =   "\\\"" | "\\\\" | "\\n" | "\\t" (*Escape character*)
let letter = ['a'-'z' 'A'-'Z']
let digit = ['0' - '9']
let whitespace = ['\t' ' ' '\r' '\n']
let id = letter (letter | digit | '_' )*
let punc = ['~' '`' '!' '@' '#' '$' '%' '^' '&' '*' '(' ')' '-' '+' '=' ',' '.' '?' '/' '<' '>' ':' '''  ';' '{' '}' '[' ']' '|' ' ']
let stringlit = '"' (letter | digit | punc | esp)*  '"'


rule token = parse
	whitespace { token lexbuf }
	| "#" { comment lexbuf }

    (* Keywords *)
    | "if"            { IF }
    | "else"          { ELSE }
    | "while"         { WHILE }
    | "for"           { FOR }
    | "return"        { RETURN }
	| "null"		  { NULL }
    | "int"           { INT_T }
    | "float"         { FLOAT_T }
    | "string"        { STRING_T }
    | "bool"          { BOOL_T }
    | "void"          { VOID }
    | "Event"      	  { EVENTTYPE }
	| "Calendar" 	  {CALENDAR}
    | "main"          as main      
                      { ID(main) }
    | "print"         as print
                      { ID(print) }
    | "size"         as size
                      { ID(size) }
	
    (* Constants *)
    | digit+          as integer        
                      { INT(int_of_string integer) }
	| digit+ '.' digit* 
	| '.' digit+  as float { FLOAT(float_of_string float) }
	
    | "true"
    | "false"         as bool
                      { BOOL(bool_of_string bool) }
	| stringlit   as string { STRING(string) }	


    | '{'    { LBRACE }
    | '}'    { RBRACE }
    | ';'    { SEMI }
    | ','    { COMMA }
    | '='    { ASSIGN }
    
    (* Operators *)
    | "||"   { OR }
    | "&&"   { AND }
    | '!'    { NOT }
    | "!="   { NEQ }
    | '>'    { GT }
    | '<'    { LT }
    | "<="   { LEQ }
    | ">="   { GEQ }
    | "=="   { EQ }
    | '+'    { PLUS }
    | '-'    { MINUS }
    | '*'    { TIMES }
    | '/'    { DIVIDE }
    | '['    { LBRACK }
    | ']'    { RBRACK }
    | '.'    { DOT }
    | '('    { LPAREN }
    | ')'    { RPAREN }
	
	| letter (letter | digit | '_')* as identifier 
				   { ID(identifier) }

    | eof { EOF }
    | _ as err_char { raise (Failure("illegal character " ^ Char.escaped err_char)) }

(* comment *)
and comment = parse
    '\n' { token lexbuf }
	| _ { comment lexbuf }

