type action = Ast | Compile | Help | Version

let version = "iCalendar version 1.0 12/19/2012"

let usage = "Usage: ica [-a|-c|-h|-v] SOURCE_FILE\n"^
			"   -a  (generate AST)\n"^
			"	-c  (compile the program)\n"^
			"	-h  (display help document)\n"^
			"	-v	(display version number)\n"
	
(* Custom exceptions. *)
exception NoInputFile
exception InvalidArgument


(* Get the name of the program from the file name. *)						
let get_prog_name source_file_path =
	let split_path = (Str.split (Str.regexp_string "/") source_file_path) in
	let file_name = List.nth split_path ((List.length split_path) - 1) in
	let split_name = (Str.split (Str.regexp_string ".") file_name) in (*want only the part before dot*)
		List.nth split_name ((List.length split_name) - 2)

(* Main entry point *)
let _ =
	try
		let action = if Array.length Sys.argv > 1 then
			match Sys.argv.(1) with 
					| "-a" -> Ast
					| "-c" -> Compile
					| "-h" -> Help
					| "-v" -> Version
					| _ -> raise InvalidArgument
			else raise InvalidArgument in
		let prog_name = 
			if Array.length Sys.argv > 2 then
				get_prog_name Sys.argv.(2)
			else raise NoInputFile in
			
	let input_chan = open_in Sys.argv.(2) in
  	let lexbuf = Lexing.from_channel input_chan in
  	let program = Parser.program Scanner.token lexbuf in
		(*let program = List.rev reversed_program in*)
		match action with 
			| Ast -> let listing = Ast.string_of_program program in print_string listing
			| Help -> print_endline (usage)
			| Version -> print_endline (version)
			| Compile -> 
					if Semantic.check program 
						then let listing = Compile.string_of_program program prog_name in print_string listing
						else raise(Failure("\nInvalid program.\n"))
			
	with 
		| InvalidArgument -> ignore (Printf.printf "InvalidArgument\n %s\n" usage)
		| NoInputFile -> ignore (Printf.printf "The second argument must be the name of an iCalendar file\n %s\n" usage)
