open Type

type expr = (* Expressions *)
    Literal of literal (* 42 *)
  | Id of string (* foo *)
  | Binop of expr * op * expr (* a + b *)
  | Assign of expr * expr (* a = b *)
  | Call of string * expr list
  | Noexpr (* While() *)
  | Uniop of op * expr   (*for unary operators *)
  | ObjValue of (expr list)
  
type init = WithInit of string * expr
			| WithoutInit of string

type init_list = init list

type var_decl = t * init_list

type event_def = {
	typename: string;
	members : var_decl list;
}

type stmt = (* Statements  nothing *)
     Block of stmt list
	| Expr of expr   (*foo = bar + 3; *)
	| Return of expr (* return 42 also includes return function_name *)
	| ReturnVoid
	| If of expr * stmt * stmt (* if (foo == 42) {} else {} *)
	| For of expr * expr * expr * stmt
	| While of expr * stmt (* while (i<10) { i = i + 1 } *)
	| Vardecl of var_decl
	| Empty 
   
type func_decl = {
	return_type: t;
    fname : string;
	params : (t * string) list;
    body : stmt list;
}
               
type program = {
	eventdef : event_def list;
	globalvar : var_decl list;
	funcdef : func_decl list;
}


(* print the AST*)
let string_of_op  = function
	Add ->"+"
	| Sub ->"-"
	| Mult ->"*"
	| Div ->"/"
	| Equal ->"=="
	| Neq ->"!="
	| Less_than->"<" 
	| Leq ->"<="
	| Greater_than ->">"
	| Geq ->">="
	| Or ->"||"
	| And ->"&&"
	| Not ->"!"
	| Dot ->"."
	| Child ->""
	
let string_of_literal = function
	IntLit(i)-> string_of_int i
  | FloatLit(f) -> string_of_float f
  | BoolLit(b)->string_of_bool b
  | StringLit(s) -> s
  
let string_of_dt = function
	Int ->"int"
	| Float ->"float"
	| String ->"String"
	| Boolean ->"boolean"
	| Void ->"void"
	| Event_type(myEvent) -> myEvent
	| Calendar-> "Calendar"
	
let rec string_of_expr = function
    Literal(l) -> string_of_literal l
  | Id(s) -> s
  | Assign(e1,e2)->string_of_expr e1 ^" = "^string_of_expr e2
  | Call(f, el) -> f ^ "(" ^ String.concat ", " (List.map string_of_expr el) ^ ")"
  | Noexpr (* While() *)->""
  | Uniop(o,e)->string_of_op o ^" "^string_of_expr e
  | ObjValue(l)->"[" ^ (String.concat ", " (List.map string_of_expr l)) ^" ]"
  | Binop(e1,o,e2)-> match o with
					 Child -> string_of_expr e1 ^ "[" ^ string_of_expr e2 ^ "]"
					| _    -> string_of_expr e1 ^ " " ^ string_of_op o ^ " " ^ string_of_expr e2
  
  
let string_of_init = function
	WithInit(v,e)-> v ^ "=" ^ string_of_expr e
	|WithoutInit(v)-> v 
	
let string_of_initlist (init) =
	(String.concat ", " (List.map string_of_init init))^";\n"

let string_of_initlist_f (init) =
	(String.concat "f, " (List.map string_of_init init ))^"f;\n"
	
let string_of_vdecl (t, l)=
	match t with
	Float -> string_of_dt t^" "^ string_of_initlist_f l^"\n"
	|_    -> string_of_dt t^" "^ string_of_initlist l^"\n"
	
let string_of_event (myevent) = 
	"Event "^ (myevent.typename) ^ "{\n"^
	String.concat "\n" (List.map string_of_vdecl myevent.members)^
	"}\n"
	
let string_of_arg(t, name) = 
	(string_of_dt t) ^ " " ^ name

let rec string_of_stmt = function
	Block(stmts) -> "{\n" ^ String.concat "\n" (List.map string_of_stmt stmts) ^ "}\n"
   | Expr(e) -> string_of_expr e ^";"
   | Return(e) ->"return "^ (string_of_expr e) ^";"
   | ReturnVoid ->"return ;"
   | If(e, s, Block([]))->"if (" ^ string_of_expr e ^ ")\n" ^ string_of_stmt s
   | If(e, s1, s2) ->  "if (" ^ string_of_expr e ^ ")\n" ^ string_of_stmt s1 ^ "else\n" ^ string_of_stmt s2
   | For(e1,e2,e3,s)->"for (" ^ string_of_expr e1  ^ " ; " ^ string_of_expr e2 ^ " ; " ^ string_of_expr e3  ^ ")\n " ^ string_of_stmt s
   | While(e, s) -> "while (" ^ string_of_expr e ^ ") " ^ string_of_stmt s
   | Vardecl(v)-> string_of_vdecl v
   | Empty->"" 
	
let string_of_func (f) =
	(string_of_dt f.return_type)^" "^ f.fname ^ " (" ^ String.concat ", " (List.map string_of_arg f.params)^")"^
	"{\n" ^ (String.concat "" (List.map string_of_stmt f.body) )^ "}\n"
	
let string_of_events = function
	[] -> ""
	| elist -> String.concat "\n" (List.map string_of_event elist)

let string_of_vars = function
    [] -> ""
	| v -> String.concat "\n" (List.map string_of_vdecl v)

let string_of_funs = function
	[] -> ""
	| flist -> String.concat "\n" (List.map string_of_func flist)

let string_of_program (p:program) =
	string_of_events p.eventdef ^ string_of_vars p.globalvar ^  string_of_funs p.funcdef
